# Terminal Management System (TMS) - DaProductApp
TMS is a Phoenix/Elixir web application for terminal management, settlement processing, and dispute handling. It provides a web UI, REST APIs, and real-time terminal communication via MQTT/TCP protocols.

**Always reference these instructions first and fallback to search or bash commands only when you encounter unexpected information that does not match the info here.**

## Working Effectively

### System Requirements - Install First
- **Elixir**: 1.14+ and **Erlang/OTP**: 25+
  ```bash
  sudo apt update && sudo apt install -y elixir erlang-dev
  ```
- **MySQL**: 8.0+ (CRITICAL - application will not work without MySQL)
  ```bash
  sudo apt install -y mysql-server
  sudo systemctl start mysql
  ```
- **Node.js**: For asset compilation (automatically handled by mix setup)

### Database Setup (MySQL 8.0 Required)
**CRITICAL: Must be completed before any other commands**
1. Start MySQL service: `sudo systemctl start mysql`
2. Connect as root: `sudo mysql -u root` (may need sudo on fresh install)
3. Set root password and create databases:
   ```sql
   ALTER USER 'root'@'localhost' IDENTIFIED BY 'dataaegis123';
   CREATE DATABASE IF NOT EXISTS lic_project_cicd;
   CREATE DATABASE IF NOT EXISTS lic_project_testingteam;
   FLUSH PRIVILEGES;
   quit
   ```
4. Test connection: `mysql -u root -pdataaegis123 -e "SHOW DATABASES;"`

### Bootstrap and Dependencies
**CRITICAL: If hex.pm is blocked, use GitHub install. Set timeout to 300+ seconds for dependency operations. NEVER CANCEL dependency downloads.**

1. Install Hex package manager:
   ```bash
   # If hex.pm is accessible:
   mix local.hex --force
   
   # If hex.pm is blocked (use this alternative):
   mix archive.install github hexpm/hex branch latest --force
   ```

2. Install project dependencies: 
   ```bash
   mix deps.get
   ```
   **Takes 3-5 minutes. NEVER CANCEL. Set timeout to 300+ seconds.**

3. Compile dependencies: 
   ```bash
   mix deps.compile
   ```
   **Takes 2-3 minutes. NEVER CANCEL. Set timeout to 180+ seconds.**

### Build and Setup
**CRITICAL: Set timeout to 720+ seconds for full setup. Build may take 12+ minutes. NEVER CANCEL.**
- **Full setup (recommended)**: `mix setup` 
  - **Takes 8-12 minutes total. NEVER CANCEL. Set timeout to 720+ seconds.**
  - Runs: `deps.get`, `ecto.setup`, `assets.setup`, `assets.build`
- Alternative step-by-step approach:
  - Database setup: `mix ecto.setup` -- creates DB, runs migrations, seeds data (2-3 minutes)
  - Asset setup: `mix assets.setup` -- installs Tailwind CSS and esbuild (1-2 minutes)
  - Asset build: `mix assets.build` -- compiles CSS and JavaScript (1-2 minutes)

### Running the Application
- **Start Phoenix server**: `mix phx.server` 
  - Server runs on **http://localhost:4000**
  - Development mode with live reload enabled
- **Start with IEx console**: `iex -S mix phx.server` 
  - Interactive Elixir shell with server running
  - Useful for debugging and inspecting application state

### Testing
**CRITICAL: Set timeout to 900+ seconds for test suite. Tests may take 15+ minutes. NEVER CANCEL.**
- **Run all tests**: `mix test` 
  - **Takes 10-15 minutes. NEVER CANCEL. Set timeout to 900+ seconds.**
  - Automatically runs: `ecto.create --quiet`, `ecto.migrate --quiet`, then tests
- Run specific test file: `mix test test/path/to/test_file.exs`
- Run tests with coverage: `mix test --cover`
- Run tests in watch mode: `mix test.watch` (if available)

### Code Quality and Linting
**ALWAYS run these before committing changes or CI will fail:**
- **Format code**: `mix format` -- auto-formats all Elixir code
- **Run Credo linter**: `mix credo` -- checks code quality and style
- **Run Credo diff**: `mix credo diff --from-git-merge-base origin/main` -- only check changed files
- **Run Sobelow security scanner**: `mix sobelow` -- security analysis (may take 2-3 minutes)

## Validation Scenarios

### Manual Validation Requirements
**ALWAYS run these validation steps after making changes:**

#### 1. Database Connectivity Validation  
```bash
# Verify MySQL is running and accessible
sudo systemctl status mysql
mysql -u root -pdataaegis123 -e "SHOW DATABASES;"

# Should show: lic_project_cicd, lic_project_testingteam, plus system DBs
```

#### 2. Application Build Validation
```bash
# Full build validation (takes 8-12 minutes)
mix setup

# Quick compilation check
mix compile

# Verify no compilation errors
echo $?  # Should output 0
```

#### 3. Web Application Testing
1. **Start the server**: `mix phx.server`
2. **Visit http://localhost:4000** - verify home page loads without errors
3. **Test user registration flow**:
   - Navigate to registration page
   - Create a test user account
   - Verify email/password validation works
4. **Test user login flow**:
   - Login with created account  
   - Verify authentication redirects work
5. **Navigate to main application sections**:
   - Terminal management dashboard
   - Settlement reporting sections
   - User management (if admin)

#### 4. API Endpoint Validation
**Use the provided demo script for comprehensive API testing:**
```bash
# Make script executable and run
chmod +x demo_api_endpoints.sh
bash demo_api_endpoints.sh

# Test individual endpoints manually:
curl -X GET "http://localhost:4000/api/v1/merchant/settlements"
curl -X GET "http://localhost:4000/api/v1/merchant/disputes"
curl -X GET "http://localhost:4000/api/v1/merchant/settlement/summary"
```

#### 5. Database Operations Validation
```bash
# Check database migrations status
mix ecto.migrations

# Verify seed data was created
mysql -u root -pdataaegis123 -e "USE lic_project_cicd; SELECT COUNT(*) FROM users; SELECT COUNT(*) FROM terminals; SHOW TABLES;"

# Test database reset (if needed)
mix ecto.reset  # CAUTION: This deletes all data
```

#### 6. Asset Compilation Validation  
```bash
# Verify assets compile without errors
mix assets.build

# Check asset files were generated
ls -la priv/static/assets/

# Verify CSS and JS files exist with recent timestamps
```

## System Requirements

### Software Dependencies
- **Elixir**: 1.14+ (project configured for 1.14.0)
- **Erlang/OTP**: 25+ (project configured for OTP 25)  
- **MySQL**: 8.0+ (REQUIRED - uses MyXQL adapter)
- **Node.js**: For asset compilation (esbuild, Tailwind CSS) - automatically installed by mix setup

### Network Requirements  
- **HTTP access** for Hex package manager (hex.pm) - if blocked, use GitHub alternative
- **GitHub access** for Git dependencies (heroicons, hex archive)
- **MQTT broker access** for terminal communication (configurable, see config/dev.exs)

### Build Times (Measured Expectations)
- **System setup (Elixir + MySQL)**: 2-3 minutes
- **First time `mix deps.get`**: 3-5 minutes (depends on network)
- **First time `mix setup`**: 8-12 minutes total (includes all setup steps)  
- **`mix compile`**: 2-3 minutes (after deps compiled)
- **`mix test`**: 10-15 minutes (full test suite with database setup)
- **Asset compilation**: 1-2 minutes
- **`mix phx.server` startup**: 30-60 seconds

### Memory and Storage
- **RAM**: Minimum 2GB, recommended 4GB+ for development  
- **Storage**: ~500MB for dependencies, ~1GB total project space
- **Database**: MySQL will use ~200-500MB RAM depending on data size

## Project Structure

### Key Directories
- `lib/da_product_app/` - Core business logic (contexts, schemas)
  - `terminal_management/` - Terminal management functionality  
  - `settlements/` - Settlement processing
  - `disputes/` - Dispute handling
  - `users/` - User management and authentication
  - `mqtt/` - MQTT communication with terminals
  - `tcp/` - TCP socket communication
- `lib/da_product_app_web/` - Web layer (controllers, views, LiveView)
- `test/` - Test files (follows lib/ structure)
- `priv/repo/migrations/` - Database migrations
- `config/` - Environment configurations
- `assets/` - Frontend assets (CSS, JavaScript)

### Important Files
- `mix.exs` - Project configuration and dependencies
- `config/dev.exs` - Development database and server configuration
- `config/test.exs` - Test environment configuration
- `demo_api_endpoints.sh` - API testing script
- `IMPLEMENTATION_SUMMARY.md` - API documentation

### Database Configuration
- **Development**: `lic_project_cicd` database, port 4002
- **Test**: `lic_project_testingteam` database  
- **Credentials**: root/dataaegis123 (both environments)

## Common Issues and Solutions

### Dependency Resolution
- If hex.pm is unreachable: `mix archive.install github hexpm/hex branch latest --force`
- If deps.get fails: Ensure network access to hex.pm and GitHub
- If compile fails: Run `mix clean` then `mix deps.compile`

### Database Issues  
- If database connection fails: Verify MySQL is running and credentials are correct
- If migrations fail: Check database exists and user has permissions
- For test database: Ensure `lic_project_testingteam` database exists

### Asset Compilation Issues
- If assets fail to build: Ensure Node.js is installed
- Missing esbuild/tailwind: Run `mix assets.setup` first
- Asset watchers in development: Automatically handled by `mix phx.server`

## Frequently Used Commands Summary

```bash
# === FIRST TIME SETUP (run once after fresh clone) ===
# Install system dependencies
sudo apt update && sudo apt install -y elixir erlang-dev mysql-server

# Setup MySQL (run commands inside mysql prompt)
sudo mysql -u root
# ALTER USER 'root'@'localhost' IDENTIFIED BY 'dataaegis123';
# CREATE DATABASE IF NOT EXISTS lic_project_cicd;  
# CREATE DATABASE IF NOT EXISTS lic_project_testingteam;
# FLUSH PRIVILEGES; quit

# Install Hex and setup project (8-12 minutes total)
mix archive.install github hexpm/hex branch latest --force
mix setup  # NEVER CANCEL - set 720+ second timeout

# === DAILY DEVELOPMENT ===
# Start development server (most common)
mix phx.server  # Access at http://localhost:4000

# Run tests (10-15 minutes)  
mix test  # NEVER CANCEL - set 900+ second timeout

# Code quality checks (ALWAYS before committing)
mix format && mix credo

# === DEBUGGING AND TROUBLESHOOTING ===
# Reset database (DESTROYS ALL DATA)
mix ecto.reset

# Clean and rebuild
mix clean && mix compile

# Check dependency status
mix deps

# Database operations
mix ecto.create      # Create databases  
mix ecto.migrate     # Run pending migrations
mix ecto.rollback    # Rollback last migration

# Asset operations  
mix assets.setup     # Install Tailwind/esbuild
mix assets.build     # Compile CSS/JS assets
```

## Development Workflow  

### Making Changes - Standard Process
1. **Before starting**: `mix phx.server` and verify app loads at http://localhost:4000
2. **Make your code changes** in `lib/` directory
3. **Test your changes**: Run relevant tests with `mix test test/path/to/specific_test.exs`
4. **Full validation**: `mix test` (NEVER CANCEL - 10-15 minutes)
5. **Code quality**: `mix format && mix credo` (REQUIRED before commit)
6. **Manual testing**: 
   - Navigate to affected UI areas
   - Test API endpoints with `bash demo_api_endpoints.sh` or manual curl commands
   - Verify no broken functionality

### API Development Workflow
1. **Understand existing APIs**: Read `IMPLEMENTATION_SUMMARY.md` and run `demo_api_endpoints.sh`
2. **Add/modify controllers**: in `lib/da_product_app_web/controllers/`
3. **Update contexts**: in `lib/da_product_app/` for business logic
4. **Update schemas**: in `lib/da_product_app/*/` for data models
5. **Add tests**: in `test/da_product_app_web/controllers/` and `test/da_product_app/`
6. **Test API endpoints**: Use curl or demo script to verify JSON responses

### Database Changes Workflow  
1. **Generate migration**: `mix ecto.gen.migration add_your_feature`
2. **Edit migration file**: in `priv/repo/migrations/`
3. **Run migration**: `mix ecto.migrate`
4. **Update schemas**: Add fields to relevant schema files
5. **Test migration**: `mix ecto.rollback && mix ecto.migrate`
6. **Add seeds if needed**: Update `priv/repo/seeds.exs`

### Troubleshooting Common Issues
- **Database connection errors**: Verify MySQL running with `sudo systemctl status mysql`
- **Dependency errors**: Delete `_build/` and `deps/`, run `mix deps.clean --all && mix setup`
- **Asset compilation errors**: Run `mix assets.setup` then `mix assets.build`  
- **Port already in use**: Check if Phoenix server already running, kill with Ctrl+C
- **Test failures**: Ensure test database exists and is migrated
- **Permission errors**: MySQL user might not have correct permissions

## Security and Production Considerations

### Important Security Notes
- **This is a production terminal management system handling financial data**
- **Always test thoroughly before deploying payment/settlement features**
- **Use `mix sobelow` to scan for security vulnerabilities** 
- **Never commit sensitive credentials to version control**
- **Review all database queries for SQL injection vulnerabilities**

### Environment-Specific Configurations  
- **Development**: Uses `lic_project_cicd` database, runs on port 4000
- **Test**: Uses `lic_project_testingteam` database, runs on port 4002  
- **Production**: See `config/prod.exs` for production settings

### Production Deployment Notes
- **Database migrations**: Run `mix ecto.migrate` in production environment
- **Asset compilation**: Use `mix assets.deploy` for optimized production assets
- **Secret management**: Set proper `SECRET_KEY_BASE` and database credentials
- **SSL configuration**: Enable HTTPS for production (see config/prod.exs)

Remember: This system processes real financial transactions and terminal communications. Always follow security best practices and test changes thoroughly in a staging environment before production deployment.