# Batch Number Logic and Merchant Batch Management

This document explains the implementation of the batch number logic update and merchant batch management features.

## Overview

The system now supports configurable batch number assignment logic:

1. **Provider-wise batch number enabled**: Batch numbers are assigned in the AANI settlement flow
2. **Provider-wise batch number disabled**: Batch numbers are assigned in the YCS settlement flow

## Configuration

Add the following to your configuration file:

```elixir
config :da_product_app, :batch_settings,
  provider_wise_batch_number_enabled: true  # or false
```

## Database Schema

### merchant_batch_numbers table

```sql
CREATE TABLE merchant_batch_numbers (
  id BINARY(16) PRIMARY KEY,
  merchant_id VARCHAR(255) NOT NULL,
  batch_number INT NOT NULL DEFAULT 1,
  provider_id INT NULL,
  inserted_at DATETIME NOT NULL,
  updated_at DATETIME NOT NULL,
  UNIQUE INDEX idx_merchant_batch_numbers_merchant_id (merchant_id),
  INDEX idx_merchant_batch_numbers_merchant_provider (merchant_id, provider_id)
);
```

## API Usage

### YCS Settlement Creation (when provider-wise is disabled)

```bash
POST /api/v1/settlements/ycs/create
Content-Type: application/json

{
  "merchant_id": "MERCHANT_123",
  "amount": "1000.00",
  "status": "pending",
  "date": "2025-09-12",
  "provider_id": 2
}
```

Response:
```json
{
  "settlement_id": "SETT20250912-12345",
  "batch_number": 1,
  "status": "created",
  "message": "YCS settlement created successfully with batch number assignment"
}
```

### AANI Settlement (when provider-wise is enabled)

The existing AANI settlement endpoint continues to work as before:

```bash
POST /api/v1/aani/settlement-summary
```

The batch number assignment now happens automatically in the AANI controller and updates the merchant_batch_numbers table.

## Context Functions

### Settlements.get_and_increment_batch_number/2

Gets the current batch number for a merchant and increments it for the next use.

```elixir
{:ok, batch_number} = Settlements.get_and_increment_batch_number("MERCHANT_123", 3)
# Returns 1 for new merchant, increments to 2 for next call
```

### Settlements.get_current_batch_number/1

Gets the current batch number without incrementing.

```elixir
current_batch = Settlements.get_current_batch_number("MERCHANT_123")
# Returns 1 if no record exists, or current batch number
```

### Settlements.provider_wise_batch_number_enabled?/0

Checks if provider-wise batch number logic is enabled.

```elixir
if Settlements.provider_wise_batch_number_enabled?() do
  # Handle in AANI controller
else
  # Handle in YCS controller
end
```

## Flow Diagrams

### Provider-wise Enabled (AANI Flow)
```
AANI Settlement Request → Check Config → Provider-wise Enabled → 
Increment Merchant Batch Number → Create Settlement with New Batch Number
```

### Provider-wise Disabled (YCS Flow)
```
YCS Settlement Request → Check Config → Provider-wise Disabled → 
Increment Merchant Batch Number → Create Settlement with New Batch Number
```

## Testing

Run the tests to verify functionality:

```bash
mix test test/da_product_app/settlements_test.exs
mix test test/da_product_app_web/controllers/settlement_controller_test.exs
```

## Migration

To apply the new table:

```bash
mix ecto.migrate
```

## Backward Compatibility

- Existing settlements continue to work without modification
- Configuration defaults to provider-wise enabled to maintain current behavior
- New merchant_batch_numbers table is created separately and doesn't affect existing data

## Error Handling

- Missing merchant_id returns 400 Bad Request
- Configuration mismatch returns appropriate error messages
- Database errors are logged and handled gracefully
- Transaction rollback ensures data consistency