# AlipayPlus Settlement Integration

This module provides comprehensive integration with AlipayPlus settlement services, implementing all requirements for Case 1: One Clearing Cycle (Payment Only).

## Architecture Overview

The AlipayPlus settlement integration consists of several key components:

### Core Modules

1. **SftpClient** (`DaProductApp.Settlements.AlipayPlus.SftpClient`)
   - Handles secure SFTP connections to AlipayPlus servers
   - Downloads settlement files according to naming conventions
   - Validates file formats and structure

2. **CsvParser** (`DaProductApp.Settlements.AlipayPlus.CsvParser`)
   - Parses AlipayPlus CSV settlement files
   - Handles both summary and detail sections
   - Converts currency amounts from smallest units to standard values

3. **Processor** (`DaProductApp.Settlements.AlipayPlus.Processor`)
   - Core business logic for settlement processing
   - Creates/updates settlement records in database
   - Updates transaction settlement statuses
   - Provides reconciliation and validation

4. **Background Jobs** (`DaProductApp.Settlements.AlipayPlus.Jobs.*`)
   - Automated SFTP file retrieval using Oban
   - Asynchronous file processing
   - Error handling and retry mechanisms

## File Format Support

The integration supports the AlipayPlus CSV format specification:

### Filename Convention
```
settlement_<participantId>_<settlementCurrency>_<settlementBatchId>_<participantAgreementId>_<seq>.csv
```

### Directory Structure
- **Production**: `/v1/settlements/settlement/<participantId>/<date>`
- **Sandbox**: `/sandbox/settlements/settlement/<participantId>/<date>`

### CSV Structure
Each file contains two sections:
1. **Summary Section**: Overall settlement information
2. **Detail Section**: Individual clearing cycle details

## API Endpoints

### Standard Settlement APIs (Enhanced)
All existing settlement APIs now support AlipayPlus settlements seamlessly:

- `GET /api/v1/merchant/settlements` - List settlements with AlipayPlus support
- `GET /api/v1/merchant/settlement/:settlement_id` - Get settlement details including AlipayPlus fields
- `GET /api/v1/merchant/settlement/:settlement_id/transactions` - List settlement transactions
- `GET /api/v1/merchant/settlement/:settlement_id/download` - Download settlement reports

### AlipayPlus Specific APIs

- `POST /api/v1/merchant/settlement/alipayplus/process` - Manual file processing
- `POST /api/v1/merchant/settlement/alipayplus/schedule_retrieval` - Schedule SFTP retrieval

## Configuration

### SFTP Configuration
Set the following environment variables:
```bash
ALIPAYPLUS_SFTP_USERNAME=your_username
ALIPAYPLUS_SFTP_PASSWORD=your_password
```

### Application Configuration
```elixir
config :da_product_app, :alipayplus_sftp, [
  host: "sftp.alipayplus.com",
  port: 22,
  connect_timeout: 30_000
]

config :da_product_app, :alipayplus_participant_ids, [
  "1000012345",
  "1000012346"
]
```

## Database Schema

### New Settlement Fields
The settlement table has been enhanced with AlipayPlus specific fields:

- `participant_id` - AlipayPlus participant identifier
- `settlement_batch_id` - Unique settlement batch ID
- `participant_agreement_id` - Agreement between AlipayPlus and ACQP
- `value_date` - Date by which AlipayPlus must pay ACQP
- `fund_direction` - CREDIT or DEBIT direction
- `transaction_currency` - Optional transaction currency
- `net_transaction_amount_value` - Optional transaction amount
- `extend_info` - Extended information

## Usage Examples

### Manual File Processing
```elixir
filename = "settlement_1000012345_AED_202000000000000000_AGR5678_001.csv"
content = File.read!("/path/to/file.csv")

{:ok, result} = DaProductApp.Settlements.process_alipayplus_file(filename, content)
```

### Schedule SFTP Retrieval
```elixir
{:ok, job} = DaProductApp.Settlements.schedule_alipayplus_retrieval(
  "1000012345",  # participant_id
  "20220421",    # date (optional, defaults to today)
  %{environment: "v1"}  # options
)
```

### Automated Daily Processing
```elixir
# Schedule daily retrieval for all participants
DaProductApp.Settlements.AlipayPlus.Jobs.SftpFileRetrieval.schedule_daily_retrieval()
```

## Currency Handling

The system properly handles currency conversion from smallest units:

- **AED, USD, EUR**: Divide by 100 (e.g., 1960 → 19.60)
- **JPY, KRW**: No division (e.g., 1960 → 1960)
- **Default**: Divide by 100 for other currencies

## Error Handling

The integration provides comprehensive error handling:

1. **SFTP Errors**: Connection timeouts, authentication failures
2. **File Format Errors**: Invalid CSV structure, missing fields
3. **Data Validation Errors**: Invalid dates, amounts, or required fields
4. **Database Errors**: Constraint violations, transaction rollbacks

All errors are logged with appropriate detail levels and stored in audit trails.

## Monitoring and Auditing

### Settlement File Audits
Every file processing operation is audited with:
- Processing status (success/error)
- Error messages
- Metadata about the operation
- Timestamp information

### Logging
Structured logging at multiple levels:
- INFO: Successful operations
- WARN: Data mismatches or reconciliation issues
- ERROR: Processing failures or system errors

## Testing

The implementation includes comprehensive testing:

- **Unit Tests**: Individual component testing
- **Integration Tests**: End-to-end workflow testing
- **API Tests**: Endpoint functionality verification

Run tests with:
```bash
mix test test/da_product_app/settlements/alipay_plus/
```

## Security Considerations

1. **SFTP Security**: Uses SSH key-based authentication
2. **Data Encryption**: All file transfers are encrypted
3. **Access Control**: API endpoints require proper authentication
4. **Audit Trails**: Complete logging of all operations
5. **Data Validation**: Strict validation of all input data

## Performance

The implementation is designed for high performance:

- **Asynchronous Processing**: Uses Oban for background jobs
- **Efficient Parsing**: Optimized CSV parsing with NimbleCSV
- **Database Optimization**: Proper indexing on key fields
- **Connection Pooling**: Reuses SFTP connections where possible

## Maintenance

### Regular Tasks
1. Monitor SFTP connection health
2. Review settlement reconciliation reports
3. Check background job queue health
4. Rotate SFTP credentials as required

### Troubleshooting
Common issues and solutions are documented in the module documentation of each component.