# MQTT Message Flow for Terminal Parameter Push and File Download

This document describes the complete MQTT message flow for the terminal parameter push and file download features.

## Architecture Overview

```
[Web UI/API] → [Phoenix App] → [MQTT Broker] → [Terminal Device]
     ↑                                              ↓
[User Interface] ← [Job Status] ← [MQTT Response] ← [Device Response]
```

## Parameter Push Flow

### 1. Parameter Push Request

**Trigger:** User clicks "Push Parameters" in UI or calls API endpoint
**Direction:** Phoenix App → MQTT Broker → Terminal Device

**Topic:** `/ota/{product_key}/{device_sn}/update`

**Message Payload:**
```json
{
  "request_id": 197,
  "merchant_config": true,
  "merchant_id": "900890089008000",
  "terminal_id": "90080002",
  "mqtt_ip": "testapp.ariticapp.com",
  "mqtt_port": 1883,
  "http_ip": "demo.ctrmv.com",
  "http_port": 4001,
  "product_key": "pFppbioOCKlo5c8E",
  "product_secret": "sj2AJl102397fQAV",
  "client_id": "device001",
  "username": "user001",
  "mqtt_topic": "/ota/pFppbioOCKlo5c8E/device001/update",
  "keepalive_time": 300,
  "play_language": -1,
  "heartbeat_interval": 300
}
```

### 2. Parameter Push Acknowledgment

**Trigger:** Terminal device receives and processes parameter push
**Direction:** Terminal Device → MQTT Broker → Phoenix App

**Topic:** `/ota/ack/{device_serial}`

**Message Payload:**
```json
{
  "request_id": "197",
  "status": "OK"
}
```

**Handler:** `DaProductApp.MQTT.Handler.handle_message/3` (ota/ack pattern)

## File Download Flow

### 1. File Download Command

**Trigger:** User clicks "File Download" in UI or calls API endpoint
**Direction:** Phoenix App → MQTT Broker → Terminal Device

**Topic:** `/ota/{product_key}/{device_sn}/update`

**Message Payload:**
```json
{
  "command": "file_download",
  "local_path_to_save": "exdata",
  "url_path_from_download": "https://example.com/files/mainb.png",
  "file_size": "2048",
  "file_category": "logo_image",
  "file_name": "mainb.png",
  "merchant_config": "true",
  "retry_count": "3",
  "request_id": "download_001"
}
```

### 2. File Download Response

**Trigger:** Terminal device completes file download (success or failure)
**Direction:** Terminal Device → MQTT Broker → Phoenix App

**Topic:** `/ota/file_download/{device_serial}`

**Success Response:**
```json
{
  "request_id": "download_001",
  "status": "success",
  "file_name": "mainb.png",
  "file_path": "exdata\\mainb.png"
}
```

**Failure Response:**
```json
{
  "request_id": "download_001",
  "status": "failed",
  "error_code": -2,
  "file_name": "mainb.png"
}
```

**Handler:** `DaProductApp.MQTT.Handler.handle_message/3` (ota/file_download pattern)

## Service Layer Architecture

### Parameter Push Services

1. **ParameterPushService** - Main service for parameter operations
   - `send_terminal_parameters/2` - Full parameter set
   - `send_quick_merchant_config/3` - Basic merchant config
   - `batch_send_parameters/2` - Multiple devices

2. **OtaService** - Low-level OTA configuration (existing)
   - `send_ota_configuration/1` - MQTT publish
   - `handle_ota_acknowledgment/3` - Response handling

### File Download Services

1. **FileDownloadService** - Main service for file download operations
   - `send_file_download_command/2` - Generic file download
   - `create_logo_download_command/3` - Logo download convenience
   - `create_firmware_download_command/3` - Firmware download
   - `create_application_download_command/3` - Application download
   - `handle_download_response/3` - Response processing

### Job Workers (Oban)

1. **ParameterPushWorker** - Scheduled parameter push jobs
   - Single device parameter push
   - Batch parameter push
   - Merchant configuration push

2. **FileDownloadWorker** - Scheduled file download jobs
   - Single device file download
   - Batch file download
   - Logo, firmware, application downloads

## Message Routing Configuration

### MQTT Client Configuration

The system uses the existing Tortoise MQTT client with client ID `phoenix_client_tms`.

### Topic Patterns

- **Outbound (Commands):** `/ota/{product_key}/{device_sn}/update`
- **Inbound (Parameter ACK):** `/ota/ack/{device_serial}`
- **Inbound (File Download Response):** `/ota/file_download/{device_serial}`
- **Inbound (Status Updates):** `/tms/status/{serial_number}` (existing)

### QoS Settings

- All command messages use QoS 1 for guaranteed delivery
- Response messages typically use QoS 0

## Error Handling and Retry Logic

### Parameter Push Errors

1. **Validation Errors** - Handled before MQTT publish
2. **MQTT Publish Errors** - Logged and job marked as failed
3. **Device Rejection** - Handled via ACK message with error status
4. **Timeout** - No ACK received within timeout period

### File Download Errors

1. **Validation Errors** - Missing required fields
2. **MQTT Publish Errors** - Network or broker issues
3. **Download Errors** - Device cannot access URL or save file
4. **File Corruption** - Size mismatch or checksum failure

### Retry Strategy

- Oban jobs automatically retry up to 3 times for MQTT publish failures
- File downloads have device-level retry configuration (retry_count parameter)
- Failed jobs are logged with detailed error information

## Real-time Updates

### Phoenix PubSub Integration

- Parameter push responses broadcast to `ota:ack:#{device_id}`
- File download responses broadcast to `file_download:#{device_serial}`
- Terminal status updates broadcast to `tms:terminals`

### UI Live Updates

- Application upgrade interface subscribes to relevant PubSub topics
- Real-time job status updates without page refresh
- Live device response notifications

## Monitoring and Logging

### Log Entries

All operations generate structured log entries including:
- Request ID for correlation
- Device serial number
- Operation type (parameter_push, file_download)
- Timestamp
- Success/failure status
- Error details when applicable

### Job Tracking

- Oban provides job execution history
- Job status visible in web interface
- Detailed error messages for debugging

### Device Response Correlation

- Each command includes unique request_id
- Responses matched by request_id for proper correlation
- Timeout handling for missing responses

## Security Considerations

### Parameter Validation

- All required fields validated before sending
- Product keys and secrets handled securely
- Device serial number validation

### File Download Security

- URL validation for download sources
- File type and size restrictions
- Path traversal prevention
- Retry limits to prevent abuse

### MQTT Security

- Uses existing MQTT authentication
- Topic-based access control
- SSL/TLS encryption (if configured)

## Performance Considerations

### Batch Operations

- Batch parameter push sends to multiple devices efficiently
- Unique request IDs per device in batch operations
- Rate limiting to prevent MQTT broker overload

### Job Queue Management

- Oban manages job execution and concurrency
- Failed jobs don't block queue
- Configurable queue sizes and worker counts

### Memory Usage

- Streaming for large file downloads
- Efficient JSON payload encoding
- Proper cleanup of completed operations