# TMS Management System - OTA Configuration & Device Monitoring

This document provides comprehensive information about the Terminal Management System (TMS) implementation for device configuration and monitoring.

## 📋 Overview

The TMS Management System implements two key functionalities:

1. **OTA Configuration Push** - Send configuration updates to devices via MQTT
2. **Enhanced Device Monitoring** - Track device heartbeats and monitor online status

## 🚀 OTA Configuration Push

### Purpose
Allows the TMS to push configuration updates to payment devices over-the-air using MQTT protocol.

### Architecture

```
TMS Server → MQTT Broker → Payment Device
    ↓              ↓            ↓
  Database    Topic: /ota/{product_key}/{client_id}/update
               ACK: /ota/ack/{device_serial}
```

### Configuration Schema

The `OtaConfiguration` schema stores configuration data:

```elixir
%OtaConfiguration{
  request_id: integer,
  merchant_config: boolean,
  merchant_id: string,
  terminal_id: string,
  device_serial: string,
  mqtt_ip: string,
  mqtt_port: integer,
  http_ip: string,
  http_port: integer,
  product_key: string,
  product_secret: string,
  client_id: string,
  username: string,
  mqtt_topic: string,
  keepalive_time: integer,
  play_language: integer,
  heartbeat_interval: integer,
  status: "pending" | "sent" | "acknowledged" | "failed",
  sent_at: datetime,
  acknowledged_at: datetime,
  error_message: string
}
```

### API Endpoints

#### 1. Send Configuration
```http
POST /api/ota/configuration
Content-Type: application/json

{
  "device_serial": "38250416780001",
  "merchant_config": {
    "merchant_id": "900890089008000",
    "terminal_id": "90080002",
    "mqtt_ip": "testapp.ariticapp.com",
    "mqtt_port": 1883,
    "http_ip": "demo.ctrmv.com",
    "http_port": 4001,
    "product_key": "pFppbioOCKlo5c8E",
    "product_secret": "sj2AJl102397fQAV",
    "username": "user001",
    "keepalive_time": 300,
    "play_language": 1,
    "heartbeat_interval": 300
  }
}
```

**Response:**
```json
{
  "status": "success",
  "message": "Configuration sent successfully"
}
```

#### 2. Get Configuration Status
```http
GET /api/ota/configuration/{device_serial}
```

**Response:**
```json
{
  "status": "success",
  "data": {
    "id": 1,
    "request_id": 197001,
    "device_serial": "38250416780001",
    "merchant_id": "900890089008000",
    "terminal_id": "90080002",
    "status": "acknowledged",
    "sent_at": "2025-06-30T16:45:00Z",
    "acknowledged_at": "2025-06-30T16:45:15Z",
    "error_message": null,
    "heartbeat_interval": 300
  }
}
```

#### 3. List All Configurations
```http
GET /api/ota/configurations?status=pending&device_serial=382504
```

#### 4. Retry Failed Configuration
```http
POST /api/ota/configuration/{id}/retry
```

### MQTT Topics

#### Configuration Push (TMS → Device)
- **Topic:** `/ota/{product_key}/{client_id}/update`
- **QoS:** 1 (At least once delivery)
- **Payload:** JSON configuration object

**Example Payload:**
```json
{
  "request_id": 197001,
  "merchant_config": true,
  "merchant_id": "900890089008000",
  "terminal_id": "90080002",
  "mqtt_ip": "testapp.ariticapp.com",
  "mqtt_port": 1883,
  "http_ip": "demo.ctrmv.com",
  "http_port": 4001,
  "product_key": "pFppbioOCKlo5c8E",
  "product_secret": "sj2AJl102397fQAV",
  "client_id": "38250416780001",
  "username": "user001",
  "mqtt_topic": "/ota/pFppbioOCKlo5c8E/38250416780001/update",
  "keepalive_time": 300,
  "play_language": 1,
  "heartbeat_interval": 300
}
```

#### Acknowledgment (Device → TMS)
- **Topic:** `/ota/ack/{device_serial}`
- **Payload:** JSON acknowledgment

**Example Acknowledgment:**
```json
{
  "request_id": 197001,
  "status": "OK"
}
```

## 📊 Enhanced Device Monitoring

### Purpose
Track device heartbeats and automatically detect offline devices for alerting and dashboard display.

### Device Monitor Schema

```elixir
%DeviceMonitor{
  device_serial: string,
  last_heartbeat: datetime,
  heartbeat_interval: integer,  # seconds
  status: "online" | "offline" | "disconnected",
  missed_heartbeats: integer,
  alert_sent: boolean,
  metadata: map
}
```

### Heartbeat Processing

#### Incoming Heartbeat (Device → TMS)
- **Topic:** `/tms/status/{device_serial}`
- **Frequency:** Every 300 seconds (configurable)

**Example Heartbeat Payload:**
```json
{
  "oid": "org-id-placeholder",
  "sn": "38250416780001",
  "uploadTime": "2025-04-26T00:00:00Z",
  "org.device": [
    {
      "itemkey": "status",
      "value": "connected",
      "timestamp": "2025-04-26T00:00:00Z",
      "message": "Operating normally"
    },
    {
      "itemkey": "battery",
      "value": "85",
      "timestamp": "2025-04-26T00:00:00Z",
      "message": "Battery level normal"
    },
    {
      "itemkey": "memory",
      "value": "45",
      "timestamp": "2025-04-26T00:00:00Z",
      "message": "Memory usage"
    }
  ]
}
```

### Offline Detection

Devices are considered offline when:
- No heartbeat received within `1.5 × heartbeat_interval`
- Default threshold: 450 seconds (7.5 minutes) for 300-second interval

### Monitoring API Endpoints

#### 1. Get Device Monitor Status
```http
GET /api/device/{device_serial}/monitor
```

**Response:**
```json
{
  "status": "success",
  "data": {
    "device_serial": "38250416780001",
    "last_heartbeat": "2025-06-30T16:40:00Z",
    "heartbeat_interval": 300,
    "status": "online",
    "missed_heartbeats": 0,
    "alert_sent": false,
    "is_offline": false,
    "metadata": {}
  }
}
```

#### 2. Check Offline Devices
```http
POST /api/device/monitor/check_offline
```

#### 3. Get Monitoring Statistics
```http
GET /api/device/monitor/stats
```

**Response:**
```json
{
  "status": "success",
  "data": {
    "total_devices": 150,
    "online_devices": 142,
    "offline_devices": 6,
    "disconnected_devices": 2,
    "missed_heartbeat_devices": 8,
    "online_rate": 94.7
  }
}
```

## 🔐 Security Features

### Credential Validation
- Product keys and secrets are validated before sending
- Secure storage of sensitive configuration data
- Request ID tracking prevents replay attacks

### Access Control
- API endpoints require authentication
- Configuration changes are logged and tracked
- Error messages don't expose sensitive information

## 🔄 Background Processes

### Offline Monitor
- Runs every 5 minutes
- Checks all devices for missed heartbeats
- Sends alerts for newly offline devices
- Updates device status automatically

### Implementation
```elixir
# Supervised process in application.ex
DaProductApp.TerminalManagement.OfflineMonitor
```

## 📈 Monitoring & Alerting

### Dashboard Integration
- Real-time device status updates
- Online/offline rate calculations
- Device health metrics (battery, memory, CPU)

### Alert System
- Automatic detection of offline devices
- Configurable alert thresholds
- Prevention of duplicate alerts

### Recommendations
1. **Dashboard Monitoring:** Flag devices offline for > 1.5 × heartbeat_interval
2. **Alert Escalation:** Send alerts after 2 missed consecutive heartbeats
3. **Health Metrics:** Track battery, network strength, software versions
4. **Maintenance Windows:** Account for scheduled downtime

## 🧪 Testing

### Unit Tests
```bash
# Run OTA service tests
mix test test/da_product_app/terminal_management/ota_service_test.exs

# Run device monitor tests  
mix test test/da_product_app/terminal_management/device_monitor_test.exs
```

### Integration Testing
- Mock MQTT broker for testing
- Database transaction rollback for test isolation
- Comprehensive error scenario coverage

## 🚀 Deployment Considerations

### Database Migrations
```bash
# Apply migrations
mix ecto.migrate
```

### MQTT Configuration
- Ensure MQTT broker connectivity
- Configure appropriate QoS levels
- Set up topic permissions

### Performance
- Index device_serial columns for fast lookups
- Consider partitioning for high-volume deployments
- Monitor MQTT message throughput

## 📝 Example Usage Scenarios

### 1. Merchant Configuration Update
```bash
curl -X POST http://localhost:4000/api/ota/configuration \
  -H "Content-Type: application/json" \
  -d '{
    "device_serial": "38250416780001",
    "merchant_config": {
      "merchant_id": "MERCHANT_123",
      "terminal_id": "TERM_456",
      "mqtt_ip": "mqtt.example.com"
    }
  }'
```

### 2. Device Status Check
```bash
curl http://localhost:4000/api/device/38250416780001/monitor
```

### 3. System Health Dashboard
```bash
curl http://localhost:4000/api/device/monitor/stats
```

This implementation provides a robust foundation for terminal management with comprehensive configuration management and device monitoring capabilities.