# Parameter Management System - Test Suite

This comprehensive test suite covers the Parameter Management System with both API testing and UI browser interactions using Playwright. The tests are designed based on the features described in `PARAMETER_MANAGEMENT_GUIDE.md`.

## 🔧 Test Coverage

### Core Features Tested
- **Parameter CRUD Operations**: Create, read, update, delete functionality
- **Parameter Templates**: Vendor/model-specific configuration management
- **Device Overrides**: Terminal-specific parameter customization
- **Bulk Operations**: Mass parameter updates across device groups
- **Real-time Monitoring**: Live device health and status tracking
- **Multi-tab Dashboards**: Analytics, Alerts, and Audit interfaces
- **Audit Trail**: Complete history of parameter changes
- **Notification System**: Multi-channel alert management

### Test Categories

#### 1. API Tests (`parameter-management.spec.ts`)
- ✅ **GET** `/api/terminals/:serial/parameters` - Parameter resolution
- ✅ **POST** `/api/terminals/:serial/apply-template` - Template application
- ✅ **PUT** `/api/terminals/:serial/parameters/:key` - Parameter overrides
- ✅ **GET** `/api/terminals/:serial/parameter-history` - Push history

#### 2. UI Interaction Tests (`parameter-management.spec.ts`)
- ✅ **Dashboard Navigation**: Main parameter management interface
- ✅ **Template Management**: Create and manage parameter templates
- ✅ **Device Parameter Overrides**: Terminal-specific configurations
- ✅ **Bulk Operations**: Mass parameter deployment
- ✅ **Search and Filtering**: Advanced parameter search functionality

#### 3. Advanced Dashboard Tests (`advanced-dashboard.spec.ts`)
- ✅ **6-Tab Analytics Dashboard**: Overview, Trends, Performance, Geographic, Compliance, Export
- ✅ **5-Tab Alert Management**: Dashboard, Active Alerts, Rules, Notifications, Escalation
- ✅ **4-Tab Audit Interface**: Audit Trail, Compliance, Security, Reports
- ✅ **Real-time Updates**: WebSocket-based live data streaming
- ✅ **Export Functionality**: CSV, JSON, PDF report generation

#### 4. Responsive & Performance Tests (`responsive-performance.spec.ts`)
- ✅ **Mobile Compatibility**: iPhone/Android responsive design
- ✅ **Tablet Interface**: iPad and tablet optimizations
- ✅ **Desktop Experience**: Full desktop functionality
- ✅ **Performance Monitoring**: Load times and responsiveness
- ✅ **Accessibility Testing**: Keyboard navigation and screen readers
- ✅ **Error Handling**: Network issues and data validation

## 🚀 Quick Start

### Prerequisites
- Node.js and npm installed
- Phoenix application running on remote server
- Playwright Test extension in VS Code (optional)

### Installation
```bash
# Install dependencies (if not already done)
npm install -D @playwright/test @types/node
npx playwright install
```

### Running Tests

#### Option 1: Complete Test Suite
```bash
# Run all tests including Elixir unit tests
./test_all.sh

# Or specify custom server URL
BASE_URL=http://your-server:4000 ./test_all.sh
```

#### Option 2: Parameter Management Tests Only
```bash
# Run just parameter management tests
./test_parameter_management.sh

# Or with custom server
./test_parameter_management.sh http://your-server:4000
```

#### Option 3: Individual Test Files
```bash
# API and basic UI tests
npx playwright test parameter-management.spec.ts

# Advanced dashboard features
npx playwright test advanced-dashboard.spec.ts

# Responsive and performance tests
npx playwright test responsive-performance.spec.ts

# Settlement and dispute APIs
npx playwright test settlement-api.spec.ts
```

#### Option 4: Specific Test Categories
```bash
# Run only mobile tests
npx playwright test --grep "Mobile"

# Run only API tests
npx playwright test --grep "API"

# Run only dashboard tests
npx playwright test --grep "Dashboard"
```

## 📊 Test Reports

### HTML Report
```bash
# Generate and view detailed HTML report
npx playwright show-report
```

### Line Reporter (CI/CD friendly)
```bash
# Run with line reporter for logs
npx playwright test --reporter=line
```

### JSON Report
```bash
# Generate JSON report for integration
npx playwright test --reporter=json
```

## 🔧 Configuration

### Environment Variables
Create `.env` file:
```bash
# Remote server configuration
BASE_URL=http://your-server:4000

# For demo server
BASE_URL=http://demo.ctrmv.com:4090
```

### Playwright Configuration
The `playwright.config.ts` is optimized for remote server testing:
- ✅ Configurable base URL via environment variables
- ✅ Multiple browser support (Chrome, Firefox, Safari)
- ✅ Mobile device emulation
- ✅ Automatic retry on failures
- ✅ Trace collection for debugging

## 🎯 Test Scenarios Based on Parameter Management Guide

### Parameter Categories Structure
Tests validate the hierarchical structure:
- **Trade Module** (security, ID, print, session)
- **Communication Module** (general, host settings)
- **Transactions** (financial, administrative)
- **Card Module** (card ranges)
- **APK** (MFSetting, launcher, appstore, system)

### Vendor/Model Templates
Tests cover template management for:
- **Ingenico iWL250** configurations
- **Custom vendor** templates
- **Bulk template application** across device groups

### Real-time Features
Tests validate live functionality:
- **Device health monitoring** (Excellent, Good, Fair, Poor, Critical)
- **Parameter push status** tracking
- **WebSocket updates** for real-time data
- **Alert notifications** across multiple channels

### Multi-channel Notifications
Tests cover notification systems:
- **📧 Email** notifications with SMTP
- **📱 SMS** gateway integration
- **💬 Slack** webhook notifications
- **🔗 Webhook** custom endpoints
- **🔔 Push** notifications for mobile/web

## 🐛 Debugging Failed Tests

### View Failed Test Details
```bash
# Run with debug info
npx playwright test --debug

# Run specific failing test
npx playwright test --grep "specific test name"
```

### Screenshot on Failure
Screenshots are automatically captured on failure and stored in `test-results/`

### Trace Viewer
```bash
# View detailed trace for failed tests
npx playwright show-trace test-results/trace.zip
```

### Server Connectivity Issues
```bash
# Check if server is reachable
curl -I http://your-server:4000

# Test specific API endpoint
curl http://your-server:4000/api/terminals/38250416780001/parameters
```

## 📱 Mobile & Responsive Testing

### Supported Viewports
- **📱 Mobile**: iPhone SE (375x667), Android (360x640)
- **📟 Tablet**: iPad (768x1024), Android Tablet (800x1280)
- **💻 Desktop**: 1920x1080, 1366x768

### Mobile-Specific Tests
- Touch navigation and gestures
- Mobile menu functionality
- Responsive form layouts
- Mobile-optimized dashboards

## ♿ Accessibility Testing

### WCAG Compliance Checks
- **Keyboard Navigation**: Tab order and Enter key functionality
- **Screen Reader**: ARIA labels and semantic HTML
- **Color Contrast**: Visual accessibility validation
- **Focus Management**: Visible focus indicators

### Accessibility Test Results
Tests automatically check for:
- Proper heading structure (h1-h6)
- ARIA attributes and roles
- Color contrast ratios
- Keyboard accessibility

## 🔄 Continuous Integration

### GitHub Actions Integration
```yaml
- name: Run Parameter Management Tests
  run: |
    npm ci
    npx playwright install
    ./test_parameter_management.sh ${{ env.BASE_URL }}
```

### Test Status Badges
Add to your README:
```markdown
![Parameter Management Tests](https://github.com/your-org/repo/workflows/playwright-tests/badge.svg)
```

## 📈 Performance Benchmarks

### Expected Performance Metrics
- **Page Load**: < 3 seconds for dashboard pages
- **API Response**: < 500ms for parameter operations
- **Real-time Updates**: < 1 second for WebSocket updates
- **Bulk Operations**: < 60 seconds for template application

### Performance Test Results
Performance tests automatically validate:
- Dashboard load times
- Real-time update frequency
- Bulk operation completion times
- Network interruption recovery

## 🔍 Test Data and Fixtures

### Sample Test Data
Tests use realistic data based on the guide:
```javascript
const testSettlement = {
  settlement_id: "SETT20250719",
  date: "2025-07-19", 
  amount: "1000.00",
  status: "settled",
  transaction_count: 5,
  provider: "QR_AANI",
  merchant_id: "MRC123"
};
```

### Parameter Test Fixtures
- Terminal serial: `38250416780001`
- Template: `Ingenico iWL250 Configuration`
- Parameters: MQTT settings, heartbeat intervals, merchant IDs

## 📞 Support and Troubleshooting

### Common Issues

1. **Server Not Reachable**: Update `BASE_URL` in `.env`
2. **Authentication Required**: Tests may need login credentials
3. **Timeout Errors**: Increase timeout in `playwright.config.ts`
4. **Mobile Tests Failing**: Check viewport configurations

### Getting Help
- Check `test-results/` for detailed error logs
- Use `npx playwright show-report` for visual debugging
- Review server logs for API endpoint issues
- Verify Phoenix application is running and accessible

---

This test suite provides comprehensive coverage of your Parameter Management System, ensuring both functionality and user experience meet enterprise standards. The tests are designed to run efficiently in both development and CI/CD environments while providing detailed feedback for debugging and optimization.
