defmodule DaProductApp.Application do
  # See https://hexdocs.pm/elixir/Application.html
  # for more information on OTP Applications
  @moduledoc false
  use Application

  @impl true
  def start(_type, _args) do
    children = [
      DaProductApp.DeviceRegistry,
      # Existing children (keep all these)
      DaProductAppWeb.Telemetry,
      DaProductApp.Repo,
      {DNSCluster, query: Application.get_env(:da_product_app, :dns_cluster_query) || :ignore},
      {Phoenix.PubSub, name: DaProductApp.PubSub},
      {Finch, name: DaProductApp.Finch},
      {Cachex, name: :general_cache},
      TwMerge.Cache,

      # NEW: Add Ranch TCP listener for ISO8583
      # {DaProductApp.TCP.Supervisor, [port: 5001]},  # Customize port if needed

      # Start MQTT client supervisor
      {DaProductApp.MQTT.Supervisor, []},
      # Ensure the client is started at boot:
      {Task, fn -> DaProductApp.MQTT.start_phoenix_client() end},

      # Start offline device monitor
      DaProductApp.TerminalManagement.OfflineMonitor,

      # Start Terminal Event System for automatic rule management
      DaProductApp.TerminalManagement.TerminalEventSupervisor,

      # Start Filter Cache Service for high-performance filter options
      DaProductApp.TerminalManagement.FilterCacheService,

      # Start Remote Log Service components
      {Registry, keys: :unique, name: DaProductApp.RemoteLogRegistry},
      {DynamicSupervisor, name: DaProductApp.RemoteLogSupervisor, strategy: :one_for_one},

      # Start AlipayPlus Settlement Scheduler
      DaProductApp.Settlements.AlipayPlus.Scheduler,
      {Oban, oban_config()},

      # Start Phoenix Endpoint (Bandit is configured here later)
      DaProductAppWeb.Endpoint
    ]

    # See https://hexdocs.pm/elixir/Supervisor.html
    # for other strategies and supported options
    opts = [strategy: :one_for_one, name: DaProductApp.Supervisor]
    Supervisor.start_link(children, opts)
  end

  # Tell Phoenix to update the endpoint configuration
  # whenever the application is updated.
  @impl true
  def config_change(changed, _new, removed) do
    DaProductAppWeb.Endpoint.config_change(changed, removed)
    :ok
  end

  #  defp oban_config do
  #  %{
  #    name: Oban,
  #    repo: DaProductApp.Repo,
  #    engine: Oban.Engines.Basic,   # Use Basic Engine instead of Postgres
  #    queues: [default: 10],        # Define your queues
  #    notifier: Oban.Notifiers.Null, # Disable PostgreSQL-based notifier
  #    peer: false                    # Disable peer tracking (Postgres-only feature)
  # }
  # end
  defp oban_config do
    Application.fetch_env!(:da_product_app, Oban)
    |> Keyword.put(:queues,
      default: 10,
      file_download: 5,
      parameter_push: 5,
      app_package: 5,
      settlements: 3
    )
  end
end
