defmodule DaProductApp.Disputes.Dispute do
  use Ecto.Schema
  import Ecto.Changeset

  @primary_key {:id, :binary_id, autogenerate: true}
  @foreign_key_type :binary_id

  schema "disputes" do
    field :dispute_id, :string
    field :txn_id, :string
    field :reason, :string
    field :merchant_comment, :string
    field :ops_response, :string
    field :contact_email, :string
    field :status, :string, default: "open"
    field :bank_user_id, :string

    timestamps(type: :utc_datetime)
  end

  @doc false
  def changeset(dispute, attrs) do
    dispute
    |> cast(attrs, [
      :dispute_id,
      :txn_id,
      :reason,
      :merchant_comment,
      :ops_response,
      :contact_email,
      :status,
      :bank_user_id
    ])
    |> validate_required([:dispute_id, :txn_id, :reason, :contact_email])
    |> validate_inclusion(:status, ["open", "resolved", "rejected"])
    |> validate_format(:contact_email, ~r/^[^\s]+@[^\s]+\.[^\s]+$/)
    |> unique_constraint(:dispute_id)
  end

  def generate_dispute_id do
    timestamp = DateTime.utc_now() |> DateTime.to_unix()
    random_part = :crypto.strong_rand_bytes(6) |> Base.encode16(case: :lower)
    "DSP#{timestamp}-#{random_part}"
  end
end
