defmodule DaProductApp.ParameterManagement.DeviceParameterOverride do
  use Ecto.Schema
  import Ecto.Changeset
  import Ecto.Query

  @primary_key {:id, :id, autogenerate: true}
  @foreign_key_type :id

  schema "device_parameter_overrides" do
    field :value, :string
    field :source, :string, default: "manual"
    field :applied_at, :utc_datetime

    belongs_to :terminal, DaProductApp.TerminalManagement.TmsTerminal, type: :integer
    belongs_to :parameter_definition, DaProductApp.ParameterManagement.ParameterDefinition
    belongs_to :created_by, DaProductApp.Users.User

    timestamps()
  end

  @valid_sources ~w(manual api bulk_import)

  def changeset(override, attrs) do
    override
    |> cast(attrs, [
      :terminal_id,
      :parameter_definition_id,
      :value,
      :source,
      :applied_at,
      :created_by_id
    ])
    |> validate_required([:terminal_id, :parameter_definition_id, :value])
    |> validate_inclusion(:source, @valid_sources)
    |> unique_constraint([:terminal_id, :parameter_definition_id])
    |> validate_parameter_value()
  end

  def by_terminal(query \\ __MODULE__, terminal_id) do
    from o in query,
      join: pd in assoc(o, :parameter_definition),
      where: o.terminal_id == ^terminal_id,
      preload: [:parameter_definition],
      order_by: [asc: pd.display_order]
  end

  def by_parameter(query \\ __MODULE__, parameter_definition_id) do
    from o in query, where: o.parameter_definition_id == ^parameter_definition_id
  end

  def applied_overrides(query \\ __MODULE__) do
    from o in query, where: not is_nil(o.applied_at)
  end

  def pending_overrides(query \\ __MODULE__) do
    from o in query, where: is_nil(o.applied_at)
  end

  defp validate_parameter_value(changeset) do
    case get_field(changeset, :parameter_definition_id) do
      nil ->
        changeset

      param_id ->
        # Load the parameter definition to validate the value
        case DaProductApp.Repo.get(DaProductApp.ParameterManagement.ParameterDefinition, param_id) do
          nil ->
            add_error(changeset, :parameter_definition_id, "does not exist")

          parameter ->
            value = get_field(changeset, :value)

            case DaProductApp.ParameterManagement.ParameterDefinition.validate_value(
                   parameter,
                   value
                 ) do
              :ok -> changeset
              {:error, message} -> add_error(changeset, :value, message)
            end
        end
    end
  end
end
