defmodule DaProductApp.ParameterManagement.ParameterCategory do
  use Ecto.Schema
  import Ecto.Changeset
  import Ecto.Query

  @primary_key {:id, :id, autogenerate: true}
  @foreign_key_type :id

  schema "parameter_categories" do
    field :name, :string
    field :code, :string
    field :description, :string
    field :sort_order, :integer, default: 0
    field :is_active, :boolean, default: true

    belongs_to :parent, __MODULE__
    has_many :children, __MODULE__, foreign_key: :parent_id
    has_many :parameter_definitions, DaProductApp.ParameterManagement.ParameterDefinition

    timestamps()
  end

  def changeset(category, attrs) do
    category
    |> cast(attrs, [:name, :code, :description, :sort_order, :is_active, :parent_id])
    |> validate_required([:name, :code])
    |> validate_length(:name, max: 255)
    |> validate_length(:code, max: 100)
    |> unique_constraint(:code)
    |> validate_number(:sort_order, greater_than_or_equal_to: 0)
  end

  def active_query(query \\ __MODULE__) do
    from c in query, where: c.is_active == true
  end

  def root_categories(query \\ __MODULE__) do
    from c in query, where: is_nil(c.parent_id), order_by: [asc: c.sort_order, asc: c.name]
  end

  def with_children(query \\ __MODULE__) do
    children_query = from(child in __MODULE__, order_by: [asc: child.sort_order, asc: child.name])
    from c in query, preload: [children: ^children_query]
  end
end
