defmodule DaProductApp.ParameterManagement.ParameterPushLog do
  use Ecto.Schema
  import Ecto.Query
  import Ecto.Changeset

  @primary_key {:id, :id, autogenerate: true}
  @foreign_key_type :id

  schema "parameter_push_logs" do
    field :parameters_sent, :map
    field :request_id, :integer
    field :status, :string, default: "pending"
    field :sent_at, :utc_datetime
    field :acknowledged_at, :utc_datetime
    field :error_message, :string
    field :push_type, :string, default: "full"

    belongs_to :terminal, DaProductApp.TerminalManagement.TmsTerminal, type: :integer
    belongs_to :template, DaProductApp.ParameterManagement.ParameterTemplate
    belongs_to :triggered_by, DaProductApp.Users.User
    has_many :push_details, DaProductApp.ParameterManagement.ParameterPushDetail

    timestamps()
  end

  @valid_statuses ~w(pending sent acknowledged failed)
  @valid_push_types ~w(full partial single)

  def changeset(push_log, attrs) do
    push_log
    |> cast(attrs, [
      :terminal_id,
      :template_id,
      :parameters_sent,
      :request_id,
      :status,
      :sent_at,
      :acknowledged_at,
      :error_message,
      :push_type,
      :triggered_by_id
    ])
    |> validate_required([:terminal_id, :request_id])
    |> validate_inclusion(:status, @valid_statuses)
    |> validate_inclusion(:push_type, @valid_push_types)
    |> unique_constraint([:terminal_id, :request_id])
  end

  def by_terminal(query \\ __MODULE__, terminal_id) do
    from p in query, where: p.terminal_id == ^terminal_id, order_by: [desc: p.inserted_at]
  end

  def by_status(query \\ __MODULE__, status) do
    from p in query, where: p.status == ^status
  end

  def recent_pushes(query \\ __MODULE__, hours \\ 24) do
    hours_ago = DateTime.utc_now() |> DateTime.add(-hours, :hour)
    from p in query, where: p.inserted_at >= ^hours_ago, order_by: [desc: p.inserted_at]
  end

  def with_details(query \\ __MODULE__) do
    from p in query, preload: [:push_details, :template, :terminal]
  end
end
