defmodule DaProductApp.ParameterManagement.ParameterTemplate do
  use Ecto.Schema
  import Ecto.Changeset
  import Ecto.Query

  @primary_key {:id, :id, autogenerate: true}
  @foreign_key_type :id

  schema "parameter_templates" do
    field :name, :string
    field :description, :string
    field :vendor, :string
    field :model, :string
    field :is_default, :boolean, default: false
    field :is_active, :boolean, default: true

    belongs_to :created_by, DaProductApp.Users.User
    has_many :template_values, DaProductApp.ParameterManagement.ParameterTemplateValue
    has_many :push_logs, DaProductApp.ParameterManagement.ParameterPushLog

    timestamps()
  end

  def changeset(template, attrs) do
    template
    |> cast(attrs, [:name, :description, :vendor, :model, :is_default, :is_active, :created_by_id])
    |> validate_required([:name])
    |> validate_length(:name, max: 255)
    |> validate_length(:vendor, max: 100)
    |> validate_length(:model, max: 100)
    |> unique_constraint([:vendor, :model, :name])
  end

  def active_query(query \\ __MODULE__) do
    from t in query, where: t.is_active == true
  end

  def by_vendor(query \\ __MODULE__, vendor) do
    from t in query, where: t.vendor == ^vendor or t.vendor == "universal"
  end

  def by_model(query \\ __MODULE__, model) do
    from t in query, where: t.model == ^model or t.model == "universal"
  end

  def by_vendor_and_model(query \\ __MODULE__, vendor, model) do
    from t in query,
      where:
        (t.vendor == ^vendor or t.vendor == "universal") and
          (t.model == ^model or t.model == "universal"),
      order_by: [
        desc:
          fragment(
            "CASE WHEN ? = ? AND ? = ? THEN 3 WHEN ? = ? THEN 2 WHEN ? = ? THEN 1 ELSE 0 END",
            t.vendor,
            ^vendor,
            t.model,
            ^model,
            t.vendor,
            ^vendor,
            t.model,
            ^model
          )
      ]
  end

  def default_templates(query \\ __MODULE__) do
    from t in query, where: t.is_default == true
  end

  def with_template_values(query \\ __MODULE__) do
    template_values_query =
      from(tv in DaProductApp.ParameterManagement.ParameterTemplateValue,
        join: pd in assoc(tv, :parameter_definition),
        preload: [:parameter_definition],
        order_by: [asc: pd.display_order]
      )

    from t in query,
      preload: [template_values: ^template_values_query]
  end
end
