defmodule DaProductApp.ParameterManagement.ParameterTemplateValue do
  use Ecto.Schema
  import Ecto.Changeset
  import Ecto.Query

  @primary_key {:id, :id, autogenerate: true}
  @foreign_key_type :id

  schema "parameter_template_values" do
    field :value, :string
    field :is_overridden, :boolean, default: false

    belongs_to :template, DaProductApp.ParameterManagement.ParameterTemplate
    belongs_to :parameter_definition, DaProductApp.ParameterManagement.ParameterDefinition

    timestamps()
  end

  def changeset(template_value, attrs) do
    template_value
    |> cast(attrs, [:template_id, :parameter_definition_id, :value, :is_overridden])
    |> validate_required([:template_id, :parameter_definition_id])
    |> unique_constraint([:template_id, :parameter_definition_id])
    |> validate_parameter_value()
  end

  def by_template(query \\ __MODULE__, template_id) do
    from tv in query,
      join: pd in assoc(tv, :parameter_definition),
      where: tv.template_id == ^template_id,
      preload: [:parameter_definition],
      order_by: [asc: pd.display_order]
  end

  def with_parameter_and_category(query \\ __MODULE__) do
    from tv in query,
      join: pd in assoc(tv, :parameter_definition),
      join: pc in assoc(pd, :category),
      preload: [parameter_definition: {pd, category: pc}]
  end

  def by_parameter(query \\ __MODULE__, parameter_definition_id) do
    from tv in query, where: tv.parameter_definition_id == ^parameter_definition_id
  end

  defp validate_parameter_value(changeset) do
    case get_field(changeset, :parameter_definition_id) do
      nil ->
        changeset

      param_id ->
        # Load the parameter definition to validate the value
        case DaProductApp.Repo.get(DaProductApp.ParameterManagement.ParameterDefinition, param_id) do
          nil ->
            add_error(changeset, :parameter_definition_id, "does not exist")

          parameter ->
            value = get_field(changeset, :value)

            case DaProductApp.ParameterManagement.ParameterDefinition.validate_value(
                   parameter,
                   value
                 ) do
              :ok -> changeset
              {:error, message} -> add_error(changeset, :value, message)
            end
        end
    end
  end
end
