defmodule DaProductApp.Settlements.AlipayPlus.Jobs.SftpFileRetrieval do
  @moduledoc """
  Oban job for retrieving AlipayPlus settlement files from SFTP server.

  This job is scheduled to run periodically to check for new settlement files
  and download them for processing.
  """

  use Oban.Worker, queue: :settlements, max_attempts: 3

  require Logger

  alias DaProductApp.Settlements.AlipayPlus.SftpClient
  alias DaProductApp.Settlements.AlipayPlus.Jobs.SettlementFileProcessing

  @impl Oban.Worker
  def perform(%Oban.Job{args: args}) do
    participant_id = Map.fetch!(args, "participant_id")
    date = Map.get(args, "date", Date.to_string(Date.utc_today()))
    environment = Map.get(args, "environment", "v1")

    Logger.info("Starting SFTP file retrieval for participant #{participant_id}, date #{date}")

    with {:ok, connection} <- connect_to_sftp(),
         {:ok, file_list} <-
           SftpClient.list_settlement_files(connection, participant_id, date, environment),
         :ok <- download_and_queue_files(connection, file_list, args) do
      SftpClient.disconnect(connection)
      Logger.info("Successfully completed SFTP file retrieval for participant #{participant_id}")
      :ok
    else
      {:error, reason} ->
        Logger.error(
          "SFTP file retrieval failed for participant #{participant_id}: #{inspect(reason)}"
        )

        {:error, reason}
    end
  end

  @doc """
  Schedules an SFTP file retrieval job for a specific participant and date.
  """
  def schedule_retrieval(participant_id, date \\ nil, options \\ %{}) do
    date = date || Date.to_string(Date.utc_today())
    environment = Map.get(options, :environment, "v1")

    args = %{
      "participant_id" => participant_id,
      "date" => date,
      "environment" => environment
    }

    %{args: args}
    |> new()
    |> Oban.insert()
  end

  @doc """
  Schedules daily SFTP file retrieval for all configured participants.
  """
  def schedule_daily_retrieval do
    participant_ids = get_configured_participant_ids()
    # Previous day
    date = Date.to_string(Date.add(Date.utc_today(), -1))

    Enum.each(participant_ids, fn participant_id ->
      schedule_retrieval(participant_id, date)
    end)
  end

  # Private functions

  defp connect_to_sftp do
    sftp_config = get_sftp_config()
    SftpClient.connect(sftp_config)
  end

  defp download_and_queue_files(connection, file_list, original_args) do
    Enum.each(file_list, fn file_info ->
      case SftpClient.download_file(connection, file_info.path) do
        {:ok, content} ->
          # Queue the file for processing
          processing_args =
            Map.merge(original_args, %{
              "filename" => file_info.filename,
              "file_path" => file_info.path,
              "file_size" => file_info.size,
              # Base64 encode for storage
              "content" => Base.encode64(content)
            })

          %{args: processing_args}
          |> SettlementFileProcessing.new()
          |> Oban.insert()

          Logger.info("Queued file for processing: #{file_info.filename}")

        {:error, reason} ->
          Logger.error("Failed to download file #{file_info.filename}: #{inspect(reason)}")
      end
    end)

    :ok
  end

  defp get_sftp_config do
    Application.get_env(:da_product_app, :alipayplus_sftp,
      host: "sftp.alipayplus.com",
      port: 22,
      username: System.get_env("ALIPAYPLUS_SFTP_USERNAME"),
      password: System.get_env("ALIPAYPLUS_SFTP_PASSWORD"),
      connect_timeout: 30_000
    )
  end

  defp get_configured_participant_ids do
    Application.get_env(:da_product_app, :alipayplus_participant_ids, [])
  end
end
