defmodule DaProductApp.Settlements.Settlement do
  use Ecto.Schema
  import Ecto.Changeset

  @primary_key {:id, :binary_id, autogenerate: true}
  @foreign_key_type :binary_id

  schema "settlements" do
    field :settlement_id, :string
    field :date, :date
    field :status, :string
    field :amount, :decimal
    field :details, :map
    field :merchant_tag, :string
    field :bank_user_id, :string
    field :qr_id, :string
    field :batch_number, :string
    field :total_transaction_count, :integer
    field :gross_settlement_amount, :decimal
    field :gross_settlement_currency, :string
    field :mdr_charges, :decimal
    field :mdr_charges_currency, :string
    field :tax_on_mdr, :decimal
    field :tax_on_mdr_currency, :string
    field :net_settlement_amount, :decimal
    field :net_settlement_currency, :string
    field :settlement_timestamp, :utc_datetime

    # AlipayPlus specific fields
    field :participant_id, :string
    field :settlement_batch_id, :string
    field :participant_agreement_id, :string
    field :value_date, :date
    field :fund_direction, :string
    field :transaction_currency, :string
    field :net_transaction_amount_value, :decimal
    field :extend_info, :string

    field :transaction_count, :integer
    field :mismatch_count, :integer
    field :discrepancy_amount, :decimal
    field :resolution_file_required, :boolean, default: false
    belongs_to :merchant, DaProductApp.Merchants.Merchant, type: :string
    belongs_to :provider, DaProductApp.Providers.Provider, type: :integer

    timestamps(type: :utc_datetime)
  end

  @doc false
  def changeset(settlement, attrs) do
    settlement
    |> cast(attrs, [
      :settlement_id,
      :merchant_id,
      :provider_id,
      :date,
      :status,
      :amount,
      :details,
      :merchant_id,
      :provider_id,
      :date,
      :status,
      :amount,
      :details,
      :merchant_tag,
      :bank_user_id,
      :qr_id,
      :batch_number,
      :total_transaction_count,
      :gross_settlement_amount,
      :gross_settlement_currency,
      :mdr_charges,
      :mdr_charges_currency,
      :tax_on_mdr,
      :tax_on_mdr_currency,
      :net_settlement_amount,
      :net_settlement_currency,
      :settlement_timestamp,
      # AlipayPlus specific fields
      :participant_id,
      :settlement_batch_id,
      :participant_agreement_id,
      :value_date,
      :fund_direction,
      :transaction_currency,
      :net_transaction_amount_value,
      :extend_info
    ])
    |> validate_required([:merchant_id, :provider_id, :date, :status, :amount])
    |> validate_required([:date, :status, :amount])
    |> validate_inclusion(:status, ["pending", "settled", "exception"])
    |> validate_inclusion(:fund_direction, ["CREDIT", "DEBIT", nil])
    |> unique_constraint(:settlement_id)
  end

  @doc false
  def aani_changeset(settlement, attrs) do
    settlement
    |> cast(attrs, [
      :merchant_id,
      :provider_id,
      :amount,
      :merchant_tag,
      :bank_user_id,
      :qr_id,
      :batch_number,
      :date,
      :status,
      :total_transaction_count,
      :gross_settlement_amount,
      :gross_settlement_currency,
      :mdr_charges,
      :mdr_charges_currency,
      :tax_on_mdr,
      :tax_on_mdr_currency,
      :net_settlement_amount,
      :net_settlement_currency,
      :settlement_timestamp,
      :mismatch_count,
      :discrepancy_amount,
      :resolution_file_required
    ])
    |> validate_required([
      :merchant_id,
      :provider_id,
      :amount,
      :merchant_tag,
      :bank_user_id,
      :date,
      :status
    ])
    |> generate_settlement_id_if_missing()
  end

  defp generate_settlement_id_if_missing(changeset) do
    case get_field(changeset, :settlement_id) do
      nil ->
        settlement_id = DaProductApp.Settlements.generate_settlement_id()
        put_change(changeset, :settlement_id, settlement_id)

      _ ->
        changeset
    end
  end
end
