defmodule DaProductApp.TerminalManagement.DeviceConfiguration do
  use Ecto.Schema
  import Ecto.Changeset
  alias DaProductApp.TerminalManagement.TmsTerminal

  @derive {Jason.Encoder,
           only: [
             :id,
             :name,
             :description,
             :device_type,
             :configuration_data,
             :status,
             :terminal_id,
             :created_by,
             :updated_by,
             :inserted_at,
             :updated_at
           ]}

  schema "device_configurations" do
    field :name, :string
    field :status, :string, default: "active"
    field :description, :string
    field :device_type, :string
    field :configuration_data, :map
    field :created_by, :string
    field :updated_by, :string

    belongs_to :terminal, TmsTerminal, foreign_key: :terminal_id

    timestamps(type: :utc_datetime)
  end

  @doc false
  def changeset(device_configuration, attrs) do
    device_configuration
    |> cast(attrs, [
      :name,
      :description,
      :device_type,
      :configuration_data,
      :status,
      :terminal_id,
      :created_by,
      :updated_by
    ])
    |> validate_required([:name, :device_type, :configuration_data, :status])
    |> validate_inclusion(:status, ["active", "inactive", "draft"])
    |> validate_inclusion(:device_type, ["SR600 Mini", "P80", "A920", "Other"])
    |> unique_constraint(:name)
    |> validate_configuration_data()
  end

  defp validate_configuration_data(changeset) do
    case get_field(changeset, :configuration_data) do
      nil -> add_error(changeset, :configuration_data, "cannot be empty")
      data when is_map(data) -> changeset
      _ -> add_error(changeset, :configuration_data, "must be a valid JSON object")
    end
  end

  @doc """
  Returns the list of device types available in the system.
  """
  def device_types do
    ["SR600 Mini", "P80", "A920", "Other"]
  end

  @doc """
  Returns the list of status options.
  """
  def status_options do
    [
      {"Active", "active"},
      {"Inactive", "inactive"},
      {"Draft", "draft"}
    ]
  end

  @doc """
  Returns default configuration template for a device type.
  """
  def default_configuration(device_type) do
    case device_type do
      "SR600 Mini" ->
        %{
          "network" => %{
            "wifi_enabled" => true,
            "ethernet_enabled" => true,
            "cellular_enabled" => false
          },
          "display" => %{
            "brightness" => 80,
            "timeout" => 30,
            "screensaver_enabled" => true
          },
          "hardware" => %{
            "printer_enabled" => true,
            "card_reader_enabled" => true,
            "nfc_enabled" => true
          }
        }

      "P80" ->
        %{
          "network" => %{
            "wifi_enabled" => true,
            "ethernet_enabled" => false,
            "cellular_enabled" => true
          },
          "display" => %{
            "brightness" => 75,
            "timeout" => 60,
            "screensaver_enabled" => true
          },
          "hardware" => %{
            "printer_enabled" => true,
            "card_reader_enabled" => true,
            "nfc_enabled" => false
          }
        }

      _ ->
        %{
          "network" => %{},
          "display" => %{},
          "hardware" => %{}
        }
    end
  end
end
