defmodule DaProductApp.TerminalManagement.FilterCacheService do
  @moduledoc """
  High-performance ETS cache service for terminal filter options.
  Handles 100K+ terminals with microsecond access times and usage-based sorting.
  """

  use GenServer
  require Logger
  import Ecto.Query
  alias DaProductApp.Repo
  alias DaProductApp.TerminalManagement.{TmsTerminal, FilterUsage}

  @table_name :terminal_filter_cache
  # 15 minutes
  @refresh_interval :timer.minutes(15)

  # Client API

  def start_link(_opts) do
    GenServer.start_link(__MODULE__, %{}, name: __MODULE__)
  end

  @doc "Get all filter options with usage-based sorting"
  def get_filter_options do
    case :ets.lookup(@table_name, :filter_options) do
      [{:filter_options, options}] ->
        options

      [] ->
        Logger.warn("Filter cache miss, rebuilding...")
        rebuild_cache()
        get_filter_options()
    end
  end

  @doc "Manually refresh the cache (for user-triggered refresh)"
  def refresh_cache do
    GenServer.call(__MODULE__, :refresh_cache, 30_000)
  end

  @doc "Track filter usage for intelligent sorting"
  def track_usage(filter_type, filter_value) when filter_type in [:area, :vendor, :model] do
    GenServer.cast(__MODULE__, {:track_usage, filter_type, filter_value})
  end

  @doc "Update cache when terminals change"
  def invalidate_cache do
    GenServer.cast(__MODULE__, :invalidate_cache)
  end

  # Server Implementation

  @impl true
  def init(_state) do
    # Create ETS table
    :ets.new(@table_name, [:set, :public, :named_table, read_concurrency: true])

    # Initial cache build
    send(self(), :build_initial_cache)

    # Schedule periodic refresh
    schedule_refresh()

    {:ok, %{last_refresh: DateTime.utc_now()}}
  end

  @impl true
  def handle_info(:build_initial_cache, state) do
    Logger.info("Building initial filter cache...")
    build_filter_cache()
    {:noreply, %{state | last_refresh: DateTime.utc_now()}}
  end

  @impl true
  def handle_info(:refresh_cache, state) do
    Logger.info("Performing scheduled filter cache refresh...")
    build_filter_cache()
    schedule_refresh()
    {:noreply, %{state | last_refresh: DateTime.utc_now()}}
  end

  @impl true
  def handle_call(:refresh_cache, _from, state) do
    Logger.info("Manual filter cache refresh requested...")
    result = build_filter_cache()
    {:reply, result, %{state | last_refresh: DateTime.utc_now()}}
  end

  @impl true
  def handle_cast({:track_usage, filter_type, filter_value}, state) do
    # Asynchronously track usage in database
    Task.start(fn ->
      track_filter_usage(filter_type, filter_value)
    end)

    {:noreply, state}
  end

  @impl true
  def handle_cast(:invalidate_cache, state) do
    # Rebuild cache after short delay to batch updates
    Process.send_after(self(), :refresh_cache, 1000)
    {:noreply, state}
  end

  # Private Functions

  defp schedule_refresh do
    Process.send_after(self(), :refresh_cache, @refresh_interval)
  end

  defp build_filter_cache do
    try do
      # Get raw filter data from terminals
      filter_data = get_terminal_filter_data()

      # Get usage statistics
      usage_stats = get_usage_statistics()

      # Build sorted options with usage frequency
      options = %{
        areas: build_sorted_options(filter_data.areas, usage_stats, :area),
        vendors: build_sorted_options(filter_data.vendors, usage_stats, :vendor),
        models: build_sorted_options(filter_data.models, usage_stats, :model)
      }

      # Store in ETS cache
      :ets.insert(@table_name, {:filter_options, options})

      # Safe logging with proper size calculation
      areas_count = length(options.areas)
      vendors_count = length(options.vendors)
      models_count = length(options.models)

      Logger.info(
        "Filter cache rebuilt: #{areas_count} areas, #{vendors_count} vendors, #{models_count} models"
      )

      {:ok, options}
    rescue
      error ->
        Logger.error("Failed to build filter cache: #{inspect(error)}")
        {:error, error}
    end
  end

  defp get_terminal_filter_data do
    # Single optimized query to get all distinct filter values
    query =
      from t in TmsTerminal,
        where: not is_nil(t.area) or not is_nil(t.vendor) or not is_nil(t.model),
        select: %{
          area: t.area,
          vendor: t.vendor,
          model: t.model
        }

    try do
      results = Repo.all(query)

      %{
        areas: extract_unique_values(results, :area),
        vendors: extract_unique_values(results, :vendor),
        models: extract_unique_values(results, :model)
      }
    rescue
      _error ->
        # Return empty sets if query fails
        %{
          areas: MapSet.new(),
          vendors: MapSet.new(),
          models: MapSet.new()
        }
    end
  end

  defp extract_unique_values(results, field) do
    results
    |> Enum.map(&Map.get(&1, field))
    |> Enum.filter(&(&1 && String.trim(&1) != ""))
    |> Enum.uniq()
    |> MapSet.new()
  end

  defp get_usage_statistics do
    # Get usage counts for each filter value
    query =
      from u in FilterUsage,
        group_by: [u.filter_type, u.filter_value],
        select: %{
          filter_type: u.filter_type,
          filter_value: u.filter_value,
          usage_count: coalesce(sum(u.usage_count), 0)
        }

    try do
      Repo.all(query)
      |> Enum.group_by(& &1.filter_type)
      |> Map.new(fn {type, values} ->
        {type, Map.new(values, &{&1.filter_value, &1.usage_count || 0})}
      end)
    rescue
      _error ->
        # Return empty map if query fails (e.g., table doesn't exist yet)
        %{}
    end
  end

  defp build_sorted_options(values, usage_stats, filter_type) do
    usage_map = Map.get(usage_stats, Atom.to_string(filter_type), %{})

    values
    |> Enum.to_list()
    |> Enum.map(fn value ->
      usage_count = Map.get(usage_map, value, 0)
      # Ensure usage_count is always an integer
      safe_usage_count = if is_integer(usage_count), do: usage_count, else: 0
      {value, safe_usage_count}
    end)
    |> Enum.sort_by(fn {value, usage_count} ->
      # Sort by usage count DESC, then alphabetically ASC
      {-usage_count, value}
    end)
    # Format for select options
    |> Enum.map(fn {value, _count} -> {value, value} end)
  end

  defp track_filter_usage(filter_type, filter_value) do
    attrs = %{
      filter_type: Atom.to_string(filter_type),
      filter_value: filter_value,
      usage_count: 1,
      last_used_at: DateTime.utc_now()
    }

    # Upsert usage record
    case Repo.get_by(FilterUsage,
           filter_type: attrs.filter_type,
           filter_value: attrs.filter_value
         ) do
      nil ->
        %FilterUsage{}
        |> FilterUsage.changeset(attrs)
        |> Repo.insert()

      existing ->
        updated_attrs = %{
          usage_count: existing.usage_count + 1,
          last_used_at: DateTime.utc_now()
        }

        existing
        |> FilterUsage.changeset(updated_attrs)
        |> Repo.update()
    end
  rescue
    error ->
      Logger.error("Failed to track filter usage: #{inspect(error)}")
  end

  defp rebuild_cache do
    build_filter_cache()
  end
end
