defmodule DaProductApp.TerminalManagement.FilterUsage do
  @moduledoc """
  Schema for tracking filter usage statistics to enable intelligent sorting.
  """

  use Ecto.Schema
  import Ecto.Changeset

  @derive {Jason.Encoder,
           only: [
             :id,
             :filter_type,
             :filter_value,
             :usage_count,
             :last_used_at,
             :inserted_at,
             :updated_at
           ]}

  schema "filter_usage" do
    # "area", "vendor", "model"
    field :filter_type, :string
    # actual filter value
    field :filter_value, :string
    field :usage_count, :integer, default: 0
    field :last_used_at, :utc_datetime

    timestamps()
  end

  @required_fields [:filter_type, :filter_value]
  @optional_fields [:usage_count, :last_used_at]
  @all_fields @required_fields ++ @optional_fields

  def changeset(filter_usage, attrs) do
    filter_usage
    |> cast(attrs, @all_fields)
    |> validate_required(@required_fields)
    |> validate_inclusion(:filter_type, ["area", "vendor", "model"])
    |> validate_length(:filter_value, min: 1, max: 255)
    |> validate_number(:usage_count, greater_than_or_equal_to: 0)
    |> unique_constraint([:filter_type, :filter_value],
      name: :filter_usage_type_value_index
    )
  end
end
