defmodule DaProductApp.TerminalManagement.GroupConfig do
  @moduledoc """
  Configuration module for Terminal Groups in production environments.

  This module provides production-ready configurations, field mappings,
  and validation rules for terminal grouping systems.
  """

  @doc """
  Get available fields for rule creation based on environment.
  In production, this could be loaded from database or configuration.
  """
  def available_fields do
    [
      %{key: "vendor", label: "Vendor", type: "string", description: "Terminal manufacturer"},
      %{key: "model", label: "Model", type: "string", description: "Terminal model number"},
      %{
        key: "status",
        label: "Status",
        type: "string",
        description: "Current operational status"
      },
      %{
        key: "area",
        label: "Area/Region",
        type: "string",
        description: "Geographic area or region"
      },
      %{
        key: "location_code",
        label: "Location Code",
        type: "string",
        description: "Specific location identifier"
      },
      %{
        key: "merchant_id",
        label: "Merchant ID",
        type: "string",
        description: "Associated merchant identifier"
      },
      %{
        key: "deployment_type",
        label: "Deployment Type",
        type: "string",
        description: "Environment type (production, staging, test)"
      },
      %{
        key: "tier",
        label: "Terminal Tier",
        type: "string",
        description: "Service tier or category"
      },
      %{
        key: "app_version",
        label: "App Version",
        type: "version",
        description: "Application version"
      },
      %{
        key: "system_version",
        label: "System Version",
        type: "version",
        description: "System software version"
      },
      %{
        key: "hardware_version",
        label: "Hardware Version",
        type: "version",
        description: "Hardware revision"
      }
    ]
  end

  @doc """
  Get available operators for rule creation.
  """
  def available_operators do
    [
      %{key: "equals", label: "Equals", description: "Exact match (case-sensitive)"},
      %{key: "not_equals", label: "Not Equals", description: "Must not equal this value"},
      %{key: "contains", label: "Contains", description: "Field must contain this text"},
      %{key: "starts_with", label: "Starts With", description: "Field must start with this text"},
      %{key: "ends_with", label: "Ends With", description: "Field must end with this text"},
      %{key: "in", label: "In List", description: "Value must be in comma-separated list"},
      %{
        key: "not_in",
        label: "Not In List",
        description: "Value must not be in comma-separated list"
      },
      %{key: "regex", label: "Regex Pattern", description: "Regular expression match"},
      %{key: "range", label: "Numeric Range", description: "Numeric range (min,max)"}
    ]
  end

  @doc """
  Get predefined rule templates for common production scenarios.
  """
  def rule_templates do
    [
      %{
        key: "vendor_ingenico",
        name: "Ingenico Terminals",
        description: "All Ingenico brand terminals",
        rule: %{
          rule_name: "Ingenico Terminals",
          rule_type: "field_match",
          field_name: "vendor",
          operator: "equals",
          value: "Ingenico",
          priority: 100
        }
      },
      %{
        key: "status_active",
        name: "Active Terminals",
        description: "Only active and operational terminals",
        rule: %{
          rule_name: "Active Terminals",
          rule_type: "field_match",
          field_name: "status",
          operator: "equals",
          value: "active",
          priority: 50
        }
      },
      %{
        key: "high_tier",
        name: "High Tier Terminals",
        description: "Premium and enterprise tier terminals",
        rule: %{
          rule_name: "High Tier Terminals",
          rule_type: "field_match",
          field_name: "tier",
          operator: "in",
          value: "premium,enterprise,high",
          priority: 75
        }
      },
      %{
        key: "production_deployment",
        name: "Production Only",
        description: "Only production environment terminals",
        rule: %{
          rule_name: "Production Deployment",
          rule_type: "field_match",
          field_name: "deployment_type",
          operator: "equals",
          value: "production",
          priority: 25
        }
      },
      %{
        key: "regional_north",
        name: "North Region",
        description: "All terminals in northern regions",
        rule: %{
          rule_name: "North Region Terminals",
          rule_type: "field_match",
          field_name: "area",
          operator: "starts_with",
          value: "North",
          priority: 60
        }
      },
      %{
        key: "maintenance_due",
        name: "Maintenance Due",
        description: "Terminals requiring maintenance",
        rule: %{
          rule_name: "Maintenance Due",
          rule_type: "field_match",
          field_name: "status",
          operator: "in",
          value: "maintenance_due,needs_update,offline",
          priority: 90
        }
      }
    ]
  end

  @doc """
  Get production-safe validation rules.
  """
  def validation_rules do
    %{
      rule_name: [
        {:required, "Rule name is required"},
        {:min_length, 3, "Rule name must be at least 3 characters"},
        {:max_length, 100, "Rule name cannot exceed 100 characters"}
      ],
      value: [
        {:required, "Rule value is required"},
        {:max_length, 500, "Rule value cannot exceed 500 characters"}
      ],
      priority: [
        {:type, :integer, "Priority must be a number"},
        {:range, 1..1000, "Priority must be between 1 and 1000"}
      ]
    }
  end

  @doc """
  Get system groups that should be created automatically in production.
  """
  def default_system_groups do
    [
      %{
        name: "All Terminals",
        description: "Default group containing all terminals",
        group_type: "system",
        color: "#6B7280",
        icon: "hero-squares-2x2",
        is_active: true
      },
      %{
        name: "Active Terminals",
        description: "Terminals currently active and operational",
        group_type: "auto",
        color: "#10B981",
        icon: "hero-check-circle",
        is_active: true,
        rules: [
          %{
            rule_name: "Active Status Rule",
            rule_type: "field_match",
            field_name: "status",
            operator: "equals",
            value: "active",
            priority: 1
          }
        ]
      },
      %{
        name: "Inactive Terminals",
        description: "Terminals that are offline or inactive",
        group_type: "auto",
        color: "#EF4444",
        icon: "hero-x-circle",
        is_active: true,
        rules: [
          %{
            rule_name: "Inactive Status Rule",
            rule_type: "field_match",
            field_name: "status",
            operator: "in",
            value: "inactive,offline,maintenance",
            priority: 1
          }
        ]
      },
      %{
        name: "Production Terminals",
        description: "Terminals deployed in production environment",
        group_type: "auto",
        color: "#3B82F6",
        icon: "hero-server",
        is_active: true,
        rules: [
          %{
            rule_name: "Production Environment Rule",
            rule_type: "field_match",
            field_name: "deployment_type",
            operator: "equals",
            value: "production",
            priority: 10
          }
        ]
      }
    ]
  end

  @doc """
  Performance and monitoring configuration for production.
  """
  def performance_config do
    %{
      # Maximum terminals to process in a single batch
      batch_size: 1000,

      # Timeout for rule application (in milliseconds)
      rule_application_timeout: 30_000,

      # Maximum concurrent rule evaluations
      max_concurrent_rules: 10,

      # Cache TTL for group statistics (in seconds)
      statistics_cache_ttl: 300,

      # Enable performance logging
      enable_performance_logging: true,

      # Log slow rule applications (threshold in milliseconds)
      slow_rule_threshold: 5_000
    }
  end
end
