defmodule DaProductApp.TerminalManagement.MerchantType do
  use Ecto.Schema
  import Ecto.Changeset
  import Ecto.Query

  @derive {Jason.Encoder,
           only: [
             :id,
             :name,
             :description,
             :code,
             :category,
             :is_active,
             :inserted_at,
             :updated_at
           ]}

  schema "merchant_types" do
    field :name, :string
    field :description, :string
    field :code, :string
    # retail, restaurant, healthcare, automotive, etc.
    field :category, :string
    field :is_active, :boolean, default: true

    # Associations
    has_many :merchants, DaProductApp.Merchants.Merchant, foreign_key: :merchant_type_id

    timestamps()
  end

  @required_fields [:name, :code, :category]
  @optional_fields [:description, :is_active]
  @all_fields @required_fields ++ @optional_fields

  @valid_categories [
    "retail",
    "restaurant",
    "healthcare",
    "automotive",
    "digital",
    "general",
    "corporate",
    "hospitality",
    "education",
    "government"
  ]

  def changeset(merchant_type, attrs) do
    merchant_type
    |> cast(attrs, @all_fields)
    |> validate_required(@required_fields)
    |> validate_length(:name, min: 2, max: 100)
    |> validate_length(:code, min: 2, max: 20)
    |> validate_format(:code, ~r/^[A-Z_]+$/,
      message: "must be uppercase letters and underscores only"
    )
    |> validate_inclusion(:category, @valid_categories)
    |> validate_length(:description, max: 500)
    |> unique_constraint(:name)
    |> unique_constraint(:code)
  end

  # Scope for active merchant types
  def active_types(query \\ __MODULE__) do
    from mt in query, where: mt.is_active == true
  end

  # Scope by category
  def by_category(query \\ __MODULE__, category) do
    from mt in query, where: mt.category == ^category
  end

  # Get all categories
  def valid_categories, do: @valid_categories
end
