defmodule DaProductApp.TerminalManagement.OfflineMonitor do
  @moduledoc """
  GenServer that periodically checks for offline devices and sends alerts.
  Runs every 5 minutes to check device heartbeat status.
  """

  use GenServer
  require Logger

  alias DaProductApp.TerminalManagement

  # 5 minutes in milliseconds
  @check_interval 5 * 60 * 1000

  def start_link(opts) do
    GenServer.start_link(__MODULE__, opts, name: __MODULE__)
  end

  @impl true
  def init(_opts) do
    # Schedule first check
    schedule_check()
    {:ok, %{}}
  end

  @impl true
  def handle_info(:check_offline_devices, state) do
    Logger.info("Running offline device check...")

    try do
      TerminalManagement.check_offline_devices()
      Logger.info("Offline device check completed successfully")
    rescue
      error ->
        Logger.error("Error during offline device check: #{inspect(error)}")
    end

    # Schedule next check
    schedule_check()
    {:noreply, state}
  end

  defp schedule_check do
    Process.send_after(self(), :check_offline_devices, @check_interval)
  end
end
