defmodule DaProductApp.TerminalManagement.OtaConfiguration do
  use Ecto.Schema
  import Ecto.Changeset

  schema "ota_configurations" do
    field :request_id, :integer
    field :merchant_config, :boolean, default: true
    field :merchant_id, :string
    field :terminal_id, :string
    field :device_serial, :string
    field :mqtt_ip, :string
    field :mqtt_port, :integer, default: 1883
    field :http_ip, :string
    field :http_port, :integer, default: 4001
    field :product_key, :string
    field :product_secret, :string
    field :client_id, :string
    field :username, :string
    field :mqtt_topic, :string
    field :keepalive_time, :integer, default: 300
    field :play_language, :integer, default: 1
    field :heartbeat_interval, :integer, default: 300
    field :status, :string, default: "pending"
    field :sent_at, :utc_datetime
    field :acknowledged_at, :utc_datetime
    field :error_message, :string
    timestamps()
  end

  def changeset(configuration, attrs) do
    configuration
    |> cast(attrs, [
      :request_id,
      :merchant_config,
      :merchant_id,
      :terminal_id,
      :device_serial,
      :mqtt_ip,
      :mqtt_port,
      :http_ip,
      :http_port,
      :product_key,
      :product_secret,
      :client_id,
      :username,
      :mqtt_topic,
      :keepalive_time,
      :play_language,
      :heartbeat_interval,
      :status,
      :sent_at,
      :acknowledged_at,
      :error_message
    ])
    |> validate_required([:device_serial, :merchant_id, :terminal_id])
    |> validate_inclusion(:status, ["pending", "sent", "acknowledged", "failed"])
    |> validate_number(:mqtt_port, greater_than: 0, less_than: 65536)
    |> validate_number(:http_port, greater_than: 0, less_than: 65536)
    |> validate_number(:heartbeat_interval, greater_than: 0)
    |> validate_number(:keepalive_time, greater_than: 0)
    |> unique_constraint([:device_serial, :request_id])
  end

  def to_mqtt_payload(%__MODULE__{} = config) do
    %{
      request_id: config.request_id,
      merchant_config: config.merchant_config,
      merchant_id: config.merchant_id,
      terminal_id: config.terminal_id,
      mqtt_ip: config.mqtt_ip,
      mqtt_port: config.mqtt_port,
      http_ip: config.http_ip,
      http_port: config.http_port,
      product_key: config.product_key,
      product_secret: config.product_secret,
      client_id: config.client_id,
      username: config.username,
      mqtt_topic: config.mqtt_topic,
      keepalive_time: config.keepalive_time,
      play_language: config.play_language,
      heartbeat_interval: config.heartbeat_interval
    }
  end
end
