defmodule DaProductApp.TerminalManagement.TerminalEventListener do
  @moduledoc """
  GenServer that listens to terminal events and ensures proper rule application.
  Provides real-time monitoring and automatic group management.
  """

  use GenServer
  require Logger
  alias Phoenix.PubSub

  @pubsub DaProductApp.PubSub
  @topic "terminal:events"

  # ============================================================================
  # Client API
  # ============================================================================

  def start_link(opts) do
    GenServer.start_link(__MODULE__, opts, name: __MODULE__)
  end

  def get_stats do
    GenServer.call(__MODULE__, :get_stats)
  end

  def reset_stats do
    GenServer.call(__MODULE__, :reset_stats)
  end

  # ============================================================================
  # Server Callbacks
  # ============================================================================

  @impl true
  def init(_opts) do
    # Subscribe to terminal events
    PubSub.subscribe(@pubsub, @topic)

    Logger.info("Terminal Event Listener started and listening to events")

    {:ok,
     %{
       events_processed: 0,
       rules_applied: 0,
       errors: 0,
       last_event: nil,
       started_at: DateTime.utc_now()
     }}
  end

  @impl true
  def handle_info(event, state) do
    Logger.debug("Processing terminal event: #{event.type}")

    try do
      process_event(event)

      new_state = %{
        state
        | events_processed: state.events_processed + 1,
          last_event: event,
          rules_applied: state.rules_applied + count_rules_applied(event)
      }

      # Broadcast update to LiveView for real-time UI updates
      PubSub.broadcast(@pubsub, "tms:groups", {:event_processed, event})

      {:noreply, new_state}
    rescue
      error ->
        Logger.error("Error processing terminal event: #{inspect(error)}")

        new_state = %{
          state
          | events_processed: state.events_processed + 1,
            errors: state.errors + 1,
            last_event: event
        }

        {:noreply, new_state}
    end
  end

  @impl true
  def handle_call(:get_stats, _from, state) do
    stats = Map.put(state, :uptime_minutes, calculate_uptime(state.started_at))
    {:reply, stats, state}
  end

  @impl true
  def handle_call(:reset_stats, _from, state) do
    new_state = %{
      state
      | events_processed: 0,
        rules_applied: 0,
        errors: 0,
        last_event: nil,
        started_at: DateTime.utc_now()
    }

    {:reply, :ok, new_state}
  end

  # ============================================================================
  # Event Processing
  # ============================================================================

  defp process_event(%{type: :terminal_created} = event) do
    Logger.info("Processing terminal_created event for terminal #{event.terminal_id}")
    # Events are already handled by TerminalEventDispatcher
    :ok
  end

  defp process_event(%{type: :terminal_updated} = event) do
    Logger.info("Processing terminal_updated event for terminal #{event.terminal_id}")
    Logger.info("Changed fields: #{inspect(event.changes)}")
    # Events are already handled by TerminalEventDispatcher
    :ok
  end

  defp process_event(%{type: :terminal_status_changed} = event) do
    Logger.info("Processing status change: #{event.old_status} -> #{event.new_status}")
    # Events are already handled by TerminalEventDispatcher
    :ok
  end

  defp process_event(%{type: :rule_created} = event) do
    Logger.info("Processing rule_created event for rule #{event.rule_id}")
    # Events are already handled by TerminalEventDispatcher
    :ok
  end

  defp process_event(%{type: :rule_updated} = event) do
    Logger.info("Processing rule_updated event for rule #{event.rule_id}")
    # Events are already handled by TerminalEventDispatcher
    :ok
  end

  defp process_event(%{type: :rule_deleted} = event) do
    Logger.info("Processing rule_deleted event for rule #{event.rule_id}")
    # Events are already handled by TerminalEventDispatcher
    :ok
  end

  defp process_event(event) do
    Logger.warn("Unknown event type: #{inspect(event.type)}")
    :ok
  end

  # ============================================================================
  # Helper Functions
  # ============================================================================

  defp count_rules_applied(%{type: :terminal_created}), do: 1
  defp count_rules_applied(%{type: :terminal_updated}), do: 1
  defp count_rules_applied(%{type: :terminal_status_changed}), do: 1
  defp count_rules_applied(%{type: :rule_created}), do: 1
  defp count_rules_applied(%{type: :rule_updated}), do: 1
  defp count_rules_applied(_), do: 0

  defp calculate_uptime(started_at) do
    (DateTime.diff(DateTime.utc_now(), started_at, :second) / 60)
    |> Float.round(2)
  end
end
