defmodule DaProductApp.Transactions.Transactions do
  import Ecto.Query, warn: false
  alias DaProductApp.Repo
  alias DaProductApp.Transactions.Transaction

  @doc """
  Returns the list of transactions.
  """
  def list_transactions do
    Repo.all(Transaction)
  end

  @doc """
  Gets a single transaction by ID.
  """
  def get_transaction!(id), do: Repo.get!(Transaction, id)

  @doc """
  Gets a transaction by m_ref_num.
  """
  def get_transaction_by_ref!(m_ref_num), do: Repo.get_by!(Transaction, m_ref_num: m_ref_num)

  @doc """
  Gets a transaction by processing_id.
  """
  def get_transaction_by_processing_id(nil), do: nil

  def get_transaction_by_processing_id(processing_id) when is_binary(processing_id) do
    from(t in Transaction, where: t.processing_id == ^processing_id)
    |> Repo.one()
  end

  @doc """
  Creates a transaction.

  ## Examples

      iex> create_transaction(%{field: value})
      {:ok, %Transaction{}}

      iex> create_transaction(%{field: bad_value})
      {:error, %Ecto.Changeset{}}

  """
  def create_transaction(attrs \\ %{}) do
    %Transaction{}
    |> Transaction.changeset(attrs)
    |> Repo.insert()
  end

  @doc """
  Updates a transaction.
  """
  def update_transaction(%Transaction{} = transaction, attrs) do
    transaction
    |> Transaction.changeset(attrs)
    |> Repo.update()
  end

  @doc """
  Deletes a transaction.
  """
  def delete_transaction(%Transaction{} = transaction) do
    Repo.delete(transaction)
  end

  @doc """
  Returns an `%Ecto.Changeset{}` for tracking transaction changes.
  """
  def change_transaction(%Transaction{} = transaction, attrs \\ %{}) do
    Transaction.changeset(transaction, attrs)
  end
end
