defmodule DaProductApp.Workers.EodFileGenerationWorker do
  @moduledoc """
  Oban worker for generating EOD settlement files in the background.

  This worker handles the background processing of EOD file generation,
  especially for large transaction volumes that would otherwise block
  the web/API request cycle.
  """

  use Oban.Worker, queue: :default, max_attempts: 3

  alias DaProductApp.Settlements.EodFileGenerator
  require Logger

  @impl Oban.Worker
  def perform(%Oban.Job{args: %{"date" => date_str, "sequence" => sequence, "force" => force}}) do
    Logger.info("Starting EOD file generation job for date: #{date_str}, sequence: #{sequence}")

    # Check if file already exists (idempotency)
    case Date.from_iso8601(date_str) do
      {:ok, date} ->
        filename = generate_filename(date, sequence)
        file_path = Path.join("/v1/settlements/settlement/eod/date", filename)

        if File.exists?(file_path) and not force do
          Logger.info("EOD file already exists: #{file_path}")

          {:ok,
           %{file_path: file_path, date: date_str, sequence: sequence, status: "already_exists"}}
        else
          case EodFileGenerator.generate_eod_file_sync(
                 date: date,
                 sequence: sequence,
                 force: force
               ) do
            {:ok, file_path} ->
              Logger.info("Successfully generated EOD file: #{file_path}")

              {:ok,
               %{file_path: file_path, date: date_str, sequence: sequence, status: "generated"}}

            {:error, reason} ->
              Logger.error("Failed to generate EOD file: #{reason}")
              {:error, reason}
          end
        end

      {:error, _} ->
        Logger.error("Invalid date format: #{date_str}")
        {:error, "Invalid date format"}
    end
  end

  # Private helper to generate filename (matching the one in EodFileGenerator)
  defp generate_filename(date, sequence) do
    mmdd = date |> Date.to_string() |> String.slice(5, 5) |> String.replace("-", "")

    hhmmss =
      DateTime.utc_now()
      |> DateTime.to_time()
      |> Time.to_string()
      |> String.slice(0, 8)
      |> String.replace(":", "")

    seq_str = sequence |> Integer.to_string() |> String.pad_leading(2, "0")

    "mercury_pay_#{mmdd}_#{hhmmss}_#{seq_str}.csv"
  end

  @impl Oban.Worker
  def perform(%Oban.Job{args: args}) do
    Logger.error("Invalid job arguments: #{inspect(args)}")
    {:error, "Invalid job arguments"}
  end
end
