defmodule DaProductApp.Workers.ParameterPushWorker do
  @moduledoc """
  Oban worker for handling scheduled parameter push jobs.
  """

  use Oban.Worker, queue: :parameter_push, max_attempts: 3

  require Logger
  alias DaProductApp.TerminalManagement.ParameterPushService

  @impl Oban.Worker
  def perform(%Oban.Job{
        args: %{
          "type" => "single_device",
          "device_serial" => device_serial,
          "parameters" => parameters
        }
      }) do
    Logger.info("Processing parameter push job for device: #{device_serial}")

    case ParameterPushService.send_terminal_parameters(device_serial, parameters) do
      {:ok, message} ->
        Logger.info("Parameter push successful for #{device_serial}: #{message}")
        :ok

      {:error, reason} ->
        Logger.error("Parameter push failed for #{device_serial}: #{reason}")
        {:error, reason}
    end
  end

  def perform(%Oban.Job{
        args: %{"type" => "batch", "device_serials" => device_serials, "parameters" => parameters}
      }) do
    Logger.info("Processing batch parameter push job for #{length(device_serials)} devices")

    case ParameterPushService.batch_send_parameters(device_serials, parameters) do
      {:ok, %{failures: 0} = result} ->
        Logger.info("Batch parameter push completed successfully: #{inspect(result)}")
        :ok

      {:ok, %{failures: failures} = result} when failures > 0 ->
        Logger.warning("Batch parameter push completed with failures: #{inspect(result)}")
        # Don't retry the whole batch, individual failures are logged
        :ok

      {:error, reason} ->
        Logger.error("Batch parameter push failed: #{reason}")
        {:error, reason}
    end
  end

  def perform(%Oban.Job{
        args: %{
          "type" => "merchant_config",
          "device_serial" => device_serial,
          "merchant_id" => merchant_id,
          "terminal_id" => terminal_id,
          "options" => options
        }
      }) do
    Logger.info("Processing merchant config push job for device: #{device_serial}")

    case ParameterPushService.send_quick_merchant_config(
           device_serial,
           merchant_id,
           terminal_id,
           options
         ) do
      {:ok, message} ->
        Logger.info("Merchant config push successful for #{device_serial}: #{message}")
        :ok

      {:error, reason} ->
        Logger.error("Merchant config push failed for #{device_serial}: #{reason}")
        {:error, reason}
    end
  end

  @doc """
  Schedules a parameter push job for a single device.
  """
  def schedule_parameter_push(device_serial, parameters, opts \\ []) do
    schedule_at = Keyword.get(opts, :schedule_at, DateTime.utc_now())

    %{
      type: "single_device",
      device_serial: device_serial,
      parameters: parameters
    }
    |> new(scheduled_at: schedule_at)
    |> Oban.insert()
  end

  @doc """
  Schedules a batch parameter push job for multiple devices.
  """
  def schedule_batch_parameter_push(device_serials, parameters, opts \\ []) do
    schedule_at = Keyword.get(opts, :schedule_at, DateTime.utc_now())

    %{
      type: "batch",
      device_serials: device_serials,
      parameters: parameters
    }
    |> new(scheduled_at: schedule_at)
    |> Oban.insert()
  end

  @doc """
  Schedules a quick merchant configuration push.
  """
  def schedule_merchant_config_push(
        device_serial,
        merchant_id,
        terminal_id,
        options \\ %{},
        opts \\ []
      ) do
    schedule_at = Keyword.get(opts, :schedule_at, DateTime.utc_now())

    %{
      type: "merchant_config",
      device_serial: device_serial,
      merchant_id: merchant_id,
      terminal_id: terminal_id,
      options: options
    }
    |> new(scheduled_at: schedule_at)
    |> Oban.insert()
  end
end
