defmodule DaProductAppWeb.FileDownloadController do
  use DaProductAppWeb, :controller

  require Logger
  alias DaProductApp.TerminalManagement.FileDownloadService
  alias DaProductApp.Workers.FileDownloadWorker

  @doc """
  Send file download command to a single device.
  """
  def send_download_command(conn, %{"device_serial" => device_serial} = params) do
    download_params = params["download_params"] || %{}

    case FileDownloadService.send_file_download_command(device_serial, download_params) do
      {:ok, message} ->
        json(conn, %{
          success: true,
          message: message,
          device_serial: device_serial,
          request_id: download_params["request_id"]
        })

      {:error, reason} ->
        conn
        |> put_status(:bad_request)
        |> json(%{
          success: false,
          error: reason,
          device_serial: device_serial
        })
    end
  end

  @doc """
  Schedule file download command for later execution.
  """
  def schedule_download_command(conn, %{"device_serial" => device_serial} = params) do
    download_params = params["download_params"] || %{}
    schedule_at = parse_schedule_time(params["schedule_at"])

    case FileDownloadWorker.schedule_file_download(device_serial, download_params,
           schedule_at: schedule_at
         ) do
      {:ok, job} ->
        json(conn, %{
          success: true,
          message: "File download job scheduled successfully",
          job_id: job.id,
          device_serial: device_serial,
          scheduled_at: schedule_at
        })

      {:error, changeset} ->
        conn
        |> put_status(:bad_request)
        |> json(%{
          success: false,
          error: "Failed to schedule download job",
          details: changeset
        })
    end
  end

  @doc """
  Send file download command to multiple devices.
  """
  def batch_send_download_command(conn, %{"device_serials" => device_serials} = params) do
    download_params = params["download_params"] || %{}
    schedule_at = parse_schedule_time(params["schedule_at"])

    case FileDownloadWorker.schedule_batch_file_download(device_serials, download_params,
           schedule_at: schedule_at
         ) do
      {:ok, job} ->
        json(conn, %{
          success: true,
          message: "Batch file download job scheduled successfully",
          job_id: job.id,
          device_count: length(device_serials),
          scheduled_at: schedule_at
        })

      {:error, changeset} ->
        conn
        |> put_status(:bad_request)
        |> json(%{
          success: false,
          error: "Failed to schedule batch download job",
          details: changeset
        })
    end
  end

  @doc """
  Send logo download command to a device.
  """
  def send_logo_download(
        conn,
        %{"device_serial" => device_serial, "logo_url" => logo_url} = params
      ) do
    file_name = params["file_name"] || "merchant_logo.png"

    case FileDownloadService.create_logo_download_command(device_serial, logo_url, file_name) do
      {:ok, message} ->
        json(conn, %{
          success: true,
          message: message,
          device_serial: device_serial,
          logo_url: logo_url,
          file_name: file_name
        })

      {:error, reason} ->
        conn
        |> put_status(:bad_request)
        |> json(%{
          success: false,
          error: reason,
          device_serial: device_serial
        })
    end
  end

  @doc """
  Schedule logo download for later execution.
  """
  def schedule_logo_download(
        conn,
        %{"device_serial" => device_serial, "logo_url" => logo_url} = params
      ) do
    file_name = params["file_name"] || "merchant_logo.png"
    schedule_at = parse_schedule_time(params["schedule_at"])

    case FileDownloadWorker.schedule_logo_download(device_serial, logo_url, file_name,
           schedule_at: schedule_at
         ) do
      {:ok, job} ->
        json(conn, %{
          success: true,
          message: "Logo download job scheduled successfully",
          job_id: job.id,
          device_serial: device_serial,
          scheduled_at: schedule_at
        })

      {:error, changeset} ->
        conn
        |> put_status(:bad_request)
        |> json(%{
          success: false,
          error: "Failed to schedule logo download job",
          details: changeset
        })
    end
  end

  @doc """
  Send firmware download command to a device.
  """
  def send_firmware_download(conn, %{
        "device_serial" => device_serial,
        "firmware_url" => firmware_url,
        "file_name" => file_name
      }) do
    case FileDownloadService.create_firmware_download_command(
           device_serial,
           firmware_url,
           file_name
         ) do
      {:ok, message} ->
        json(conn, %{
          success: true,
          message: message,
          device_serial: device_serial,
          firmware_url: firmware_url,
          file_name: file_name
        })

      {:error, reason} ->
        conn
        |> put_status(:bad_request)
        |> json(%{
          success: false,
          error: reason,
          device_serial: device_serial
        })
    end
  end

  @doc """
  Send application download command to a device.
  """
  def send_application_download(conn, %{
        "device_serial" => device_serial,
        "app_url" => app_url,
        "file_name" => file_name
      }) do
    case FileDownloadService.create_application_download_command(
           device_serial,
           app_url,
           file_name
         ) do
      {:ok, message} ->
        json(conn, %{
          success: true,
          message: message,
          device_serial: device_serial,
          app_url: app_url,
          file_name: file_name
        })

      {:error, reason} ->
        conn
        |> put_status(:bad_request)
        |> json(%{
          success: false,
          error: reason,
          device_serial: device_serial
        })
    end
  end

  defp parse_schedule_time(nil), do: DateTime.utc_now()

  defp parse_schedule_time(schedule_str) when is_binary(schedule_str) do
    case DateTime.from_iso8601(schedule_str) do
      {:ok, datetime, _} -> datetime
      {:error, _} -> DateTime.utc_now()
    end
  end

  defp parse_schedule_time(%DateTime{} = datetime), do: datetime
  defp parse_schedule_time(_), do: DateTime.utc_now()
end
