defmodule DaProductAppWeb.ParameterController do
  use DaProductAppWeb, :controller

  alias DaProductApp.{ParameterManagement, TerminalManagement}

  action_fallback DaProductAppWeb.FallbackController

  def index(conn, params) do
    filters = Map.take(params, ["category_id", "vendor", "is_active"])
    parameters = ParameterManagement.list_parameter_definitions(filters)
    render(conn, :index, parameters: parameters)
  end

  def show(conn, %{"id" => id}) do
    parameter = ParameterManagement.get_parameter_definition!(id)
    render(conn, :show, parameter: parameter)
  end

  def create(conn, %{"parameter" => parameter_params}) do
    with {:ok, parameter} <- ParameterManagement.create_parameter_definition(parameter_params) do
      conn
      |> put_status(:created)
      |> put_resp_header("location", ~p"/api/parameters/#{parameter.id}")
      |> render(:show, parameter: parameter)
    end
  end

  def update(conn, %{"id" => id, "parameter" => parameter_params}) do
    parameter = ParameterManagement.get_parameter_definition!(id)

    with {:ok, parameter} <-
           ParameterManagement.update_parameter_definition(parameter, parameter_params) do
      render(conn, :show, parameter: parameter)
    end
  end

  def delete(conn, %{"id" => id}) do
    parameter = ParameterManagement.get_parameter_definition!(id)

    with {:ok, _parameter} <- ParameterManagement.delete_parameter_definition(parameter) do
      send_resp(conn, :no_content, "")
    end
  end

  # Template endpoints
  def list_templates(conn, params) do
    filters = Map.take(params, ["vendor", "model", "is_active"])
    templates = ParameterManagement.list_parameter_templates(filters)
    render(conn, :templates_index, templates: templates)
  end

  def show_template(conn, %{"id" => id}) do
    template = ParameterManagement.get_parameter_template!(id)
    render(conn, :template_show, template: template)
  end

  def create_template(conn, %{"template" => template_params}) do
    with {:ok, template} <- ParameterManagement.create_parameter_template(template_params) do
      conn
      |> put_status(:created)
      |> put_resp_header("location", ~p"/api/parameter_templates/#{template.id}")
      |> render(:template_show, template: template)
    end
  end

  def update_template(conn, %{"id" => id, "template" => template_params}) do
    template = ParameterManagement.get_parameter_template!(id)

    with {:ok, template} <-
           ParameterManagement.update_parameter_template(template, template_params) do
      render(conn, :template_show, template: template)
    end
  end

  def delete_template(conn, %{"id" => id}) do
    template = ParameterManagement.get_parameter_template!(id)

    with {:ok, _template} <- ParameterManagement.delete_parameter_template(template) do
      send_resp(conn, :no_content, "")
    end
  end

  # Terminal parameter operations
  def get_terminal_parameters(conn, %{"serial_number" => serial_number}) do
    case TerminalManagement.get_terminal_parameters(serial_number) do
      {:ok, parameters} ->
        render(conn, :terminal_parameters, parameters: parameters)

      {:error, :not_found} ->
        conn
        |> put_status(:not_found)
        |> json(%{error: "Terminal not found"})
    end
  end

  def push_parameters_to_terminal(conn, %{
        "serial_number" => serial_number,
        "parameters" => parameters
      }) do
    opts = Map.take(conn.params, ["push_type", "user_id"])

    case TerminalManagement.push_terminal_parameters(serial_number, parameters, opts) do
      {:ok, push_log} ->
        conn
        |> put_status(:accepted)
        |> render(:push_response, push_log: push_log)

      {:error, reason} ->
        conn
        |> put_status(:unprocessable_entity)
        |> json(%{error: reason})
    end
  end

  def apply_template_to_terminal(conn, %{
        "serial_number" => serial_number,
        "template_id" => template_id
      }) do
    opts = Map.take(conn.params, ["user_id", "override_existing"])

    case TerminalManagement.apply_parameter_template(serial_number, template_id, opts) do
      {:ok, push_log} ->
        conn
        |> put_status(:accepted)
        |> render(:push_response, push_log: push_log)

      {:error, reason} ->
        conn
        |> put_status(:unprocessable_entity)
        |> json(%{error: reason})
    end
  end

  def set_terminal_parameter(conn, %{
        "serial_number" => serial_number,
        "parameter_key" => parameter_key,
        "value" => value
      }) do
    opts = Map.take(conn.params, ["user_id", "source"])

    case TerminalManagement.set_terminal_parameter(serial_number, parameter_key, value, opts) do
      {:ok, push_log} ->
        conn
        |> put_status(:accepted)
        |> render(:push_response, push_log: push_log)

      {:error, reason} ->
        conn
        |> put_status(:unprocessable_entity)
        |> json(%{error: reason})
    end
  end

  def get_push_logs(conn, %{"serial_number" => serial_number}) do
    logs = ParameterManagement.list_parameter_push_logs(serial_number)
    render(conn, :push_logs, logs: logs)
  end

  def get_push_log(conn, %{"id" => id}) do
    push_log = ParameterManagement.get_parameter_push_log!(id)
    render(conn, :push_log_detail, push_log: push_log)
  end
end
