defmodule DaProductAppWeb.ParameterJSON do
  alias DaProductApp.ParameterManagement.{
    ParameterDefinition,
    ParameterTemplate,
    ParameterPushLog
  }

  @doc """
  Renders a list of parameters.
  """
  def index(%{parameters: parameters}) do
    %{data: for(parameter <- parameters, do: data(parameter))}
  end

  @doc """
  Renders a single parameter.
  """
  def show(%{parameter: parameter}) do
    %{data: data(parameter)}
  end

  @doc """
  Renders a list of templates.
  """
  def templates_index(%{templates: templates}) do
    %{data: for(template <- templates, do: template_data(template))}
  end

  @doc """
  Renders a single template.
  """
  def template_show(%{template: template}) do
    %{data: template_data(template)}
  end

  @doc """
  Renders terminal parameters.
  """
  def terminal_parameters(%{parameters: parameters}) do
    %{data: parameters}
  end

  @doc """
  Renders push response.
  """
  def push_response(%{push_log: push_log}) do
    %{
      data: %{
        push_log_id: push_log.id,
        request_id: push_log.request_id,
        status: push_log.status,
        sent_at: push_log.sent_at,
        message: "Parameters push initiated successfully"
      }
    }
  end

  @doc """
  Renders push logs.
  """
  def push_logs(%{logs: logs}) do
    %{data: for(log <- logs, do: push_log_data(log))}
  end

  @doc """
  Renders push log detail.
  """
  def push_log_detail(%{push_log: push_log}) do
    %{data: push_log_detail_data(push_log)}
  end

  defp data(%ParameterDefinition{} = parameter) do
    %{
      id: parameter.id,
      key: parameter.key,
      name: parameter.name,
      description: parameter.description,
      data_type: parameter.data_type,
      default_value: parameter.default_value,
      is_required: parameter.is_required,
      is_system: parameter.is_system,
      is_active: parameter.is_active,
      display_order: parameter.display_order,
      validation_rules: parameter.validation_rules,
      category_id: parameter.category_id,
      inserted_at: parameter.inserted_at,
      updated_at: parameter.updated_at
    }
  end

  defp template_data(%ParameterTemplate{} = template) do
    %{
      id: template.id,
      name: template.name,
      description: template.description,
      vendor: template.vendor,
      model: template.model,
      is_default: template.is_default,
      is_active: template.is_active,
      created_by_id: template.created_by_id,
      inserted_at: template.inserted_at,
      updated_at: template.updated_at
    }
  end

  defp push_log_data(%ParameterPushLog{} = log) do
    %{
      id: log.id,
      terminal_id: log.terminal_id,
      template_id: log.template_id,
      request_id: log.request_id,
      status: log.status,
      push_type: log.push_type,
      sent_at: log.sent_at,
      acknowledged_at: log.acknowledged_at,
      error_message: log.error_message,
      inserted_at: log.inserted_at
    }
  end

  defp push_log_detail_data(%ParameterPushLog{} = log) do
    %{
      id: log.id,
      terminal_id: log.terminal_id,
      template_id: log.template_id,
      request_id: log.request_id,
      status: log.status,
      push_type: log.push_type,
      parameters_sent: log.parameters_sent,
      sent_at: log.sent_at,
      acknowledged_at: log.acknowledged_at,
      error_message: log.error_message,
      triggered_by_id: log.triggered_by_id,
      push_details: log.push_details || [],
      inserted_at: log.inserted_at,
      updated_at: log.updated_at
    }
  end
end
