defmodule DaProductAppWeb.ParameterPushController do
  use DaProductAppWeb, :controller

  require Logger
  alias DaProductApp.TerminalManagement.ParameterPushService
  alias DaProductApp.Workers.ParameterPushWorker

  @doc """
  Immediately push parameters to a single device.
  """
  def push_parameters(conn, %{"device_serial" => device_serial} = params) do
    parameters = params["parameters"] || %{}

    case ParameterPushService.send_terminal_parameters(device_serial, parameters) do
      {:ok, message} ->
        json(conn, %{
          success: true,
          message: message,
          device_serial: device_serial
        })

      {:error, reason} ->
        conn
        |> put_status(:bad_request)
        |> json(%{
          success: false,
          error: reason,
          device_serial: device_serial
        })
    end
  end

  @doc """
  Schedule parameter push for later execution.
  """
  def schedule_parameter_push(conn, %{"device_serial" => device_serial} = params) do
    parameters = params["parameters"] || %{}
    schedule_at = parse_schedule_time(params["schedule_at"])

    case ParameterPushWorker.schedule_parameter_push(device_serial, parameters,
           schedule_at: schedule_at
         ) do
      {:ok, job} ->
        json(conn, %{
          success: true,
          message: "Parameter push job scheduled successfully",
          job_id: job.id,
          device_serial: device_serial,
          scheduled_at: schedule_at
        })

      {:error, changeset} ->
        conn
        |> put_status(:bad_request)
        |> json(%{
          success: false,
          error: "Failed to schedule job",
          details: changeset
        })
    end
  end

  @doc """
  Push parameters to multiple devices at once.
  """
  def batch_push_parameters(conn, %{"device_serials" => device_serials} = params) do
    parameters = params["parameters"] || %{}

    case ParameterPushService.batch_send_parameters(device_serials, parameters) do
      {:ok, result} ->
        json(conn, %{
          success: true,
          message: "Batch parameter push completed",
          result: result
        })

      {:error, reason} ->
        conn
        |> put_status(:bad_request)
        |> json(%{
          success: false,
          error: reason
        })
    end
  end

  @doc """
  Schedule batch parameter push for later execution.
  """
  def schedule_batch_parameter_push(conn, %{"device_serials" => device_serials} = params) do
    parameters = params["parameters"] || %{}
    schedule_at = parse_schedule_time(params["schedule_at"])

    case ParameterPushWorker.schedule_batch_parameter_push(device_serials, parameters,
           schedule_at: schedule_at
         ) do
      {:ok, job} ->
        json(conn, %{
          success: true,
          message: "Batch parameter push job scheduled successfully",
          job_id: job.id,
          device_count: length(device_serials),
          scheduled_at: schedule_at
        })

      {:error, changeset} ->
        conn
        |> put_status(:bad_request)
        |> json(%{
          success: false,
          error: "Failed to schedule batch job",
          details: changeset
        })
    end
  end

  @doc """
  Push quick merchant configuration to a device.
  """
  def push_merchant_config(
        conn,
        %{
          "device_serial" => device_serial,
          "merchant_id" => merchant_id,
          "terminal_id" => terminal_id
        } = params
      ) do
    options = params["options"] || %{}

    case ParameterPushService.send_quick_merchant_config(
           device_serial,
           merchant_id,
           terminal_id,
           options
         ) do
      {:ok, message} ->
        json(conn, %{
          success: true,
          message: message,
          device_serial: device_serial,
          merchant_id: merchant_id,
          terminal_id: terminal_id
        })

      {:error, reason} ->
        conn
        |> put_status(:bad_request)
        |> json(%{
          success: false,
          error: reason,
          device_serial: device_serial
        })
    end
  end

  @doc """
  Schedule merchant configuration push for later execution.
  """
  def schedule_merchant_config_push(
        conn,
        %{
          "device_serial" => device_serial,
          "merchant_id" => merchant_id,
          "terminal_id" => terminal_id
        } = params
      ) do
    options = params["options"] || %{}
    schedule_at = parse_schedule_time(params["schedule_at"])

    case ParameterPushWorker.schedule_merchant_config_push(
           device_serial,
           merchant_id,
           terminal_id,
           options,
           schedule_at: schedule_at
         ) do
      {:ok, job} ->
        json(conn, %{
          success: true,
          message: "Merchant config push job scheduled successfully",
          job_id: job.id,
          device_serial: device_serial,
          scheduled_at: schedule_at
        })

      {:error, changeset} ->
        conn
        |> put_status(:bad_request)
        |> json(%{
          success: false,
          error: "Failed to schedule merchant config job",
          details: changeset
        })
    end
  end

  defp parse_schedule_time(nil), do: DateTime.utc_now()

  defp parse_schedule_time(schedule_str) when is_binary(schedule_str) do
    case DateTime.from_iso8601(schedule_str) do
      {:ok, datetime, _} -> datetime
      {:error, _} -> DateTime.utc_now()
    end
  end

  defp parse_schedule_time(%DateTime{} = datetime), do: datetime
  defp parse_schedule_time(_), do: DateTime.utc_now()
end
