defmodule DaProductAppWeb.AlertsLive do
  use DaProductAppWeb, :live_view
  alias DaProductApp.TerminalManagement

  @impl true
  def mount(_params, _session, socket) do
    # Subscribe to real-time alerts
    if connected?(socket) do
      TerminalManagement.subscribe_to_device_updates()
      # Check for alerts every 60 seconds
      Process.send_after(self(), :check_alerts, 60_000)
    end

    socket =
      socket
      |> assign(:page_title, "Alert & Notification Management")
      |> assign(:current_tab, "dashboard")
      |> assign(:loading, true)
      |> assign(:current_page, "alerts")
      |> load_alert_data()

    {:ok, socket}
  end

  @impl true
  def handle_params(params, _url, socket) do
    tab = params["tab"] || "dashboard"

    socket =
      socket
      |> assign(:current_tab, tab)
      |> load_alert_data()

    {:noreply, socket}
  end

  @impl true
  def handle_event("change_tab", %{"tab" => tab}, socket) do
    {:noreply, push_patch(socket, to: ~p"/alerts?tab=#{tab}")}
  end

  @impl true
  def handle_event("acknowledge_alert", %{"alert_id" => alert_id}, socket) do
    user_id = socket.assigns[:current_user][:id] || "system"

    case TerminalManagement.acknowledge_alert(
           String.to_integer(alert_id),
           user_id,
           "Acknowledged via dashboard"
         ) do
      {:ok, _updated_alert} ->
        socket =
          socket
          |> put_flash(:info, "Alert acknowledged successfully")
          |> load_alert_data()

        {:noreply, socket}

      {:error, reason} ->
        socket = put_flash(socket, :error, "Failed to acknowledge alert: #{reason}")
        {:noreply, socket}
    end
  end

  @impl true
  def handle_event("resolve_alert", %{"alert_id" => alert_id}, socket) do
    user_id = socket.assigns[:current_user][:id] || "system"

    case TerminalManagement.resolve_alert(
           String.to_integer(alert_id),
           user_id,
           "Resolved via dashboard"
         ) do
      {:ok, _updated_alert} ->
        socket =
          socket
          |> put_flash(:info, "Alert resolved successfully")
          |> load_alert_data()

        {:noreply, socket}

      {:error, reason} ->
        socket = put_flash(socket, :error, "Failed to resolve alert: #{reason}")
        {:noreply, socket}
    end
  end

  @impl true
  def handle_event("create_alert_rule", params, socket) do
    rule_config = %{
      name: params["name"],
      description: params["description"],
      alert_type: params["alert_type"],
      severity: params["severity"],
      conditions: %{},
      notification_channels: String.split(params["notification_channels"], ","),
      enabled: params["enabled"] == "true"
    }

    case TerminalManagement.create_alert_rule(rule_config) do
      {:ok, _rule} ->
        socket =
          socket
          |> put_flash(:info, "Alert rule created successfully")
          |> load_alert_data()

        {:noreply, socket}

      {:error, reason} ->
        socket = put_flash(socket, :error, "Failed to create alert rule: #{reason}")
        {:noreply, socket}
    end
  end

  @impl true
  def handle_event("test_notification", %{"channel" => channel}, socket) do
    test_alert = %{
      id: "test",
      type: "test",
      severity: "medium",
      title: "Test Notification",
      message: "This is a test notification from the alert system",
      created_at: DateTime.utc_now()
    }

    case TerminalManagement.send_notification(test_alert, [channel]) do
      %{successful: 1} ->
        socket = put_flash(socket, :info, "Test notification sent successfully via #{channel}")
        {:noreply, socket}

      _ ->
        socket = put_flash(socket, :error, "Failed to send test notification via #{channel}")
        {:noreply, socket}
    end
  end

  @impl true
  def handle_event("refresh_alerts", _params, socket) do
    socket =
      socket
      |> assign(:loading, true)
      |> load_alert_data()

    {:noreply, socket}
  end

  @impl true
  def handle_info(:check_alerts, socket) do
    # Process real-time alerts
    alert_results = TerminalManagement.process_realtime_alerts()

    # Broadcast new alerts if any
    if alert_results.new_alerts > 0 do
      send(self(), {:new_alerts, alert_results.new_alerts})
    end

    # Schedule next check
    Process.send_after(self(), :check_alerts, 60_000)

    socket = load_alert_data(socket)
    {:noreply, socket}
  end

  @impl true
  def handle_info({:new_alerts, count}, socket) do
    socket = put_flash(socket, :info, "#{count} new alerts detected")
    {:noreply, socket}
  end

  @impl true
  def handle_info({:device_update, _device_data}, socket) do
    # Refresh alerts when device data changes
    socket = load_alert_data(socket)
    {:noreply, socket}
  end

  @impl true
  def handle_info({:monitoring_update, _monitoring_data}, socket) do
    # Refresh alerts when monitoring data changes
    socket = load_alert_data(socket)
    {:noreply, socket}
  end

  defp load_alert_data(socket) do
    # Load alert dashboard data
    alert_stats = TerminalManagement.get_alert_dashboard_stats()
    alert_config = TerminalManagement.get_alert_configuration()
    current_alerts = TerminalManagement.process_realtime_alerts()

    socket
    |> assign(:alert_stats, alert_stats)
    |> assign(:alert_config, alert_config)
    |> assign(:current_alerts, current_alerts)
    |> assign(:loading, false)
  end

  @impl true
  def render(assigns) do
    ~H"""
    <div class="min-h-screen bg-gradient-to-br from-slate-50 to-blue-50">
      <!-- Header Section -->
      <div class="bg-white shadow-sm border-b">
        <div class="px-6 py-6">
          <div class="flex justify-between items-center">
            <div>
              <h1 class="text-3xl font-bold text-gray-900">🔔 Alert & Notification Management</h1>
              <p class="text-gray-600 mt-2">
                Real-time alerts, notifications, and escalation management
              </p>
            </div>
            
    <!-- Action Buttons -->
            <div class="flex items-center space-x-4">
              <button
                phx-click="refresh_alerts"
                class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors flex items-center"
              >
                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path
                    stroke-linecap="round"
                    stroke-linejoin="round"
                    stroke-width="2"
                    d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"
                  />
                </svg>
                Refresh
              </button>

              <div class="bg-green-100 text-green-800 px-3 py-2 rounded-lg">
                <span class="text-sm font-medium">
                  Last Check: {Calendar.strftime(DateTime.utc_now(), "%H:%M:%S")}
                </span>
              </div>
            </div>
          </div>
          
    <!-- Loading Indicator -->
          <%= if @loading do %>
            <div class="mt-4 flex items-center text-blue-600">
              <svg
                class="animate-spin -ml-1 mr-3 h-5 w-5 text-blue-600"
                fill="none"
                viewBox="0 0 24 24"
              >
                <circle
                  class="opacity-25"
                  cx="12"
                  cy="12"
                  r="10"
                  stroke="currentColor"
                  stroke-width="4"
                >
                </circle>
                <path
                  class="opacity-75"
                  fill="currentColor"
                  d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                >
                </path>
              </svg>
              Loading alert data...
            </div>
          <% end %>
        </div>
        
    <!-- Tab Navigation -->
        <div class="border-t border-gray-200">
          <nav class="px-6 flex space-x-8">
            <button
              phx-click="change_tab"
              phx-value-tab="dashboard"
              class={"py-4 border-b-2 font-medium text-sm transition-colors " <> 
                     if(@current_tab == "dashboard", do: "border-blue-500 text-blue-600", else: "border-transparent text-gray-500 hover:text-gray-700")}
            >
              📊 Dashboard
            </button>
            <button
              phx-click="change_tab"
              phx-value-tab="active_alerts"
              class={"py-4 border-b-2 font-medium text-sm transition-colors " <> 
                     if(@current_tab == "active_alerts", do: "border-blue-500 text-blue-600", else: "border-transparent text-gray-500 hover:text-gray-700")}
            >
              🚨 Active Alerts
            </button>
            <button
              phx-click="change_tab"
              phx-value-tab="rules"
              class={"py-4 border-b-2 font-medium text-sm transition-colors " <> 
                     if(@current_tab == "rules", do: "border-blue-500 text-blue-600", else: "border-transparent text-gray-500 hover:text-gray-700")}
            >
              ⚙️ Alert Rules
            </button>
            <button
              phx-click="change_tab"
              phx-value-tab="notifications"
              class={"py-4 border-b-2 font-medium text-sm transition-colors " <> 
                     if(@current_tab == "notifications", do: "border-blue-500 text-blue-600", else: "border-transparent text-gray-500 hover:text-gray-700")}
            >
              📤 Notifications
            </button>
            <button
              phx-click="change_tab"
              phx-value-tab="escalation"
              class={"py-4 border-b-2 font-medium text-sm transition-colors " <> 
                     if(@current_tab == "escalation", do: "border-blue-500 text-blue-600", else: "border-transparent text-gray-500 hover:text-gray-700")}
            >
              📈 Escalation
            </button>
          </nav>
        </div>
      </div>
      
    <!-- Main Content -->
      <div :if={!@loading} class="p-6">
        <%= case @current_tab do %>
          <% "dashboard" -> %>
            {render_dashboard_tab(assigns)}
          <% "active_alerts" -> %>
            {render_active_alerts_tab(assigns)}
          <% "rules" -> %>
            {render_rules_tab(assigns)}
          <% "notifications" -> %>
            {render_notifications_tab(assigns)}
          <% "escalation" -> %>
            {render_escalation_tab(assigns)}
          <% _ -> %>
            {render_dashboard_tab(assigns)}
        <% end %>
      </div>
    </div>
    """
  end

  defp render_dashboard_tab(assigns) do
    ~H"""
    <div class="space-y-6">
      <!-- Alert Summary Cards -->
      <div class="grid grid-cols-1 md:grid-cols-5 gap-6">
        <div class="bg-white rounded-xl shadow-sm p-6 border-l-4 border-red-500">
          <div class="flex items-center justify-between">
            <div>
              <p class="text-sm font-medium text-gray-600">Critical Alerts</p>
              <p class="text-3xl font-bold text-red-600">{@alert_stats.active_alerts.critical}</p>
            </div>
            <div class="bg-red-100 rounded-full p-3">
              <svg class="w-6 h-6 text-red-600" fill="currentColor" viewBox="0 0 20 20">
                <path
                  fill-rule="evenodd"
                  d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z"
                  clip-rule="evenodd"
                />
              </svg>
            </div>
          </div>
        </div>

        <div class="bg-white rounded-xl shadow-sm p-6 border-l-4 border-orange-500">
          <div class="flex items-center justify-between">
            <div>
              <p class="text-sm font-medium text-gray-600">High Priority</p>
              <p class="text-3xl font-bold text-orange-600">{@alert_stats.active_alerts.high}</p>
            </div>
            <div class="bg-orange-100 rounded-full p-3">
              <svg class="w-6 h-6 text-orange-600" fill="currentColor" viewBox="0 0 20 20">
                <path
                  fill-rule="evenodd"
                  d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z"
                  clip-rule="evenodd"
                />
              </svg>
            </div>
          </div>
        </div>

        <div class="bg-white rounded-xl shadow-sm p-6 border-l-4 border-yellow-500">
          <div class="flex items-center justify-between">
            <div>
              <p class="text-sm font-medium text-gray-600">Medium Priority</p>
              <p class="text-3xl font-bold text-yellow-600">{@alert_stats.active_alerts.medium}</p>
            </div>
            <div class="bg-yellow-100 rounded-full p-3">
              <svg class="w-6 h-6 text-yellow-600" fill="currentColor" viewBox="0 0 20 20">
                <path
                  fill-rule="evenodd"
                  d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z"
                  clip-rule="evenodd"
                />
              </svg>
            </div>
          </div>
        </div>

        <div class="bg-white rounded-xl shadow-sm p-6 border-l-4 border-blue-500">
          <div class="flex items-center justify-between">
            <div>
              <p class="text-sm font-medium text-gray-600">Low Priority</p>
              <p class="text-3xl font-bold text-blue-600">{@alert_stats.active_alerts.low}</p>
            </div>
            <div class="bg-blue-100 rounded-full p-3">
              <svg class="w-6 h-6 text-blue-600" fill="currentColor" viewBox="0 0 20 20">
                <path
                  fill-rule="evenodd"
                  d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z"
                  clip-rule="evenodd"
                />
              </svg>
            </div>
          </div>
        </div>

        <div class="bg-white rounded-xl shadow-sm p-6 border-l-4 border-gray-500">
          <div class="flex items-center justify-between">
            <div>
              <p class="text-sm font-medium text-gray-600">Total Active</p>
              <p class="text-3xl font-bold text-gray-700">{@alert_stats.active_alerts.total}</p>
            </div>
            <div class="bg-gray-100 rounded-full p-3">
              <svg class="w-6 h-6 text-gray-600" fill="currentColor" viewBox="0 0 20 20">
                <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
            </div>
          </div>
        </div>
      </div>
      
    <!-- Alert Trends and Response Metrics -->
      <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
          <h3 class="text-lg font-semibold text-gray-900 mb-4">📈 Alert Trends</h3>
          <div class="space-y-4">
            <div class="flex items-center justify-between">
              <span class="text-sm text-gray-600">Last 24 Hours</span>
              <span class="text-lg font-semibold text-gray-900">
                {@alert_stats.alert_trends.last_24h}
              </span>
            </div>
            <div class="flex items-center justify-between">
              <span class="text-sm text-gray-600">Last 7 Days</span>
              <span class="text-lg font-semibold text-gray-900">
                {@alert_stats.alert_trends.last_7d}
              </span>
            </div>
            <div class="flex items-center justify-between">
              <span class="text-sm text-gray-600">Growth Rate</span>
              <span class="text-lg font-semibold text-green-600">
                +{@alert_stats.alert_trends.growth_rate}%
              </span>
            </div>
          </div>
        </div>

        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
          <h3 class="text-lg font-semibold text-gray-900 mb-4">⚡ Response Metrics</h3>
          <div class="space-y-4">
            <div class="flex items-center justify-between">
              <span class="text-sm text-gray-600">Avg Acknowledgment Time</span>
              <span class="text-lg font-semibold text-blue-600">
                {@alert_stats.response_metrics.avg_acknowledgment_time}
              </span>
            </div>
            <div class="flex items-center justify-between">
              <span class="text-sm text-gray-600">Avg Resolution Time</span>
              <span class="text-lg font-semibold text-green-600">
                {@alert_stats.response_metrics.avg_resolution_time}
              </span>
            </div>
            <div class="flex items-center justify-between">
              <span class="text-sm text-gray-600">Escalation Rate</span>
              <span class="text-lg font-semibold text-orange-600">
                {@alert_stats.response_metrics.escalation_rate}%
              </span>
            </div>
          </div>
        </div>
      </div>
      
    <!-- Top Alert Sources and Notification Stats -->
      <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
          <h3 class="text-lg font-semibold text-gray-900 mb-4">🎯 Top Alert Sources</h3>
          <div class="space-y-3">
            <%= for source <- @alert_stats.top_alert_sources do %>
              <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                <span class="text-sm font-medium text-gray-700">{source.source}</span>
                <span class="text-sm text-gray-900 bg-blue-100 px-2 py-1 rounded">
                  {source.count}
                </span>
              </div>
            <% end %>
          </div>
        </div>

        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
          <h3 class="text-lg font-semibold text-gray-900 mb-4">📤 Notification Statistics</h3>
          <div class="space-y-3">
            <div class="flex items-center justify-between">
              <span class="text-sm text-gray-600">📧 Email Sent (Via Aritic)</span>
              <span class="text-sm font-semibold text-gray-900">
                {@alert_stats.notification_stats.email_sent}
              </span>
            </div>
            <div class="flex items-center justify-between">
              <span class="text-sm text-gray-600">📱 SMS Sent (Via Aritic)</span>
              <span class="text-sm font-semibold text-gray-900">
                {@alert_stats.notification_stats.sms_sent}
              </span>
            </div>
            <div class="flex items-center justify-between">
              <span class="text-sm text-gray-600">🔗 Webhook Calls </span>
              <span class="text-sm font-semibold text-gray-900">
                {@alert_stats.notification_stats.webhook_calls}
              </span>
            </div>
            <div class="flex items-center justify-between">
              <span class="text-sm text-gray-600">🔔 Push Notifications (Via Aritic)</span>
              <span class="text-sm font-semibold text-gray-900">
                {@alert_stats.notification_stats.push_notifications}
              </span>
            </div>
            <div class="flex items-center justify-between border-t pt-2">
              <span class="text-sm font-medium text-gray-700">Delivery Rate</span>
              <span class="text-sm font-bold text-green-600">
                {@alert_stats.notification_stats.delivery_rate}%
              </span>
            </div>
          </div>
        </div>
      </div>
    </div>
    """
  end

  defp render_active_alerts_tab(assigns) do
    ~H"""
    <div class="space-y-6">
      <!-- Current Alerts Summary -->
      <div class="bg-gradient-to-r from-red-50 to-orange-50 rounded-xl p-6 border border-red-100">
        <div class="flex items-center justify-between">
          <div>
            <h3 class="text-lg font-semibold text-gray-900">🚨 Current Alert Status</h3>
            <p class="text-gray-600">Real-time alert monitoring and management</p>
          </div>
          <div class="flex items-center space-x-6">
            <div class="text-center">
              <div class="text-2xl font-bold text-red-600">{@current_alerts.new_alerts}</div>
              <div class="text-sm text-gray-600">New</div>
            </div>
            <div class="text-center">
              <div class="text-2xl font-bold text-yellow-600">
                {@current_alerts.acknowledged_alerts}
              </div>
              <div class="text-sm text-gray-600">Acknowledged</div>
            </div>
            <div class="text-center">
              <div class="text-2xl font-bold text-green-600">{@current_alerts.resolved_alerts}</div>
              <div class="text-sm text-gray-600">Resolved</div>
            </div>
            <div class="text-center">
              <div class="text-2xl font-bold text-gray-700">{@current_alerts.total_alerts}</div>
              <div class="text-sm text-gray-600">Total</div>
            </div>
          </div>
        </div>
      </div>
      
    <!-- Active Alerts List -->
      <div class="bg-white rounded-xl shadow-sm border border-gray-100">
        <div class="px-6 py-4 border-b border-gray-200">
          <h3 class="text-lg font-semibold text-gray-900">Active Alerts</h3>
        </div>

        <div class="divide-y divide-gray-200">
          <%= if length(@current_alerts.alerts) > 0 do %>
            <%= for alert <- @current_alerts.alerts do %>
              <div class="p-6 hover:bg-gray-50 transition-colors">
                <div class="flex items-start justify-between">
                  <div class="flex-1">
                    <div class="flex items-center space-x-3">
                      <span class={"inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium " <> 
                                   case alert.severity do
                                     "critical" -> "bg-red-100 text-red-800"
                                     "high" -> "bg-orange-100 text-orange-800"
                                     "medium" -> "bg-yellow-100 text-yellow-800"
                                     "low" -> "bg-blue-100 text-blue-800"
                                     _ -> "bg-gray-100 text-gray-800"
                                   end}>
                        {String.upcase(alert.severity)}
                      </span>
                      <span class={"inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium " <> 
                                   case alert.status do
                                     "new" -> "bg-red-100 text-red-800"
                                     "acknowledged" -> "bg-yellow-100 text-yellow-800"
                                     "resolved" -> "bg-green-100 text-green-800"
                                     _ -> "bg-gray-100 text-gray-800"
                                   end}>
                        {String.upcase(alert.status)}
                      </span>
                    </div>

                    <h4 class="text-lg font-semibold text-gray-900 mt-2">{alert.title}</h4>
                    <p class="text-gray-600 mt-1">{alert.message}</p>

                    <div class="flex items-center space-x-4 mt-3 text-sm text-gray-500">
                      <span>ID: #{alert.id}</span>
                      <span>Type: {alert.type}</span>
                      <span>Created: {Calendar.strftime(alert.created_at, "%Y-%m-%d %H:%M")}</span>
                    </div>
                  </div>

                  <div class="flex items-center space-x-2 ml-4">
                    <%= if alert.status == "new" do %>
                      <button
                        phx-click="acknowledge_alert"
                        phx-value-alert_id={alert.id}
                        class="bg-yellow-600 text-white px-3 py-1 rounded text-sm hover:bg-yellow-700 transition-colors"
                      >
                        Acknowledge
                      </button>
                    <% end %>

                    <%= if alert.status in ["new", "acknowledged"] do %>
                      <button
                        phx-click="resolve_alert"
                        phx-value-alert_id={alert.id}
                        class="bg-green-600 text-white px-3 py-1 rounded text-sm hover:bg-green-700 transition-colors"
                      >
                        Resolve
                      </button>
                    <% end %>
                  </div>
                </div>
              </div>
            <% end %>
          <% else %>
            <div class="p-8 text-center">
              <svg
                class="w-12 h-12 text-gray-400 mx-auto mb-4"
                fill="none"
                stroke="currentColor"
                viewBox="0 0 24 24"
              >
                <path
                  stroke-linecap="round"
                  stroke-linejoin="round"
                  stroke-width="2"
                  d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"
                />
              </svg>
              <h3 class="text-lg font-medium text-gray-900 mb-2">No Active Alerts</h3>
              <p class="text-gray-500">All systems are operating normally</p>
            </div>
          <% end %>
        </div>
      </div>
    </div>
    """
  end

  defp render_rules_tab(assigns) do
    ~H"""
    <div class="space-y-6">
      <!-- Create Alert Rule Form -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">➕ Create New Alert Rule</h3>

        <form phx-submit="create_alert_rule" class="space-y-4">
          <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label class="block text-sm font-medium text-gray-700 mb-2">Rule Name</label>
              <input
                type="text"
                name="name"
                required
                class="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                placeholder="e.g., High Device Offline Count"
              />
            </div>

            <div>
              <label class="block text-sm font-medium text-gray-700 mb-2">Alert Type</label>
              <select
                name="alert_type"
                required
                class="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
              >
                <%= for alert_type <- @alert_config.alert_types do %>
                  <option value={alert_type.type}>{alert_type.name}</option>
                <% end %>
              </select>
            </div>
          </div>

          <div>
            <label class="block text-sm font-medium text-gray-700 mb-2">Description</label>
            <textarea
              name="description"
              rows="3"
              class="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
              placeholder="Describe when this alert should trigger..."
            ></textarea>
          </div>

          <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label class="block text-sm font-medium text-gray-700 mb-2">Severity</label>
              <select
                name="severity"
                required
                class="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
              >
                <option value="low">Low</option>
                <option value="medium">Medium</option>
                <option value="high">High</option>
                <option value="critical">Critical</option>
              </select>
            </div>

            <div>
              <label class="block text-sm font-medium text-gray-700 mb-2">
                Notification Channels
              </label>
              <input
                type="text"
                name="notification_channels"
                class="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                placeholder="email,sms,push"
              />
            </div>
          </div>

          <div class="flex items-center">
            <input
              type="checkbox"
              name="enabled"
              value="true"
              class="rounded border-gray-300 text-blue-600 shadow-sm focus:border-blue-300 focus:ring focus:ring-blue-200"
              checked
            />
            <label class="ml-2 text-sm text-gray-700">Enable this alert rule</label>
          </div>

          <button
            type="submit"
            class="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 transition-colors"
          >
            Create Alert Rule
          </button>
        </form>
      </div>
      
    <!-- Existing Alert Rules -->
      <div class="bg-white rounded-xl shadow-sm border border-gray-100">
        <div class="px-6 py-4 border-b border-gray-200">
          <h3 class="text-lg font-semibold text-gray-900">Configured Alert Rules</h3>
        </div>

        <div class="divide-y divide-gray-200">
          <%= for rule <- @alert_config.alert_rules do %>
            <div class="p-6">
              <div class="flex items-center justify-between">
                <div class="flex-1">
                  <div class="flex items-center space-x-3">
                    <h4 class="text-lg font-semibold text-gray-900">{rule.name}</h4>
                    <span class={"inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium " <> 
                                 case rule.severity do
                                   "critical" -> "bg-red-100 text-red-800"
                                   "high" -> "bg-orange-100 text-orange-800"
                                   "medium" -> "bg-yellow-100 text-yellow-800"
                                   "low" -> "bg-blue-100 text-blue-800"
                                 end}>
                      {String.upcase(rule.severity)}
                    </span>
                    <span class={"inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium " <> 
                                 if(rule.enabled, do: "bg-green-100 text-green-800", else: "bg-gray-100 text-gray-800")}>
                      {if rule.enabled, do: "ENABLED", else: "DISABLED"}
                    </span>
                  </div>

                  <div class="mt-2 text-sm text-gray-500">
                    <span>Type: {rule.type}</span>
                    <span class="mx-2">•</span>
                    <span>ID: #{rule.id}</span>
                  </div>
                </div>

                <div class="flex items-center space-x-2">
                  <button class="text-blue-600 hover:text-blue-800 text-sm">Edit</button>
                  <button class="text-red-600 hover:text-red-800 text-sm">Delete</button>
                </div>
              </div>
            </div>
          <% end %>
        </div>
      </div>
    </div>
    """
  end

  defp render_notifications_tab(assigns) do
    ~H"""
    <div class="space-y-6">
      <!-- Notification Channels Status -->
      <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <%= for channel <- @alert_config.notification_channels do %>
          <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
            <div class="flex items-center justify-between mb-4">
              <h3 class="text-lg font-semibold text-gray-900 capitalize">{channel.name}</h3>
              <span class={"inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium " <> 
                           if(channel.enabled, do: "bg-green-100 text-green-800", else: "bg-red-100 text-red-800")}>
                {if channel.enabled, do: "ENABLED", else: "DISABLED"}
              </span>
            </div>

            <div class="space-y-2">
              <button
                phx-click="test_notification"
                phx-value-channel={channel.channel}
                class="w-full bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors text-sm"
              >
                Test {channel.name}
              </button>

              <button class="w-full bg-gray-100 text-gray-700 px-4 py-2 rounded-lg hover:bg-gray-200 transition-colors text-sm">
                Configure
              </button>
            </div>
          </div>
        <% end %>
      </div>
      
    <!-- Notification Templates -->
      <div class="bg-white rounded-xl shadow-sm border border-gray-100">
        <div class="px-6 py-4 border-b border-gray-200">
          <h3 class="text-lg font-semibold text-gray-900">📝 Notification Templates</h3>
        </div>

        <div class="p-6">
          <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <div>
              <h4 class="font-medium text-gray-900 mb-3">Email Template</h4>
              <div class="bg-gray-50 p-4 rounded-lg">
                <div class="text-sm">
                  <div>
                    <div class="text-sm">
                      <div class="font-medium mb-2">
                        Subject: Alert: &#123;&#123;alert.title&#125;&#125;
                      </div>
                      <div class="text-gray-600">
                        Alert: &#123;&#123;alert.title&#125;&#125;<br />
                        Severity: &#123;&#123;alert.severity&#125;&#125;<br />
                        Time: &#123;&#123;alert.created_at&#125;&#125;<br /><br />
                        Description: &#123;&#123;alert.message&#125;&#125;<br /><br />
                        Please investigate and acknowledge this alert.
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>

            <div>
              <h4 class="font-medium text-gray-900 mb-3">SMS Template</h4>
              <div class="bg-gray-50 p-4 rounded-lg">
                <div>
                  <div class="text-sm text-gray-600">
                    ALERT: &#123;&#123;alert.title&#125;&#125;. &#123;&#123;alert.message&#125;&#125;. Check dashboard for details.
                  </div>
                </div>
              </div>
            </div>
          </div>

          <div class="mt-6 flex space-x-4">
            <button class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors">
              Create Template
            </button>
            <button class="bg-gray-100 text-gray-700 px-4 py-2 rounded-lg hover:bg-gray-200 transition-colors">
              Manage Templates
            </button>
          </div>
        </div>
      </div>
      
    <!-- Recent Notifications -->
      <div class="bg-white rounded-xl shadow-sm border border-gray-100">
        <div class="px-6 py-4 border-b border-gray-200">
          <h3 class="text-lg font-semibold text-gray-900">📤 Recent Notifications</h3>
        </div>

        <div class="divide-y divide-gray-200">
          <%= for i <- 1..5 do %>
            <div class="p-6">
              <div class="flex items-start justify-between">
                <div class="flex-1">
                  <div class="flex items-center space-x-3">
                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                      EMAIL
                    </span>
                    <span class="text-sm text-gray-500">
                      {Calendar.strftime(DateTime.add(DateTime.utc_now(), -i * 15, :minute), "%H:%M")}
                    </span>
                  </div>

                  <h4 class="text-lg font-semibold text-gray-900 mt-2">Device Offline Alert</h4>
                  <p class="text-gray-600 mt-1">Sent to admin@company.com</p>
                </div>

                <div class="ml-4">
                  <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                    DELIVERED
                  </span>
                </div>
              </div>
            </div>
          <% end %>
        </div>
      </div>
    </div>
    """
  end

  defp render_escalation_tab(assigns) do
    ~H"""
    <div class="space-y-6">
      <!-- Escalation Policies Overview -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">📈 Escalation Policies</h3>

        <%= for policy <- @alert_config.escalation_policies do %>
          <div class="border border-gray-200 rounded-lg p-4 mb-4">
            <div class="flex items-center justify-between mb-3">
              <h4 class="font-semibold text-gray-900">{policy.name}</h4>
              <span class="text-sm text-gray-500">Policy #{policy.id}</span>
            </div>

            <div class="space-y-2">
              <%= for step <- policy.steps do %>
                <div class="flex items-center space-x-4 p-3 bg-gray-50 rounded-lg">
                  <span class="inline-flex items-center justify-center w-6 h-6 bg-blue-600 text-white text-xs rounded-full">
                    {step.level}
                  </span>
                  <span class="text-sm text-gray-700">
                    After {step.delay_minutes} minutes → {Enum.join(step.channels, ", ")}
                  </span>
                </div>
              <% end %>
            </div>
          </div>
        <% end %>
      </div>
      
    <!-- Create Escalation Policy -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">➕ Create Escalation Policy</h3>

        <form class="space-y-4">
          <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label class="block text-sm font-medium text-gray-700 mb-2">Policy Name</label>
              <input
                type="text"
                class="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                placeholder="e.g., Critical Alert Escalation"
              />
            </div>

            <div>
              <label class="block text-sm font-medium text-gray-700 mb-2">Max Escalations</label>
              <select class="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500">
                <option value="3">3 levels</option>
                <option value="4">4 levels</option>
                <option value="5">5 levels</option>
              </select>
            </div>
          </div>

          <div>
            <label class="block text-sm font-medium text-gray-700 mb-2">Description</label>
            <textarea
              rows="3"
              class="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
              placeholder="Describe the escalation policy..."
            ></textarea>
          </div>

          <div class="border border-gray-200 rounded-lg p-4">
            <h4 class="font-medium text-gray-900 mb-3">Escalation Steps</h4>

            <div class="space-y-3">
              <div class="grid grid-cols-1 md:grid-cols-3 gap-3">
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-1">
                    Level 1 - Delay (minutes)
                  </label>
                  <input
                    type="number"
                    value="5"
                    class="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                  />
                </div>
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-1">Channels</label>
                  <input
                    type="text"
                    value="email"
                    class="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                  />
                </div>
                <div class="flex items-end">
                  <button
                    type="button"
                    class="bg-green-600 text-white px-3 py-2 rounded-lg hover:bg-green-700 transition-colors text-sm"
                  >
                    + Add Step
                  </button>
                </div>
              </div>
            </div>
          </div>

          <button
            type="submit"
            class="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 transition-colors"
          >
            Create Policy
          </button>
        </form>
      </div>
      
    <!-- Active Escalations -->
      <div class="bg-white rounded-xl shadow-sm border border-gray-100">
        <div class="px-6 py-4 border-b border-gray-200">
          <h3 class="text-lg font-semibold text-gray-900">🔄 Active Escalations</h3>
        </div>

        <div class="p-6">
          <div class="text-center py-8">
            <svg
              class="w-12 h-12 text-gray-400 mx-auto mb-4"
              fill="none"
              stroke="currentColor"
              viewBox="0 0 24 24"
            >
              <path
                stroke-linecap="round"
                stroke-linejoin="round"
                stroke-width="2"
                d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6"
              />
            </svg>
            <h3 class="text-lg font-medium text-gray-900 mb-2">No Active Escalations</h3>
            <p class="text-gray-500">All alerts are being handled within normal response times</p>
          </div>
        </div>
      </div>
    </div>
    """
  end
end
