defmodule DaProductAppWeb.AnalyticsLive do
  use DaProductAppWeb, :live_view
  alias DaProductApp.TerminalManagement

  @impl true
  def mount(_params, _session, socket) do
    # Subscribe to real-time updates
    if connected?(socket) do
      TerminalManagement.subscribe_to_device_updates()
    end

    socket =
      socket
      |> assign(:page_title, "Analytics & Reporting")
      |> assign(:current_tab, "overview")
      |> assign(:date_range, :last_30_days)
      |> assign(:loading, true)
      |> assign(:current_page, "anaylytics")
      |> load_analytics_data()

    {:ok, socket}
  end

  @impl true
  def handle_params(params, _url, socket) do
    tab = params["tab"] || "overview"
    date_range = String.to_atom(params["range"] || "last_30_days")

    socket =
      socket
      |> assign(:current_tab, tab)
      |> assign(:date_range, date_range)
      |> load_analytics_data()

    {:noreply, socket}
  end

  @impl true
  def handle_event("change_tab", %{"tab" => tab}, socket) do
    {:noreply,
     push_patch(socket, to: ~p"/analytics?tab=#{tab}&range=#{socket.assigns.date_range}")}
  end

  @impl true
  def handle_event("change_date_range", %{"range" => range}, socket) do
    date_range = String.to_atom(range)

    {:noreply,
     push_patch(socket, to: ~p"/analytics?tab=#{socket.assigns.current_tab}&range=#{range}")}
  end

  @impl true
  def handle_event("export_report", %{"format" => format}, socket) do
    analytics_data = socket.assigns.analytics_data

    case TerminalManagement.export_analytics_report(analytics_data, String.to_atom(format)) do
      {:ok, exported_data} ->
        # Generate filename with timestamp
        timestamp = DateTime.utc_now() |> DateTime.to_unix()

        # Determine content type and push appropriate download event
        case format do
          "json" ->
            filename = "analytics_report_#{timestamp}.json"

            {:noreply,
             push_event(socket, "download_file", %{
               content: exported_data,
               filename: filename,
               type: "application/json"
             })}

          "csv" ->
            filename = "analytics_report_#{timestamp}.csv"

            {:noreply,
             push_event(socket, "download_file", %{
               content: exported_data,
               filename: filename,
               type: "text/csv"
             })}

          "pdf" ->
            filename = "analytics_report_#{timestamp}.html"

            {:noreply,
             push_event(socket, "download_file", %{
               content: exported_data,
               filename: filename,
               type: "text/html"
             })}

          _ ->
            socket = put_flash(socket, :error, "Unsupported format: #{format}")
            {:noreply, socket}
        end

      {:error, reason} ->
        socket = put_flash(socket, :error, "Export failed: #{reason}")
        {:noreply, socket}
    end
  end

  @impl true
  def handle_event("schedule_report", params, socket) do
    schedule_config = %{
      schedule: params["schedule"],
      recipients: String.split(params["recipients"], ","),
      format: String.to_atom(params["format"]),
      filters: %{}
    }

    case TerminalManagement.schedule_analytics_report(schedule_config) do
      %{status: "scheduled"} = result ->
        socket =
          put_flash(socket, :info, "Report scheduled successfully (ID: #{result.report_id})")

        {:noreply, socket}

      _ ->
        socket = put_flash(socket, :error, "Failed to schedule report")
        {:noreply, socket}
    end
  end

  @impl true
  def handle_event("refresh_analytics", _params, socket) do
    socket =
      socket
      |> assign(:loading, true)
      |> load_analytics_data()

    {:noreply, socket}
  end

  @impl true
  def handle_info({:device_update, _device_data}, socket) do
    # Refresh analytics when device data changes
    socket = load_analytics_data(socket)
    {:noreply, socket}
  end

  @impl true
  def handle_info({:monitoring_update, _monitoring_data}, socket) do
    # Refresh analytics when monitoring data changes
    socket = load_analytics_data(socket)
    {:noreply, socket}
  end

  defp load_analytics_data(socket) do
    date_range = socket.assigns[:date_range] || :last_30_days

    analytics_data =
      TerminalManagement.generate_device_analytics(date_range, %{
        include_utilization: true,
        include_cost_analysis: true,
        include_predictions: true,
        include_location_data: true
      })

    # Load actual device data for geographic mapping
    devices = TerminalManagement.list_terminals_with_latest_status()

    # Prepare device data for map with location information
    map_devices =
      Enum.map(devices, fn device ->
        location = TerminalManagement.get_latest_terminal_location(device.id)

        %{
          serial_number: device.serial_number,
          status: device.status,
          area: device.area,
          vendor: device.vendor,
          model: device.model,
          lat: parse_float(location.lat),
          lng: parse_float(location.lng)
        }
      end)
      |> Enum.filter(fn device ->
        # Only include devices with valid coordinates
        device.lat != nil and device.lng != nil and
          is_number(device.lat) and is_number(device.lng)
      end)

    socket
    |> assign(:analytics_data, analytics_data)
    |> assign(:map_devices, map_devices)
    |> assign(:loading, false)
  end

  # Helper function to parse float values
  defp parse_float(nil), do: nil
  defp parse_float(value) when is_number(value), do: value

  defp parse_float(value) when is_binary(value) do
    case Float.parse(value) do
      {float_val, _} -> float_val
      :error -> nil
    end
  end

  defp parse_float(_), do: nil

  @impl true
  def render(assigns) do
    ~H"""
    <div class="min-h-screen bg-gradient-to-br from-slate-50 to-blue-50">
      <!-- Header Section -->
      <div class="bg-white shadow-sm border-b">
        <div class="px-6 py-6">
          <div class="flex justify-between items-center">
            <div>
              <h1 class="text-3xl font-bold text-gray-900">📊 Analytics & Reporting</h1>
              <p class="text-gray-600 mt-2">Comprehensive insights into your device ecosystem</p>
            </div>
            
    <!-- Date Range Selector -->
            <div class="flex items-center space-x-4">
              <select
                phx-change="change_date_range"
                name="range"
                value={@date_range}
                class="rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
              >
                <option value="today">Today</option>
                <option value="last_7_days">Last 7 Days</option>
                <option value="last_30_days">Last 30 Days</option>
                <option value="last_90_days">Last 90 Days</option>
              </select>

              <button
                phx-click="refresh_analytics"
                class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors flex items-center"
              >
                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path
                    stroke-linecap="round"
                    stroke-linejoin="round"
                    stroke-width="2"
                    d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"
                  />
                </svg>
                Refresh
              </button>
            </div>
          </div>
          
    <!-- Loading Indicator -->
          <%= if @loading do %>
            <div class="mt-4 flex items-center text-blue-600">
              <svg
                class="animate-spin -ml-1 mr-3 h-5 w-5 text-blue-600"
                fill="none"
                viewBox="0 0 24 24"
              >
                <circle
                  class="opacity-25"
                  cx="12"
                  cy="12"
                  r="10"
                  stroke="currentColor"
                  stroke-width="4"
                >
                </circle>
                <path
                  class="opacity-75"
                  fill="currentColor"
                  d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                >
                </path>
              </svg>
              Loading analytics data...
            </div>
          <% end %>
        </div>
        
    <!-- Tab Navigation -->
        <div class="border-t border-gray-200">
          <nav class="px-6 flex space-x-8">
            <button
              phx-click="change_tab"
              phx-value-tab="overview"
              class={"py-4 border-b-2 font-medium text-sm transition-colors " <> 
                     if(@current_tab == "overview", do: "border-blue-500 text-blue-600", else: "border-transparent text-gray-500 hover:text-gray-700")}
            >
              📈 Overview
            </button>
            <button
              phx-click="change_tab"
              phx-value-tab="trends"
              class={"py-4 border-b-2 font-medium text-sm transition-colors " <> 
                     if(@current_tab == "trends", do: "border-blue-500 text-blue-600", else: "border-transparent text-gray-500 hover:text-gray-700")}
            >
              📊 Trends
            </button>
            <button
              phx-click="change_tab"
              phx-value-tab="performance"
              class={"py-4 border-b-2 font-medium text-sm transition-colors " <> 
                     if(@current_tab == "performance", do: "border-blue-500 text-blue-600", else: "border-transparent text-gray-500 hover:text-gray-700")}
            >
              ⚡ Performance
            </button>
            <button
              phx-click="change_tab"
              phx-value-tab="geographic"
              class={"py-4 border-b-2 font-medium text-sm transition-colors " <> 
                     if(@current_tab == "geographic", do: "border-blue-500 text-blue-600", else: "border-transparent text-gray-500 hover:text-gray-700")}
            >
              🌍 Geographic
            </button>
            <button
              phx-click="change_tab"
              phx-value-tab="compliance"
              class={"py-4 border-b-2 font-medium text-sm transition-colors " <> 
                     if(@current_tab == "compliance", do: "border-blue-500 text-blue-600", else: "border-transparent text-gray-500 hover:text-gray-700")}
            >
              🛡️ Compliance
            </button>
            <button
              phx-click="change_tab"
              phx-value-tab="export"
              class={"py-4 border-b-2 font-medium text-sm transition-colors " <> 
                     if(@current_tab == "export", do: "border-blue-500 text-blue-600", else: "border-transparent text-gray-500 hover:text-gray-700")}
            >
              📤 Export
            </button>
          </nav>
        </div>
      </div>
      
    <!-- Main Content -->
      <div :if={!@loading and @analytics_data} class="p-6">
        <%= case @current_tab do %>
          <% "overview" -> %>
            {render_overview_tab(assigns)}
          <% "trends" -> %>
            {render_trends_tab(assigns)}
          <% "performance" -> %>
            {render_performance_tab(assigns)}
          <% "geographic" -> %>
            {render_geographic_tab(assigns)}
          <% "compliance" -> %>
            {render_compliance_tab(assigns)}
          <% "export" -> %>
            {render_export_tab(assigns)}
          <% _ -> %>
            {render_overview_tab(assigns)}
        <% end %>
      </div>
    </div>
    """
  end

  defp render_overview_tab(assigns) do
    ~H"""
    <div class="space-y-6">
      <!-- Key Metrics Cards -->
      <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
          <div class="flex items-center justify-between">
            <div>
              <p class="text-sm font-medium text-gray-600">Total Devices</p>
              <p class="text-3xl font-bold text-gray-900">{@analytics_data.summary.total_devices}</p>
            </div>
            <div class="bg-blue-100 rounded-full p-3">
              <svg class="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path
                  stroke-linecap="round"
                  stroke-linejoin="round"
                  stroke-width="2"
                  d="M9 12l2 2 4-4M7.835 4.697a3.42 3.42 0 001.946-.806 3.42 3.42 0 014.438 0 3.42 3.42 0 001.946.806 3.42 3.42 0 013.138 3.138 3.42 3.42 0 00.806 1.946 3.42 3.42 0 010 4.438 3.42 3.42 0 00-.806 1.946 3.42 3.42 0 01-3.138 3.138 3.42 3.42 0 00-1.946.806 3.42 3.42 0 01-4.438 0 3.42 3.42 0 00-1.946-.806 3.42 3.42 0 01-3.138-3.138 3.42 3.42 0 00-.806-1.946 3.42 3.42 0 010-4.438 3.42 3.42 0 00.806-1.946 3.42 3.42 0 013.138-3.138z"
                />
              </svg>
            </div>
          </div>
          <p class="text-sm text-gray-500 mt-2">
            +{@analytics_data.summary.new_devices} new devices this period
          </p>
        </div>

        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
          <div class="flex items-center justify-between">
            <div>
              <p class="text-sm font-medium text-gray-600">Active Devices</p>
              <p class="text-3xl font-bold text-green-600">
                {@analytics_data.summary.active_devices}
              </p>
            </div>
            <div class="bg-green-100 rounded-full p-3">
              <svg
                class="w-6 h-6 text-green-600"
                fill="none"
                stroke="currentColor"
                viewBox="0 0 24 24"
              >
                <path
                  stroke-linecap="round"
                  stroke-linejoin="round"
                  stroke-width="2"
                  d="M13 10V3L4 14h7v7l9-11h-7z"
                />
              </svg>
            </div>
          </div>
          <p class="text-sm text-gray-500 mt-2">
            {@analytics_data.summary.activation_rate}% activation rate
          </p>
        </div>

        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
          <div class="flex items-center justify-between">
            <div>
              <p class="text-sm font-medium text-gray-600">Performance Score</p>
              <p class="text-3xl font-bold text-purple-600">
                {@analytics_data.performance.overall_kpis.performance_grade}
              </p>
            </div>
            <div class="bg-purple-100 rounded-full p-3">
              <svg
                class="w-6 h-6 text-purple-600"
                fill="none"
                stroke="currentColor"
                viewBox="0 0 24 24"
              >
                <path
                  stroke-linecap="round"
                  stroke-linejoin="round"
                  stroke-width="2"
                  d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"
                />
              </svg>
            </div>
          </div>
          <p class="text-sm text-gray-500 mt-2">
            {@analytics_data.performance.overall_kpis.reliability_score}% reliability
          </p>
        </div>

        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
          <div class="flex items-center justify-between">
            <div>
              <p class="text-sm font-medium text-gray-600">Compliance Rate</p>
              <p class="text-3xl font-bold text-orange-600">
                {@analytics_data.compliance.compliance_score}%
              </p>
            </div>
            <div class="bg-orange-100 rounded-full p-3">
              <svg
                class="w-6 h-6 text-orange-600"
                fill="none"
                stroke="currentColor"
                viewBox="0 0 24 24"
              >
                <path
                  stroke-linecap="round"
                  stroke-linejoin="round"
                  stroke-width="2"
                  d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z"
                />
              </svg>
            </div>
          </div>
          <p class="text-sm text-gray-500 mt-2">
            {@analytics_data.compliance.device_compliance.non_compliant_devices} devices need attention
          </p>
        </div>
      </div>
      
    <!-- Status Distribution Chart -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">Device Status Distribution</h3>
        <div class="grid grid-cols-2 md:grid-cols-4 gap-4">
          <%= for {status, count} <- @analytics_data.summary.status_distribution do %>
            <div class="text-center p-4 rounded-lg border-2 border-gray-100">
              <p class="text-2xl font-bold text-gray-900">{count}</p>
              <p class="text-sm text-gray-600 capitalize">{status}</p>
            </div>
          <% end %>
        </div>
      </div>
      
    <!-- Model & Vendor Distribution -->
      <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
          <h3 class="text-lg font-semibold text-gray-900 mb-4">Top Device Models</h3>
          <div class="space-y-3">
            <%= for model_data <- Enum.take(@analytics_data.summary.model_distribution, 5) do %>
              <div class="flex items-center justify-between">
                <span class="text-sm font-medium text-gray-700">{model_data.model || "SR-600"}</span>
                <span class="text-sm text-gray-500">{model_data.count} devices</span>
              </div>
            <% end %>
          </div>
        </div>

        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
          <h3 class="text-lg font-semibold text-gray-900 mb-4">Top Vendors</h3>
          <div class="space-y-3">
            <%= for vendor_data <- Enum.take(@analytics_data.summary.vendor_distribution, 5) do %>
              <div class="flex items-center justify-between">
                <span class="text-sm font-medium text-gray-700">
                  {vendor_data.vendor || "Morefun"}
                </span>
                <span class="text-sm text-gray-500">{vendor_data.count} devices</span>
              </div>
            <% end %>
          </div>
        </div>
      </div>
    </div>
    """
  end

  defp render_trends_tab(assigns) do
    ~H"""
    <div class="space-y-6">
      <!-- Growth Metrics -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">📈 Growth Trends</h3>
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
          <div class="text-center">
            <p class="text-3xl font-bold text-green-600">{@analytics_data.trends.growth_rate}%</p>
            <p class="text-sm text-gray-600">Growth Rate</p>
          </div>
          <div class="text-center">
            <p class="text-3xl font-bold text-blue-600">
              {length(@analytics_data.trends.daily.registrations)}
            </p>
            <p class="text-sm text-gray-600">Active Days</p>
          </div>
          <div class="text-center">
            <p class="text-3xl font-bold text-purple-600">{length(@analytics_data.trends.weekly)}</p>
            <p class="text-sm text-gray-600">Weeks Tracked</p>
          </div>
        </div>
      </div>
      
    <!-- Daily Registration Trend -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">Daily Device Registrations</h3>
        <div class="space-y-2">
          <%= for reg_data <- Enum.take(@analytics_data.trends.daily.registrations, 10) do %>
            <div class="flex items-center justify-between py-2 border-b border-gray-100">
              <span class="text-sm text-gray-600">{reg_data.date}</span>
              <span class="text-sm font-medium text-gray-900">
                {reg_data.registrations} registrations
              </span>
            </div>
          <% end %>
        </div>
      </div>
      
    <!-- Weekly Trends -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">Weekly Trends</h3>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
          <%= for week_data <- @analytics_data.trends.weekly do %>
            <div class="p-4 bg-gray-50 rounded-lg">
              <p class="font-medium text-gray-900">Week {week_data.week}</p>
              <p class="text-sm text-gray-600">{week_data.registrations} new registrations</p>
            </div>
          <% end %>
        </div>
      </div>
    </div>
    """
  end

  defp render_performance_tab(assigns) do
    ~H"""
    <div class="space-y-6">
      <!-- Performance KPIs -->
      <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100 text-center">
          <p class="text-2xl font-bold text-green-600">
            {@analytics_data.performance.overall_kpis.overall_success_rate}%
          </p>
          <p class="text-sm text-gray-600">Overall Success Rate</p>
        </div>
        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100 text-center">
          <p class="text-2xl font-bold text-blue-600">
            {@analytics_data.performance.overall_kpis.operational_efficiency}%
          </p>
          <p class="text-sm text-gray-600">Operational Efficiency</p>
        </div>
        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100 text-center">
          <p class="text-2xl font-bold text-purple-600">
            {@analytics_data.performance.overall_kpis.reliability_score}%
          </p>
          <p class="text-sm text-gray-600">Reliability Score</p>
        </div>
        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100 text-center">
          <p class="text-2xl font-bold text-indigo-600">
            {@analytics_data.performance.overall_kpis.performance_grade}
          </p>
          <p class="text-sm text-gray-600">Performance Grade</p>
        </div>
      </div>
      
    <!-- Parameter Push Performance -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">📤 Parameter Push Performance</h3>
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
          <div>
            <p class="text-sm text-gray-600">Total Pushes</p>
            <p class="text-2xl font-bold text-gray-900">
              {@analytics_data.performance.parameter_push.total_pushes}
            </p>
          </div>
          <div>
            <p class="text-sm text-gray-600">Success Rate</p>
            <p class="text-2xl font-bold text-green-600">
              {@analytics_data.performance.parameter_push.success_rate}%
            </p>
          </div>
          <div>
            <p class="text-sm text-gray-600">Avg Response Time</p>
            <p class="text-2xl font-bold text-blue-600">
              {@analytics_data.performance.parameter_push.avg_response_time}
            </p>
          </div>
        </div>

        <div class="mt-4">
          <p class="text-sm font-medium text-gray-700 mb-2">Most Pushed Parameters:</p>
          <div class="flex flex-wrap gap-2">
            <%= for param <- @analytics_data.performance.parameter_push.most_pushed_parameters do %>
              <span class="px-3 py-1 bg-blue-100 text-blue-800 text-xs rounded-full">{param}</span>
            <% end %>
          </div>
        </div>
      </div>
      
    <!-- OTA Update Performance -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">🔄 OTA Update Performance</h3>
        <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
          <div>
            <p class="text-sm text-gray-600">Total Updates</p>
            <p class="text-2xl font-bold text-gray-900">
              {@analytics_data.performance.ota_updates.total_updates}
            </p>
          </div>
          <div>
            <p class="text-sm text-gray-600">Successful</p>
            <p class="text-2xl font-bold text-green-600">
              {@analytics_data.performance.ota_updates.successful_updates}
            </p>
          </div>
          <div>
            <p class="text-sm text-gray-600">Pending</p>
            <p class="text-2xl font-bold text-yellow-600">
              {@analytics_data.performance.ota_updates.pending_updates}
            </p>
          </div>
          <div>
            <p class="text-sm text-gray-600">Failed</p>
            <p class="text-2xl font-bold text-red-600">
              {@analytics_data.performance.ota_updates.failed_updates}
            </p>
          </div>
        </div>
      </div>
      
    <!-- Response Time Analytics -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">⚡ Response Time Analytics</h3>
        <div class="grid grid-cols-1 md:grid-cols-5 gap-4">
          <div class="text-center">
            <p class="text-xl font-bold text-gray-900">
              {@analytics_data.performance.response_times.avg_response_time}ms
            </p>
            <p class="text-xs text-gray-600">Average</p>
          </div>
          <div class="text-center">
            <p class="text-xl font-bold text-yellow-600">
              {@analytics_data.performance.response_times.p95_response_time}ms
            </p>
            <p class="text-xs text-gray-600">95th Percentile</p>
          </div>
          <div class="text-center">
            <p class="text-xl font-bold text-orange-600">
              {@analytics_data.performance.response_times.p99_response_time}ms
            </p>
            <p class="text-xs text-gray-600">99th Percentile</p>
          </div>
          <div class="text-center">
            <p class="text-xl font-bold text-green-600">
              {@analytics_data.performance.response_times.fastest_response}ms
            </p>
            <p class="text-xs text-gray-600">Fastest</p>
          </div>
          <div class="text-center">
            <p class="text-xl font-bold text-red-600">
              {@analytics_data.performance.response_times.slowest_response}ms
            </p>
            <p class="text-xs text-gray-600">Slowest</p>
          </div>
        </div>
      </div>
    </div>
    """
  end

  defp render_geographic_tab(assigns) do
    ~H"""
    <div class="space-y-6">
      <!-- Coverage Metrics -->
      <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100 text-center">
          <p class="text-2xl font-bold text-blue-600">
            {@analytics_data.geographic.coverage_metrics.total_areas_covered}
          </p>
          <p class="text-sm text-gray-600">Areas Covered</p>
        </div>
        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100 text-center">
          <p class="text-2xl font-bold text-green-600">
            {@analytics_data.geographic.coverage_metrics.avg_devices_per_area}
          </p>
          <p class="text-sm text-gray-600">Avg Devices/Area</p>
        </div>
        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100 text-center">
          <p class="text-2xl font-bold text-purple-600">
            {@analytics_data.geographic.coverage_metrics.area_coverage_score}
          </p>
          <p class="text-sm text-gray-600">Coverage Score</p>
        </div>
      </div>
      
    <!-- Area Distribution -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">🌍 Geographic Distribution</h3>
        <div class="space-y-4">
          <%= for area <- @analytics_data.geographic.area_distribution do %>
            <div class="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
              <div>
                <p class="font-medium text-gray-900">{area.area}</p>
                <p class="text-sm text-gray-600">{area.online_count}/{area.device_count} online</p>
              </div>
              <div class="text-right">
                <p class="text-lg font-bold text-gray-900">{area.device_count}</p>
                <p class="text-sm text-gray-600">devices</p>
              </div>
            </div>
          <% end %>
        </div>
      </div>
      
    <!-- Location Analytics (if enabled) -->
      <%= if @analytics_data.geographic.geographic_activity.enabled do %>
        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
          <h3 class="text-lg font-semibold text-gray-900 mb-4">📍 Location Analytics</h3>
          <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div>
              <h4 class="font-medium text-gray-900 mb-3">Regional Clusters</h4>
              <div class="space-y-2">
                <%= for cluster <- @analytics_data.geographic.geographic_activity.location_clusters do %>
                  <div class="flex items-center justify-between">
                    <span class="text-sm text-gray-700">{cluster.region}</span>
                    <div class="text-right">
                      <span class="text-sm font-medium">{cluster.device_count} devices</span>
                      <span class="text-xs text-gray-500">({cluster.activity_score}% active)</span>
                    </div>
                  </div>
                <% end %>
              </div>
            </div>

            <div>
              <h4 class="font-medium text-gray-900 mb-3">Mobility Patterns</h4>
              <div class="space-y-2">
                <div class="flex items-center justify-between">
                  <span class="text-sm text-gray-700">Stationary Devices</span>
                  <span class="text-sm font-medium">
                    {@analytics_data.geographic.geographic_activity.mobility_patterns.stationary_devices}%
                  </span>
                </div>
                <div class="flex items-center justify-between">
                  <span class="text-sm text-gray-700">Mobile Devices</span>
                  <span class="text-sm font-medium">
                    {@analytics_data.geographic.geographic_activity.mobility_patterns.mobile_devices}%
                  </span>
                </div>
                <div class="flex items-center justify-between">
                  <span class="text-sm text-gray-700">Avg Daily Movement</span>
                  <span class="text-sm font-medium">
                    {@analytics_data.geographic.geographic_activity.mobility_patterns.avg_daily_movement}
                  </span>
                </div>
              </div>
            </div>
          </div>
        </div>
      <% end %>
      
    <!-- Interactive Geographic Map -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <div class="flex items-center justify-between mb-4">
          <h3 class="text-lg font-semibold text-gray-900">🗺️ Device Location Map</h3>
          <div class="text-sm text-gray-600">
            Showing {length(@map_devices)} devices with location data
          </div>
        </div>

        <div
          id="analytics-geographic-map"
          phx-hook="AnalyticsGeographicMapHook"
          data-devices={Jason.encode!(@map_devices)}
          class="w-full h-96 rounded-lg shadow border"
        >
        </div>

        <div class="mt-4 flex items-center justify-center space-x-6 text-sm">
          <div class="flex items-center space-x-2">
            <div class="w-3 h-3 rounded-full bg-green-500"></div>
            <span class="text-gray-600">Online</span>
          </div>
          <div class="flex items-center space-x-2">
            <div class="w-3 h-3 rounded-full bg-red-500"></div>
            <span class="text-gray-600">Offline</span>
          </div>
          <div class="flex items-center space-x-2">
            <div class="w-3 h-3 rounded-full bg-orange-500"></div>
            <span class="text-gray-600">Unknown</span>
          </div>
        </div>
      </div>
    </div>
    """
  end

  defp render_compliance_tab(assigns) do
    ~H"""
    <div class="space-y-6">
      <!-- Compliance Overview -->
      <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100 text-center">
          <p class="text-3xl font-bold text-green-600">
            {@analytics_data.compliance.compliance_score}%
          </p>
          <p class="text-sm text-gray-600">Overall Compliance</p>
        </div>
        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100 text-center">
          <p class="text-3xl font-bold text-blue-600">
            {@analytics_data.compliance.security_metrics.security_score}%
          </p>
          <p class="text-sm text-gray-600">Security Score</p>
        </div>
        <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100 text-center">
          <p class="text-3xl font-bold text-orange-600">
            {@analytics_data.compliance.update_compliance.update_compliance_rate}%
          </p>
          <p class="text-sm text-gray-600">Update Compliance</p>
        </div>
      </div>
      
    <!-- Device Compliance -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">🛡️ Device Compliance Status</h3>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div>
            <div class="space-y-3">
              <div class="flex items-center justify-between">
                <span class="text-sm text-gray-700">Compliant Devices</span>
                <span class="text-sm font-medium text-green-600">
                  {@analytics_data.compliance.device_compliance.compliant_devices}
                </span>
              </div>
              <div class="flex items-center justify-between">
                <span class="text-sm text-gray-700">Non-Compliant Devices</span>
                <span class="text-sm font-medium text-red-600">
                  {@analytics_data.compliance.device_compliance.non_compliant_devices}
                </span>
              </div>
              <div class="flex items-center justify-between">
                <span class="text-sm text-gray-700">Compliance Rate</span>
                <span class="text-sm font-medium text-blue-600">
                  {@analytics_data.compliance.device_compliance.compliance_rate}%
                </span>
              </div>
            </div>
          </div>

          <div>
            <h4 class="font-medium text-gray-900 mb-2">Common Violations</h4>
            <div class="space-y-1">
              <%= for violation <- @analytics_data.compliance.device_compliance.common_violations do %>
                <span class="inline-block px-3 py-1 bg-red-100 text-red-800 text-xs rounded-full mr-2 mb-1">
                  {violation}
                </span>
              <% end %>
            </div>
          </div>
        </div>
      </div>
      
    <!-- Update Compliance -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">🔄 Update Compliance</h3>
        <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
          <div class="text-center">
            <p class="text-2xl font-bold text-yellow-600">
              {@analytics_data.compliance.update_compliance.devices_requiring_updates}
            </p>
            <p class="text-sm text-gray-600">Need Updates</p>
          </div>
          <div class="text-center">
            <p class="text-2xl font-bold text-red-600">
              {@analytics_data.compliance.update_compliance.overdue_updates}
            </p>
            <p class="text-sm text-gray-600">Overdue</p>
          </div>
          <div class="text-center">
            <p class="text-2xl font-bold text-blue-600">
              {@analytics_data.compliance.update_compliance.update_compliance_rate}%
            </p>
            <p class="text-sm text-gray-600">Compliance Rate</p>
          </div>
          <div class="text-center">
            <p class="text-2xl font-bold text-purple-600">
              {@analytics_data.compliance.update_compliance.avg_update_delay}
            </p>
            <p class="text-sm text-gray-600">Avg Delay</p>
          </div>
        </div>
      </div>
      
    <!-- Security Metrics -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">🔒 Security Metrics</h3>
        <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
          <div class="text-center">
            <p class="text-2xl font-bold text-red-600">
              {@analytics_data.compliance.security_metrics.vulnerabilities_detected}
            </p>
            <p class="text-sm text-gray-600">Vulnerabilities</p>
          </div>
          <div class="text-center">
            <p class="text-2xl font-bold text-green-600">
              {@analytics_data.compliance.security_metrics.security_patches_applied}
            </p>
            <p class="text-sm text-gray-600">Patches Applied</p>
          </div>
          <div class="text-center">
            <p class="text-2xl font-bold text-blue-600">
              {@analytics_data.compliance.security_metrics.encryption_compliance}%
            </p>
            <p class="text-sm text-gray-600">Encryption Compliance</p>
          </div>
          <div class="text-center">
            <p class="text-2xl font-bold text-purple-600">
              {@analytics_data.compliance.security_metrics.security_score}
            </p>
            <p class="text-sm text-gray-600">Security Score</p>
          </div>
        </div>
      </div>
    </div>
    """
  end

  defp render_export_tab(assigns) do
    ~H"""
    <div class="space-y-6">
      <!-- Quick Export -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">📤 Quick Export</h3>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
          <button
            phx-click="export_report"
            phx-value-format="json"
            class="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition-colors flex items-center justify-center"
          >
            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path
                stroke-linecap="round"
                stroke-linejoin="round"
                stroke-width="2"
                d="M12 10v6m0 0l-3-3m3 3l3-3M3 17V7a2 2 0 012-2h6l2 2h6a2 2 0 012 2v10a2 2 0 01-2 2H5a2 2 0 01-2-2z"
              />
            </svg>
            Export JSON
          </button>

          <button
            phx-click="export_report"
            phx-value-format="csv"
            class="bg-green-600 text-white px-6 py-3 rounded-lg hover:bg-green-700 transition-colors flex items-center justify-center"
          >
            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path
                stroke-linecap="round"
                stroke-linejoin="round"
                stroke-width="2"
                d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"
              />
            </svg>
            Export CSV
          </button>
        </div>
      </div>
      
    <!-- Scheduled Reports -->
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">⏰ Schedule Automated Reports</h3>

        <form phx-submit="schedule_report" class="space-y-4">
          <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label class="block text-sm font-medium text-gray-700 mb-2">Schedule Frequency</label>
              <select
                name="schedule"
                class="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
              >
                <option value="daily">Daily</option>
                <option value="weekly">Weekly</option>
                <option value="monthly">Monthly</option>
              </select>
            </div>

            <div>
              <label class="block text-sm font-medium text-gray-700 mb-2">Export Format</label>
              <select
                name="format"
                class="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
              >
                <option value="csv">CSV</option>
                <option value="json">JSON</option>
              </select>
            </div>
          </div>

          <div>
            <label class="block text-sm font-medium text-gray-700 mb-2">
              Email Recipients (comma-separated)
            </label>
            <input
              type="text"
              name="recipients"
              placeholder="admin@company.com, manager@company.com"
              class="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
            />
          </div>

          <button
            type="submit"
            class="bg-purple-600 text-white px-6 py-2 rounded-lg hover:bg-purple-700 transition-colors flex items-center"
          >
            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path
                stroke-linecap="round"
                stroke-linejoin="round"
                stroke-width="2"
                d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"
              />
            </svg>
            Schedule Report
          </button>
        </form>
      </div>
      
    <!-- Report Summary -->
      <div class="bg-gradient-to-r from-blue-50 to-purple-50 rounded-xl p-6 border border-blue-100">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">📊 Current Report Summary</h3>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
          <div>
            <p class="text-gray-600">
              Generated: {@analytics_data.generated_at |> DateTime.to_string()}
            </p>
            <p class="text-gray-600">
              Date Range: {@analytics_data.date_range.start |> DateTime.to_date()} to {@analytics_data.date_range.end
              |> DateTime.to_date()}
            </p>
            <p class="text-gray-600">Total Devices: {@analytics_data.summary.total_devices}</p>
          </div>
          <div>
            <p class="text-gray-600">
              Performance Grade: {@analytics_data.performance.overall_kpis.performance_grade}
            </p>
            <p class="text-gray-600">
              Compliance Score: {@analytics_data.compliance.compliance_score}%
            </p>
            <p class="text-gray-600">
              Coverage Areas: {@analytics_data.geographic.coverage_metrics.total_areas_covered}
            </p>
          </div>
        </div>
      </div>
    </div>

    <script>
      // Handle file download events for analytics export
      window.addEventListener("phx:download_file", (e) => {
        const {content, filename, type} = e.detail;
        const blob = new Blob([content], {type: type || 'text/plain'});
        const url = URL.createObjectURL(blob);
        const a = document.createElement("a");
        a.href = url;
        a.download = filename;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
      });
    </script>
    """
  end
end
