defmodule DaProductAppWeb.ApplicationPackageLive.Index do
  use DaProductAppWeb, :live_view
  alias DaProductApp.TerminalManagement
  alias DaProductApp.TerminalManagement.AppPackage

  @impl true
  def mount(_params, _session, socket) do
    changeset = AppPackage.changeset(%AppPackage{}, %{})
    form = to_form(changeset)
    # Enable file upload for :file_path (removed .img, only using known MIME types)
    {:ok,
     socket
     |> assign(
       filters: %{},
       app_packages: [],
       show_form: false,
       show_edit_form: false,
       edit_package: nil,
       form_changeset: changeset,
       current_page: "app_package",
       form: form
     )
     |> allow_upload(:file_path, accept: ~w(.zip .tar .apk .bin), max_entries: 1)}
  end

  @impl true
  def handle_event("close_panel", _params, socket) do
    changeset = AppPackage.changeset(%AppPackage{}, %{})
    form = to_form(changeset)

    {:noreply,
     assign(socket,
       show_form: false,
       show_edit_form: false,
       edit_package: nil,
       form_changeset: changeset,
       form: form
     )}
  end

  @impl true
  def handle_event("close_slide_over", _params, socket) do
    changeset = AppPackage.changeset(%AppPackage{}, %{})
    form = to_form(changeset)

    {:noreply,
     assign(socket,
       show_form: false,
       show_edit_form: false,
       edit_package: nil,
       form_changeset: changeset,
       form: form
     )}
  end

  @impl true
  def handle_params(_params, _url, socket) do
    app_packages = TerminalManagement.list_app_packages()
    {:noreply, assign(socket, app_packages: app_packages)}
  end

  @impl true
  def handle_event("show_form", _params, socket) do
    changeset = AppPackage.changeset(%AppPackage{}, %{})
    form = to_form(changeset)
    {:noreply, assign(socket, show_form: true, form_changeset: changeset, form: form)}
  end

  def handle_event("hide_form", _params, socket) do
    {:noreply, assign(socket, show_form: false)}
  end

  def handle_event("validate", %{"app_package" => params}, socket) do
    changeset = AppPackage.changeset(%AppPackage{}, params)
    form = to_form(%{changeset | action: :validate})
    {:noreply, assign(socket, form_changeset: %{changeset | action: :validate}, form: form)}
  end

  def handle_event("save", %{"app_package" => params}, socket) do
    # Handle file upload
    uploaded_files =
      consume_uploaded_entries(socket, :file_path, fn %{path: path}, _entry ->
        dest = Path.join(["priv/static/uploads/app_packages", Path.basename(path)])
        File.mkdir_p!(Path.dirname(dest))
        File.cp!(path, dest)
        {:ok, "/uploads/app_packages/" <> Path.basename(dest)}
      end)

    file_path = List.first(uploaded_files)
    params = Map.put(params, "file_path", file_path)

    case TerminalManagement.create_app_package(params) do
      {:ok, _pkg} ->
        app_packages = TerminalManagement.list_app_packages()
        changeset = AppPackage.changeset(%AppPackage{}, %{})
        form = to_form(changeset)

        {:noreply,
         assign(socket,
           app_packages: app_packages,
           show_form: false,
           form_changeset: changeset,
           form: form
         )}

      {:error, changeset} ->
        form = to_form(changeset)
        {:noreply, assign(socket, form_changeset: changeset, form: form)}
    end
  end

  def handle_event("edit", %{"id" => id}, socket) do
    pkg = TerminalManagement.get_app_package!(id)
    changeset = AppPackage.changeset(pkg, %{})
    form = to_form(changeset)

    {:noreply,
     assign(socket,
       show_edit_form: true,
       show_form: false,
       edit_package: pkg,
       form_changeset: changeset,
       form: form
     )}
  end

  def handle_event("update", %{"app_package" => params}, socket) do
    pkg = socket.assigns.edit_package
    # Handle file upload (optional, only if a new file is uploaded)
    uploaded_files =
      consume_uploaded_entries(socket, :file_path, fn %{path: path}, _entry ->
        dest = Path.join(["priv/static/uploads/app_packages", Path.basename(path)])
        File.mkdir_p!(Path.dirname(dest))
        File.cp!(path, dest)
        {:ok, "/uploads/app_packages/" <> Path.basename(dest)}
      end)

    params =
      case uploaded_files do
        [file_path | _] -> Map.put(params, "file_path", file_path)
        _ -> params
      end

    case TerminalManagement.update_app_package(pkg, params) do
      {:ok, _pkg} ->
        app_packages = TerminalManagement.list_app_packages()
        changeset = AppPackage.changeset(%AppPackage{}, %{})
        form = to_form(changeset)

        {:noreply,
         assign(socket,
           app_packages: app_packages,
           show_edit_form: false,
           show_form: false,
           edit_package: nil,
           form_changeset: changeset,
           form: form
         )}

      {:error, changeset} ->
        form = to_form(changeset)
        {:noreply, assign(socket, form_changeset: changeset, form: form)}
    end
  end
end
