defmodule DaProductAppWeb.AuditComplianceLive do
  use DaProductAppWeb, :live_view
  alias DaProductApp.TerminalManagement

  def mount(_params, _session, socket) do
    if connected?(socket) do
      # Subscribe to audit events
      Phoenix.PubSub.subscribe(DaProductApp.PubSub, "audit_events")
      # Schedule periodic updates
      :timer.send_interval(30_000, self(), :update_data)
    end

    socket =
      socket
      |> assign(:current_page, "audit_compliance")
      |> assign(:active_tab, "audit_trail")
      |> assign(:audit_data, %{})
      |> assign(:compliance_data, %{})
      |> assign(:security_data, %{})
      |> assign(:current_page, "audit")
      |> assign(:loading, true)
      |> load_initial_data()

    {:ok, socket}
  end

  def handle_params(params, _url, socket) do
    tab = params["tab"] || "audit_trail"
    {:noreply, assign(socket, :active_tab, tab)}
  end

  def handle_event("change_tab", %{"tab" => tab}, socket) do
    {:noreply, push_patch(socket, to: ~p"/audit-compliance?tab=#{tab}")}
  end

  def handle_event("filter_audit", %{"date_range" => date_range}, socket) do
    # Apply audit filters
    filters = %{date_range: date_range}
    audit_data = TerminalManagement.get_audit_trail(filters)

    socket = assign(socket, :audit_data, audit_data)
    {:noreply, socket}
  end

  def handle_event("export_audit", %{"format" => format}, socket) do
    # Export audit data
    case TerminalManagement.export_audit_data(format) do
      {:ok, export_url} ->
        {:noreply,
         socket
         |> put_flash(:info, "Audit export completed successfully")
         |> push_event("download", %{url: export_url})}

      {:error, reason} ->
        {:noreply, put_flash(socket, :error, "Export failed: #{reason}")}
    end
  end

  def handle_event("generate_compliance_report", %{"framework" => framework}, socket) do
    case TerminalManagement.generate_compliance_report(framework) do
      {:ok, report} ->
        {:noreply,
         socket
         |> put_flash(:info, "Compliance report generated: #{report.id}")
         |> update(:compliance_data, fn data ->
           Map.put(data, :latest_report, report)
         end)}

      {:error, reason} ->
        {:noreply, put_flash(socket, :error, "Report generation failed: #{reason}")}
    end
  end

  def handle_event("run_security_assessment", _params, socket) do
    case TerminalManagement.perform_security_assessment() do
      {:ok, assessment} ->
        security_data = TerminalManagement.get_security_dashboard()

        {:noreply,
         socket
         |> assign(:security_data, security_data)
         |> put_flash(:info, "Security assessment completed: #{assessment.id}")}

      {:error, reason} ->
        {:noreply, put_flash(socket, :error, "Assessment failed: #{reason}")}
    end
  end

  def handle_event("acknowledge_violation", %{"violation_id" => violation_id}, socket) do
    # Acknowledge compliance violation
    {:noreply, put_flash(socket, :info, "Violation #{violation_id} acknowledged")}
  end

  def handle_event("resolve_incident", %{"incident_id" => incident_id}, socket) do
    # Resolve security incident
    {:noreply, put_flash(socket, :info, "Security incident #{incident_id} resolved")}
  end

  def handle_info(:update_data, socket) do
    socket = load_initial_data(socket)
    {:noreply, socket}
  end

  def handle_info({:audit_event, audit_entry}, socket) do
    # Handle real-time audit events
    socket =
      update(socket, :audit_data, fn data ->
        recent_entries =
          [audit_entry | data[:recent_entries] || []]
          # Keep only recent 50 entries
          |> Enum.take(50)

        Map.put(data, :recent_entries, recent_entries)
      end)

    {:noreply, socket}
  end

  defp load_initial_data(socket) do
    audit_data = TerminalManagement.get_audit_trail()
    compliance_data = TerminalManagement.get_compliance_dashboard()
    security_data = TerminalManagement.get_security_dashboard()

    socket
    |> assign(:audit_data, audit_data)
    |> assign(:compliance_data, compliance_data)
    |> assign(:security_data, security_data)
    |> assign(:loading, false)
  end

  def render(assigns) do
    ~H"""
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
      <!-- Header -->
      <div class="mb-8">
        <h1 class="text-3xl font-bold text-gray-900">Audit & Compliance</h1>
        <p class="mt-2 text-gray-600">
          Monitor audit trails, compliance status, and security metrics
        </p>
      </div>
      
    <!-- Tab Navigation -->
      <div class="border-b border-gray-200 mb-6">
        <nav class="-mb-px flex space-x-8">
          <button
            phx-click="change_tab"
            phx-value-tab="audit_trail"
            class={[
              "py-2 px-1 border-b-2 font-medium text-sm",
              if(@active_tab == "audit_trail",
                do: "border-blue-500 text-blue-600",
                else: "border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300"
              )
            ]}
          >
            <.icon name="hero-document-text" class="w-5 h-5 inline mr-2" /> Audit Trail
          </button>

          <button
            phx-click="change_tab"
            phx-value-tab="compliance"
            class={[
              "py-2 px-1 border-b-2 font-medium text-sm",
              if(@active_tab == "compliance",
                do: "border-blue-500 text-blue-600",
                else: "border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300"
              )
            ]}
          >
            <.icon name="hero-shield-check" class="w-5 h-5 inline mr-2" /> Compliance
          </button>

          <button
            phx-click="change_tab"
            phx-value-tab="security"
            class={[
              "py-2 px-1 border-b-2 font-medium text-sm",
              if(@active_tab == "security",
                do: "border-blue-500 text-blue-600",
                else: "border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300"
              )
            ]}
          >
            <.icon name="hero-lock-closed" class="w-5 h-5 inline mr-2" /> Security
          </button>

          <button
            phx-click="change_tab"
            phx-value-tab="reports"
            class={[
              "py-2 px-1 border-b-2 font-medium text-sm",
              if(@active_tab == "reports",
                do: "border-blue-500 text-blue-600",
                else: "border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300"
              )
            ]}
          >
            <.icon name="hero-chart-bar" class="w-5 h-5 inline mr-2" /> Reports
          </button>
        </nav>
      </div>
      
    <!-- Tab Content -->
      <div class="space-y-6">
        <%= if @active_tab == "audit_trail" do %>
          {render_audit_trail(assigns)}
        <% end %>

        <%= if @active_tab == "compliance" do %>
          {render_compliance_dashboard(assigns)}
        <% end %>

        <%= if @active_tab == "security" do %>
          {render_security_monitoring(assigns)}
        <% end %>

        <%= if @active_tab == "reports" do %>
          {render_reports_section(assigns)}
        <% end %>
      </div>
    </div>
    """
  end

  defp render_audit_trail(assigns) do
    ~H"""
    <div class="space-y-6">
      <!-- Audit Summary Cards -->
      <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <div class="bg-white rounded-lg shadow p-6">
          <div class="flex items-center">
            <div class="flex-shrink-0">
              <.icon name="hero-document-text" class="h-8 w-8 text-blue-500" />
            </div>
            <div class="ml-5 w-0 flex-1">
              <dl>
                <dt class="text-sm font-medium text-gray-500 truncate">Total Entries</dt>
                <dd class="text-lg font-medium text-gray-900">{@audit_data[:total_entries] || 0}</dd>
              </dl>
            </div>
          </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
          <div class="flex items-center">
            <div class="flex-shrink-0">
              <.icon name="hero-users" class="h-8 w-8 text-green-500" />
            </div>
            <div class="ml-5 w-0 flex-1">
              <dl>
                <dt class="text-sm font-medium text-gray-500 truncate">User Actions</dt>
                <dd class="text-lg font-medium text-gray-900">
                  {get_in(@audit_data, [:summary, :user_actions]) || 0}
                </dd>
              </dl>
            </div>
          </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
          <div class="flex items-center">
            <div class="flex-shrink-0">
              <.icon name="hero-cog-6-tooth" class="h-8 w-8 text-yellow-500" />
            </div>
            <div class="ml-5 w-0 flex-1">
              <dl>
                <dt class="text-sm font-medium text-gray-500 truncate">System Events</dt>
                <dd class="text-lg font-medium text-gray-900">
                  {get_in(@audit_data, [:summary, :system_events]) || 0}
                </dd>
              </dl>
            </div>
          </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
          <div class="flex items-center">
            <div class="flex-shrink-0">
              <.icon name="hero-exclamation-triangle" class="h-8 w-8 text-red-500" />
            </div>
            <div class="ml-5 w-0 flex-1">
              <dl>
                <dt class="text-sm font-medium text-gray-500 truncate">Security Events</dt>
                <dd class="text-lg font-medium text-gray-900">
                  {get_in(@audit_data, [:summary, :security_events]) || 0}
                </dd>
              </dl>
            </div>
          </div>
        </div>
      </div>
      
    <!-- Audit Controls -->
      <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between mb-4">
          <h3 class="text-lg font-medium text-gray-900">Audit Controls</h3>
          <div class="flex space-x-4">
            <form phx-submit="filter_audit" class="flex items-center space-x-2">
              <select
                name="date_range"
                class="border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500"
              >
                <option value="7d">Last 7 days</option>
                <option value="30d" selected>Last 30 days</option>
                <option value="90d">Last 90 days</option>
                <option value="1y">Last year</option>
              </select>
              <button
                type="submit"
                class="bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700"
              >
                Filter
              </button>
            </form>

            <button
              phx-click="export_audit"
              phx-value-format="csv"
              class="bg-green-600 text-white px-4 py-2 rounded-md hover:bg-green-700"
            >
              Export CSV
            </button>
          </div>
        </div>
      </div>
      
    <!-- Recent Audit Entries -->
      <div class="bg-white rounded-lg shadow">
        <div class="px-6 py-4 border-b border-gray-200">
          <h3 class="text-lg font-medium text-gray-900">Recent Audit Entries</h3>
        </div>
        <div class="overflow-hidden">
          <table class="min-w-full divide-y divide-gray-200">
            <thead class="bg-gray-50">
              <tr>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Timestamp
                </th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Event Type
                </th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  User
                </th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Resource
                </th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Details
                </th>
              </tr>
            </thead>
            <tbody class="bg-white divide-y divide-gray-200">
              <%= for entry <- (@audit_data[:recent_entries] || []) do %>
                <tr>
                  <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    {Calendar.strftime(entry.timestamp, "%Y-%m-%d %H:%M")}
                  </td>
                  <td class="px-6 py-4 whitespace-nowrap">
                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                      {entry.event_type}
                    </span>
                  </td>
                  <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    {entry.user_id}
                  </td>
                  <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                    {"#{entry.resource_type}/#{entry.resource_id}"}
                  </td>
                  <td class="px-6 py-4 text-sm text-gray-500">
                    {inspect(entry.details)}
                  </td>
                </tr>
              <% end %>
            </tbody>
          </table>
        </div>
      </div>
    </div>
    """
  end

  defp render_compliance_dashboard(assigns) do
    ~H"""
    <div class="space-y-6">
      <!-- Compliance Overview -->
      <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between mb-6">
          <h3 class="text-lg font-medium text-gray-900">Compliance Overview</h3>
          <div class="flex items-center space-x-4">
            <div class="text-center">
              <div class="text-3xl font-bold text-green-600">
                {@compliance_data[:overall_score] || 0}%
              </div>
              <div class="text-sm text-gray-500">Overall Score</div>
            </div>
          </div>
        </div>
        
    <!-- Compliance Frameworks -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          <%= for {framework, data} <- (@compliance_data[:compliance_frameworks] || %{}) do %>
            <div class="border rounded-lg p-4">
              <div class="flex items-center justify-between mb-2">
                <h4 class="font-medium text-gray-900 uppercase">{framework}</h4>
                <span class={[
                  "inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium",
                  case data.status do
                    "compliant" -> "bg-green-100 text-green-800"
                    "mostly_compliant" -> "bg-yellow-100 text-yellow-800"
                    "partially_compliant" -> "bg-orange-100 text-orange-800"
                    _ -> "bg-red-100 text-red-800"
                  end
                ]}>
                  {String.replace(data.status, "_", " ")}
                </span>
              </div>
              <div class="space-y-1">
                <div class="flex justify-between text-sm">
                  <span>Score:</span>
                  <span class="font-medium">{data.score}%</span>
                </div>
                <div class="flex justify-between text-sm">
                  <span>Requirements:</span>
                  <span>{data.requirements_met}/{data.total_requirements}</span>
                </div>
                <div class="flex justify-between text-sm">
                  <span>Violations:</span>
                  <span class={
                    if data.violations > 0, do: "text-red-600 font-medium", else: "text-green-600"
                  }>
                    {data.violations}
                  </span>
                </div>
              </div>
            </div>
          <% end %>
        </div>
      </div>
      
    <!-- Active Violations -->
      <div class="bg-white rounded-lg shadow">
        <div class="px-6 py-4 border-b border-gray-200">
          <h3 class="text-lg font-medium text-gray-900">Active Compliance Violations</h3>
        </div>
        <div class="overflow-hidden">
          <%= if (@compliance_data[:active_violations] || []) == [] do %>
            <div class="px-6 py-8 text-center">
              <.icon name="hero-check-circle" class="mx-auto h-12 w-12 text-green-400" />
              <p class="mt-2 text-sm text-gray-500">No active compliance violations</p>
            </div>
          <% else %>
            <table class="min-w-full divide-y divide-gray-200">
              <thead class="bg-gray-50">
                <tr>
                  <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Framework
                  </th>
                  <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Requirement
                  </th>
                  <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Severity
                  </th>
                  <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Status
                  </th>
                  <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Actions
                  </th>
                </tr>
              </thead>
              <tbody class="bg-white divide-y divide-gray-200">
                <%= for violation <- (@compliance_data[:active_violations] || []) do %>
                  <tr>
                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                      {violation.framework}
                    </td>
                    <td class="px-6 py-4 text-sm text-gray-900">
                      {violation.requirement}
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                      <span class={[
                        "inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium",
                        case violation.severity do
                          "high" -> "bg-red-100 text-red-800"
                          "medium" -> "bg-yellow-100 text-yellow-800"
                          _ -> "bg-green-100 text-green-800"
                        end
                      ]}>
                        {violation.severity}
                      </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                      {String.replace(violation.status, "_", " ")}
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                      <button
                        phx-click="acknowledge_violation"
                        phx-value-violation_id={violation.id}
                        class="text-blue-600 hover:text-blue-900"
                      >
                        Acknowledge
                      </button>
                    </td>
                  </tr>
                <% end %>
              </tbody>
            </table>
          <% end %>
        </div>
      </div>
      
    <!-- Compliance Reports -->
      <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between mb-4">
          <h3 class="text-lg font-medium text-gray-900">Compliance Reports</h3>
          <div class="flex space-x-2">
            <button
              phx-click="generate_compliance_report"
              phx-value-framework="pci_dss"
              class="bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700"
            >
              Generate PCI DSS Report
            </button>
            <button
              phx-click="generate_compliance_report"
              phx-value-framework="gdpr"
              class="bg-green-600 text-white px-4 py-2 rounded-md hover:bg-green-700"
            >
              Generate GDPR Report
            </button>
          </div>
        </div>

        <%= if @compliance_data[:latest_report] do %>
          <div class="border rounded-lg p-4 bg-green-50">
            <p class="text-sm text-green-800">
              Latest report generated: <strong>{@compliance_data[:latest_report].id}</strong>
              for {@compliance_data[:latest_report].framework}
            </p>
          </div>
        <% end %>
      </div>
    </div>
    """
  end

  defp render_security_monitoring(assigns) do
    ~H"""
    <div class="space-y-6">
      <!-- Security Overview -->
      <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between mb-6">
          <h3 class="text-lg font-medium text-gray-900">Security Dashboard</h3>
          <div class="flex items-center space-x-4">
            <div class="text-center">
              <div class="text-3xl font-bold text-green-600">
                {@security_data[:security_score] || 0}%
              </div>
              <div class="text-sm text-gray-500">Security Score</div>
            </div>
            <div class="text-center">
              <div class={[
                "text-2xl font-bold",
                case @security_data[:threat_level] do
                  "low" -> "text-green-600"
                  "medium" -> "text-yellow-600"
                  "high" -> "text-orange-600"
                  "critical" -> "text-red-600"
                  _ -> "text-gray-600"
                end
              ]}>
                {String.upcase(@security_data[:threat_level] || "UNKNOWN")}
              </div>
              <div class="text-sm text-gray-500">Threat Level</div>
            </div>
          </div>
        </div>
        
    <!-- Security Metrics -->
        <div class="grid grid-cols-1 md:grid-cols-3 lg:grid-cols-6 gap-4">
          <%= for {metric, value} <- (@security_data[:security_metrics] || %{}) do %>
            <div class="text-center p-3 border rounded-lg">
              <div class="text-lg font-semibold text-gray-900">{value}</div>
              <div class="text-xs text-gray-500">
                {String.replace(to_string(metric), "_", " ") |> String.capitalize()}
              </div>
            </div>
          <% end %>
        </div>
      </div>
      
    <!-- Security Assessment -->
      <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between mb-4">
          <h3 class="text-lg font-medium text-gray-900">Security Assessment</h3>
          <button
            phx-click="run_security_assessment"
            class="bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700"
          >
            Run Assessment
          </button>
        </div>

        <div class="text-sm text-gray-600">
          Last security scan:
          <%= if @security_data[:last_security_scan] do %>
            {Calendar.strftime(@security_data[:last_security_scan], "%Y-%m-%d %H:%M")}
          <% else %>
            Never
          <% end %>
        </div>
      </div>
      
    <!-- Recent Security Events -->
      <div class="bg-white rounded-lg shadow">
        <div class="px-6 py-4 border-b border-gray-200">
          <h3 class="text-lg font-medium text-gray-900">Recent Security Events</h3>
        </div>
        <div class="overflow-hidden">
          <%= if (@security_data[:recent_security_events] || []) == [] do %>
            <div class="px-6 py-8 text-center">
              <.icon name="hero-shield-check" class="mx-auto h-12 w-12 text-green-400" />
              <p class="mt-2 text-sm text-gray-500">No recent security events</p>
            </div>
          <% else %>
            <table class="min-w-full divide-y divide-gray-200">
              <thead class="bg-gray-50">
                <tr>
                  <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Timestamp
                  </th>
                  <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Event Type
                  </th>
                  <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Severity
                  </th>
                  <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Source
                  </th>
                  <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Details
                  </th>
                </tr>
              </thead>
              <tbody class="bg-white divide-y divide-gray-200">
                <%= for event <- (@security_data[:recent_security_events] || []) do %>
                  <tr>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                      {Calendar.strftime(event.timestamp, "%Y-%m-%d %H:%M")}
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                      {event.type}
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                      <span class={[
                        "inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium",
                        case event.severity do
                          "critical" -> "bg-red-100 text-red-800"
                          "high" -> "bg-orange-100 text-orange-800"
                          "medium" -> "bg-yellow-100 text-yellow-800"
                          _ -> "bg-green-100 text-green-800"
                        end
                      ]}>
                        {event.severity}
                      </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                      {event.source_ip}
                    </td>
                    <td class="px-6 py-4 text-sm text-gray-500">
                      {event.details}
                    </td>
                  </tr>
                <% end %>
              </tbody>
            </table>
          <% end %>
        </div>
      </div>
    </div>
    """
  end

  defp render_reports_section(assigns) do
    ~H"""
    <div class="space-y-6">
      <!-- Report Generation -->
      <div class="bg-white rounded-lg shadow p-6">
        <h3 class="text-lg font-medium text-gray-900 mb-4">Generate Reports</h3>

        <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
          <!-- Audit Reports -->
          <div class="border rounded-lg p-4">
            <h4 class="font-medium text-gray-900 mb-2">Audit Reports</h4>
            <p class="text-sm text-gray-600 mb-4">Comprehensive audit trail reports</p>
            <div class="space-y-2">
              <button class="w-full bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700">
                Download Audit Report
              </button>
              <button class="w-full bg-gray-600 text-white px-4 py-2 rounded-md hover:bg-gray-700">
                Schedule Report
              </button>
            </div>
          </div>
          
    <!-- Compliance Reports -->
          <div class="border rounded-lg p-4">
            <h4 class="font-medium text-gray-900 mb-2">Compliance Reports</h4>
            <p class="text-sm text-gray-600 mb-4">Regulatory compliance assessments</p>
            <div class="space-y-2">
              <button class="w-full bg-green-600 text-white px-4 py-2 rounded-md hover:bg-green-700">
                Download Compliance Report
              </button>
              <button class="w-full bg-gray-600 text-white px-4 py-2 rounded-md hover:bg-gray-700">
                Schedule Assessment
              </button>
            </div>
          </div>
          
    <!-- Security Reports -->
          <div class="border rounded-lg p-4">
            <h4 class="font-medium text-gray-900 mb-2">Security Reports</h4>
            <p class="text-sm text-gray-600 mb-4">Security monitoring and incidents</p>
            <div class="space-y-2">
              <button class="w-full bg-red-600 text-white px-4 py-2 rounded-md hover:bg-red-700">
                Download Security Report
              </button>
              <button class="w-full bg-gray-600 text-white px-4 py-2 rounded-md hover:bg-gray-700">
                Schedule Security Scan
              </button>
            </div>
          </div>
        </div>
      </div>
      
    <!-- Report History -->
      <div class="bg-white rounded-lg shadow">
        <div class="px-6 py-4 border-b border-gray-200">
          <h3 class="text-lg font-medium text-gray-900">Report History</h3>
        </div>
        <div class="overflow-hidden">
          <table class="min-w-full divide-y divide-gray-200">
            <thead class="bg-gray-50">
              <tr>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Report Type
                </th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Generated
                </th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Status
                </th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Size
                </th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Actions
                </th>
              </tr>
            </thead>
            <tbody class="bg-white divide-y divide-gray-200">
              <tr>
                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                  Compliance Assessment
                </td>
                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">2024-01-15 10:30</td>
                <td class="px-6 py-4 whitespace-nowrap">
                  <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                    Completed
                  </span>
                </td>
                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">2.4 MB</td>
                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                  <button class="text-blue-600 hover:text-blue-900 mr-4">Download</button>
                  <button class="text-red-600 hover:text-red-900">Delete</button>
                </td>
              </tr>
              <tr>
                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                  Security Scan
                </td>
                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">2024-01-14 15:45</td>
                <td class="px-6 py-4 whitespace-nowrap">
                  <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                    Completed
                  </span>
                </td>
                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">1.8 MB</td>
                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                  <button class="text-blue-600 hover:text-blue-900 mr-4">Download</button>
                  <button class="text-red-600 hover:text-red-900">Delete</button>
                </td>
              </tr>
            </tbody>
          </table>
        </div>
      </div>
    </div>
    """
  end
end
