defmodule DaProductAppWeb.DashboardLive do
  use DaProductAppWeb, :live_view
  import Logger
  alias DaProductApp.Users
  alias DaProductApp.TerminalManagement

  # on_mount DaProductAppWeb.Live.SetCurrentPage
  def mount(_params, session, socket) do
    user_token = session["user_token"]
    user = user_token && Users.get_user_by_session_token(user_token)

    current_user = user || nil
    # Correct usage
    Logger.debug("Socket Assigns: #{inspect(socket.assigns)}", [])

    # current_user = Map.get(session, "current_user", nil)

    stats = TerminalManagement.get_dashboard_statistics()
    online_rate = TerminalManagement.get_device_online_rate()
    terminals = TerminalManagement.list_terminals_with_latest_status()
    chart_data = TerminalManagement.get_chart_data("today")

    # Phase 3.1: Add real-time monitoring data
    monitoring_data = TerminalManagement.get_realtime_monitoring_data()

    # Subscribe to real-time updates
    if connected?(socket) do
      TerminalManagement.subscribe_to_device_updates()
      # Set up periodic updates every 30 seconds
      Process.send_after(self(), :update_monitoring_data, 30_000)
    end

    socket =
      socket
      |> assign(:current_user, current_user)
      # Enable sidebar for dashboard
      |> assign(:show_sidebar, true)
      # Set the page title
      |> assign(:page_title, "Dashboard")
      # Set the current page for sidebar highlighting
      |> assign(:current_page, :dashboard)
      |> assign(:stats, stats)
      |> assign(:online_rate, online_rate)
      |> assign(:terminals, terminals)
      |> assign(:selected_terminal, nil)
      |> assign(:terminal_logs, [])
      |> assign(:show_panel, false)
      |> assign(:chart_data, chart_data)
      |> assign(:selected_period, "today")
      # Phase 3.1: Real-time monitoring assigns
      |> assign(:monitoring_data, monitoring_data)
      |> assign(:device_health, monitoring_data.device_health)
      |> assign(:connection_status, monitoring_data.connection_status)
      |> assign(:performance_metrics, monitoring_data.performance_metrics)
      |> assign(:alert_summary, monitoring_data.alert_summary)
      |> assign(:show_monitoring_details, false)

    {:ok, socket}
  end

  def render(assigns) do
    ~H"""
    <!-- <p>Welcome to the dashboard, <%= @current_user && @current_user.email || "Guest" %>!</p> -->

    <!-- Phase 3.1: Real-time Monitoring Header -->
    <div class="mb-6 bg-gradient-to-r from-blue-600 to-indigo-600 text-white rounded-lg p-4">
      <div class="flex items-center justify-between">
        <div>
          <h2 class="text-xl font-bold">Real-time Device Monitoring</h2>
          <p class="text-blue-100">
            Live updates every 30 seconds • Last updated: {Calendar.strftime(
              @monitoring_data.last_updated,
              "%H:%M:%S"
            )}
          </p>
        </div>
        <div class="flex items-center space-x-4">
          <div class="text-center">
            <div class="text-2xl font-bold">{@device_health.total}</div>
            <div class="text-sm text-blue-100">Total Devices</div>
          </div>
          <div class="text-center">
            <div class="text-2xl font-bold text-green-300">{@device_health.healthy}</div>
            <div class="text-sm text-blue-100">Healthy</div>
          </div>
          <%= if @alert_summary.total_alerts > 0 do %>
            <div class="text-center">
              <div class="text-2xl font-bold text-red-300">{@alert_summary.total_alerts}</div>
              <div class="text-sm text-blue-100">Alerts</div>
            </div>
          <% end %>
        </div>
      </div>
    </div>

    <!-- Phase 3.1: Device Health Status Cards -->
    <div class="grid grid-cols-4 gap-4 mb-6">
      <div class="bg-white rounded-lg shadow p-4 border-l-4 border-green-500">
        <div class="flex items-center">
          <div class="flex-shrink-0">
            <div class="w-8 h-8 bg-green-100 rounded-full flex items-center justify-center">
              <svg class="w-4 h-4 text-green-600" fill="currentColor" viewBox="0 0 20 20">
                <path
                  fill-rule="evenodd"
                  d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                  clip-rule="evenodd"
                >
                </path>
              </svg>
            </div>
          </div>
          <div class="ml-4">
            <div class="text-2xl font-bold text-gray-900">{@device_health.healthy}</div>
            <div class="text-sm text-gray-500">Healthy Devices</div>
            <div class="text-xs text-green-600">Last 5 minutes</div>
          </div>
        </div>
      </div>

      <div class="bg-white rounded-lg shadow p-4 border-l-4 border-yellow-500">
        <div class="flex items-center">
          <div class="flex-shrink-0">
            <div class="w-8 h-8 bg-yellow-100 rounded-full flex items-center justify-center">
              <svg class="w-4 h-4 text-yellow-600" fill="currentColor" viewBox="0 0 20 20">
                <path
                  fill-rule="evenodd"
                  d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z"
                  clip-rule="evenodd"
                >
                </path>
              </svg>
            </div>
          </div>
          <div class="ml-4">
            <div class="text-2xl font-bold text-gray-900">{@device_health.warning}</div>
            <div class="text-sm text-gray-500">Warning</div>
            <div class="text-xs text-yellow-600">5-30 minutes ago</div>
          </div>
        </div>
      </div>

      <div class="bg-white rounded-lg shadow p-4 border-l-4 border-red-500">
        <div class="flex items-center">
          <div class="flex-shrink-0">
            <div class="w-8 h-8 bg-red-100 rounded-full flex items-center justify-center">
              <svg class="w-4 h-4 text-red-600" fill="currentColor" viewBox="0 0 20 20">
                <path
                  fill-rule="evenodd"
                  d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z"
                  clip-rule="evenodd"
                >
                </path>
              </svg>
            </div>
          </div>
          <div class="ml-4">
            <div class="text-2xl font-bold text-gray-900">{@device_health.critical}</div>
            <div class="text-sm text-gray-500">Critical</div>
            <div class="text-xs text-red-600">30-60 minutes ago</div>
          </div>
        </div>
      </div>

      <div class="bg-white rounded-lg shadow p-4 border-l-4 border-gray-500">
        <div class="flex items-center">
          <div class="flex-shrink-0">
            <div class="w-8 h-8 bg-gray-100 rounded-full flex items-center justify-center">
              <svg class="w-4 h-4 text-gray-600" fill="currentColor" viewBox="0 0 20 20">
                <path
                  fill-rule="evenodd"
                  d="M13.477 14.89A6 6 0 015.11 6.524l8.367 8.368zm1.414-1.414L6.524 5.11a6 6 0 008.367 8.367zM18 10a8 8 0 11-16 0 8 8 0 0116 0z"
                  clip-rule="evenodd"
                >
                </path>
              </svg>
            </div>
          </div>
          <div class="ml-4">
            <div class="text-2xl font-bold text-gray-900">{@device_health.offline}</div>
            <div class="text-sm text-gray-500">Offline</div>
            <div class="text-xs text-gray-600">Over 1 hour ago</div>
          </div>
        </div>
      </div>
    </div>

    <!-- Phase 3.1: Performance Metrics -->
    <div class="grid grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
      <div class="bg-white rounded-lg shadow p-4">
        <div class="text-sm font-medium text-gray-500">Success Rate</div>
        <div class="text-2xl font-bold text-green-600">
          {Float.round(@performance_metrics.parameter_push_success_rate, 1)}%
        </div>
        <div class="text-xs text-gray-400">Parameter Operations</div>
      </div>

      <div class="bg-white rounded-lg shadow p-4">
        <div class="text-sm font-medium text-gray-500">Avg Response</div>
        <div class="text-2xl font-bold text-blue-600">
          {@performance_metrics.avg_response_time_ms}ms
        </div>
        <div class="text-xs text-gray-400">Last Hour</div>
      </div>

      <div class="bg-white rounded-lg shadow p-4">
        <div class="text-sm font-medium text-gray-500">Uptime</div>
        <div class="text-2xl font-bold text-indigo-600">
          {@performance_metrics.uptime_percentage}%
        </div>
        <div class="text-xs text-gray-400">Last Hour</div>
      </div>

      <div class="bg-white rounded-lg shadow p-4">
        <div class="text-sm font-medium text-gray-500">Connections</div>
        <div class="text-2xl font-bold text-purple-600">{@connection_status.current_connections}</div>
        <div class="text-xs text-gray-400">Per Minute</div>
      </div>
    </div>

    <!-- Dashboard Statistics Cards -->
    <div class="dashboard-summary grid grid-cols-1 md:grid-cols-3 lg:grid-cols-5 gap-6 mb-6">
      <div
        class="bg-white rounded-lg shadow-md p-6 border-l-4 border-blue-500 cursor-pointer hover:shadow-lg transition-shadow duration-200"
        phx-click="navigate_to_devices"
      >
        <div class="flex items-center">
          <div class="flex-shrink-0">
            <div class="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
              <svg class="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path
                  stroke-linecap="round"
                  stroke-linejoin="round"
                  stroke-width="2"
                  d="M9.75 17L9 20l-1 1h8l-1-1-.75-3M3 13h18M5 17h14a2 2 0 002-2V5a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"
                />
              </svg>
            </div>
          </div>
          <div class="ml-4 flex-1">
            <div class="text-sm font-bold text-gray-700 mb-1">Total Devices</div>
            <div class="text-3xl font-bold text-gray-900">{@stats.total_devices}</div>
            <div class="text-xs text-gray-500">Registered devices</div>
          </div>
        </div>
      </div>

      <div
        class="bg-white rounded-lg shadow-md p-6 border-l-4 border-green-500 cursor-pointer hover:shadow-lg transition-shadow duration-200"
        phx-click="navigate_to_new_devices"
      >
        <div class="flex items-center">
          <div class="flex-shrink-0">
            <div class="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center">
              <svg
                class="w-6 h-6 text-green-600"
                fill="none"
                stroke="currentColor"
                viewBox="0 0 24 24"
              >
                <path
                  stroke-linecap="round"
                  stroke-linejoin="round"
                  stroke-width="2"
                  d="M12 6v6m0 0v6m0-6h6m-6 0H6"
                />
              </svg>
            </div>
          </div>
          <div class="ml-4 flex-1">
            <div class="text-sm font-bold text-gray-700 mb-1">New Devices</div>
            <div class="text-3xl font-bold text-gray-900">{@stats.new_devices_today}</div>
            <div class="text-xs text-green-600">Added today</div>
          </div>
        </div>
      </div>

      <div
        class="bg-white rounded-lg shadow-md p-6 border-l-4 border-purple-500 cursor-pointer hover:shadow-lg transition-shadow duration-200"
        phx-click="navigate_to_activated_devices"
      >
        <div class="flex items-center">
          <div class="flex-shrink-0">
            <div class="w-12 h-12 bg-purple-100 rounded-lg flex items-center justify-center">
              <svg
                class="w-6 h-6 text-purple-600"
                fill="none"
                stroke="currentColor"
                viewBox="0 0 24 24"
              >
                <path
                  stroke-linecap="round"
                  stroke-linejoin="round"
                  stroke-width="2"
                  d="M13 10V3L4 14h7v7l9-11h-7z"
                />
              </svg>
            </div>
          </div>
          <div class="ml-4 flex-1">
            <div class="text-sm font-bold text-gray-700 mb-1">Activated</div>
            <div class="text-3xl font-bold text-gray-900">{@stats.activated_devices}</div>
            <div class="text-xs text-purple-600">Active today</div>
          </div>
        </div>
      </div>

      <div
        class="bg-white rounded-lg shadow-md p-6 border-l-4 border-orange-500 cursor-pointer hover:shadow-lg transition-shadow duration-200"
        phx-click="navigate_to_upgrades"
      >
        <div class="flex items-center">
          <div class="flex-shrink-0">
            <div class="w-12 h-12 bg-orange-100 rounded-lg flex items-center justify-center">
              <svg
                class="w-6 h-6 text-orange-600"
                fill="none"
                stroke="currentColor"
                viewBox="0 0 24 24"
              >
                <path
                  stroke-linecap="round"
                  stroke-linejoin="round"
                  stroke-width="2"
                  d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M9 19l3 3m0 0l3-3m-3 3V10"
                />
              </svg>
            </div>
          </div>
          <div class="ml-4 flex-1">
            <div class="text-sm font-bold text-gray-700 mb-1">Upgrades</div>
            <div class="text-3xl font-bold text-gray-900">{@stats.upgrades_today}</div>
            <div class="text-xs text-orange-600">Completed today</div>
          </div>
        </div>
      </div>

      <div
        class="bg-white rounded-lg shadow-md p-6 border-l-4 border-indigo-500 cursor-pointer hover:shadow-lg transition-shadow duration-200"
        phx-click="navigate_to_app_downloads"
      >
        <div class="flex items-center">
          <div class="flex-shrink-0">
            <div class="w-12 h-12 bg-indigo-100 rounded-lg flex items-center justify-center">
              <svg
                class="w-6 h-6 text-indigo-600"
                fill="none"
                stroke="currentColor"
                viewBox="0 0 24 24"
              >
                <path
                  stroke-linecap="round"
                  stroke-linejoin="round"
                  stroke-width="2"
                  d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"
                />
              </svg>
            </div>
          </div>
          <div class="ml-4 flex-1">
            <div class="text-sm font-bold text-gray-700 mb-1">App Downloads</div>
            <div class="text-3xl font-bold text-gray-900">{@stats.app_downloads_today}</div>
            <div class="text-xs text-indigo-600">Downloaded today</div>
          </div>
        </div>
      </div>
    </div>
    <div class="flex gap-4 mb-6">
      <div class="w-2/3 bg-white rounded shadow p-4">
        <div class="flex items-center justify-between mb-2">
          <h2 class="text-lg font-semibold">Update statistics</h2>
          <button
            phx-click="refresh_update_statistics"
            class="bg-blue-600 text-white px-3 py-1.5 rounded text-sm hover:bg-blue-700 transition-colors flex items-center"
          >
            <svg class="w-4 h-4 mr-1.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path
                stroke-linecap="round"
                stroke-linejoin="round"
                stroke-width="2"
                d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"
              >
              </path>
            </svg>
            Update
          </button>
        </div>
        <div class="flex gap-2 mb-2">
          <button
            class={"tab-btn #{if @selected_period == "today", do: "active", else: ""}"}
            phx-click="set_chart_range"
            phx-value-range="today"
          >
            today
          </button>
          <button
            class={"tab-btn #{if @selected_period == "week", do: "active", else: ""}"}
            phx-click="set_chart_range"
            phx-value-range="week"
          >
            week
          </button>
          <button
            class={"tab-btn #{if @selected_period == "month", do: "active", else: ""}"}
            phx-click="set_chart_range"
            phx-value-range="month"
          >
            month
          </button>
        </div>
        <div
          id="updateStatsChart"
          phx-hook="UpdateStatsChartHook"
          data-chart-data={Jason.encode!(@chart_data)}
          class="h-64"
        >
        </div>
        <div class="flex gap-4 text-xs mt-2">
          <span class="text-blue-400">■ OTA updates</span>
          <span class="text-green-300">■ total number of APP downloads</span>
        </div>
      </div>
      <div class="w-1/3 bg-white rounded shadow p-4 flex flex-col items-center">
        <h2 class="text-lg font-semibold mb-2">The device is online Rate</h2>
        <div
          id="onlineRateGauge"
          phx-hook="OnlineRateGaugeHook"
          data-rate={@online_rate.online_rate}
          class="h-64 w-full"
        >
        </div>
        <div class="text-center mt-2 text-2xl font-bold">{@online_rate.online_rate}%</div>
        <div class="text-center text-sm">{@online_rate.online_devices} stand</div>
      </div>
    </div>
    <div class="ag-table-container">
      <div
        id="gridContainer"
        phx-hook="TerminalDashboardAgGridHook"
        phx-update="ignore"
        data-rows={Jason.encode!(@terminals)}
        class="ag-theme-alpine w-full h-[500px]"
      >
      </div>
    </div>
    """
  end

  def handle_event("set_chart_range", %{"range" => range}, socket) do
    chart_data = TerminalManagement.get_chart_data(range)
    stats = TerminalManagement.get_dashboard_statistics(range)

    {:noreply,
     assign(socket, :chart_data, chart_data)
     |> assign(:selected_period, range)
     |> assign(:stats, stats)}
  end

  def handle_event("refresh_update_statistics", _params, socket) do
    # Refresh all statistics and chart data based on the currently selected period
    stats = TerminalManagement.get_dashboard_statistics(socket.assigns.selected_period)
    chart_data = TerminalManagement.get_chart_data(socket.assigns.selected_period)
    online_rate = TerminalManagement.get_device_online_rate()

    socket =
      socket
      |> assign(:stats, stats)
      |> assign(:chart_data, chart_data)
      |> assign(:online_rate, online_rate)
      |> put_flash(:info, "Statistics updated successfully")

    {:noreply, socket}
  end

  # Navigation event handlers for dashboard cards
  def handle_event("navigate_to_devices", _params, socket) do
    {:noreply, push_navigate(socket, to: ~p"/terminals")}
  end

  def handle_event("navigate_to_new_devices", _params, socket) do
    # Navigate to terminals page with today's filter
    {:noreply, push_navigate(socket, to: ~p"/terminals?filter=today")}
  end

  def handle_event("navigate_to_activated_devices", _params, socket) do
    # Navigate to terminals page with active filter
    {:noreply, push_navigate(socket, to: ~p"/terminals?status=online")}
  end

  def handle_event("navigate_to_upgrades", _params, socket) do
    # Navigate to app upgrade status page
    {:noreply, push_navigate(socket, to: ~p"/appupgrade/status")}
  end

  def handle_event("navigate_to_app_downloads", _params, socket) do
    # Navigate to app packages page
    {:noreply, push_navigate(socket, to: ~p"/apppackage")}
  end

  # Phase 3.1: Real-time monitoring event handlers
  def handle_event("refresh_monitoring", _params, socket) do
    monitoring_data = TerminalManagement.get_realtime_monitoring_data()

    socket =
      socket
      |> assign(:monitoring_data, monitoring_data)
      |> assign(:device_health, monitoring_data.device_health)
      |> assign(:connection_status, monitoring_data.connection_status)
      |> assign(:performance_metrics, monitoring_data.performance_metrics)
      |> assign(:alert_summary, monitoring_data.alert_summary)

    {:noreply, socket}
  end

  def handle_event("toggle_monitoring_details", _params, socket) do
    {:noreply, assign(socket, :show_monitoring_details, !socket.assigns.show_monitoring_details)}
  end

  # Handle real-time PubSub messages
  def handle_info({:device_update, device_data}, socket) do
    # Update specific device in the terminals list
    updated_terminals = update_terminal_in_list(socket.assigns.terminals, device_data)
    {:noreply, assign(socket, :terminals, updated_terminals)}
  end

  def handle_info({:monitoring_update, monitoring_data}, socket) do
    socket =
      socket
      |> assign(:monitoring_data, monitoring_data)
      |> assign(:device_health, monitoring_data.device_health)
      |> assign(:connection_status, monitoring_data.connection_status)
      |> assign(:performance_metrics, monitoring_data.performance_metrics)
      |> assign(:alert_summary, monitoring_data.alert_summary)

    {:noreply, socket}
  end

  # Periodic updates
  def handle_info(:update_monitoring_data, socket) do
    monitoring_data = TerminalManagement.get_realtime_monitoring_data()

    socket =
      socket
      |> assign(:monitoring_data, monitoring_data)
      |> assign(:device_health, monitoring_data.device_health)
      |> assign(:connection_status, monitoring_data.connection_status)
      |> assign(:performance_metrics, monitoring_data.performance_metrics)
      |> assign(:alert_summary, monitoring_data.alert_summary)

    # Schedule next update
    Process.send_after(self(), :update_monitoring_data, 30_000)

    {:noreply, socket}
  end

  # Helper function to update terminal in list
  defp update_terminal_in_list(terminals, device_data) do
    Enum.map(terminals, fn terminal ->
      if terminal.serial_number == device_data.serial_number do
        Map.merge(terminal, device_data)
      else
        terminal
      end
    end)
  end

  # Helper function to get period description for UI labels
  defp period_description("today"), do: "today"
  defp period_description("week"), do: "this week"
  defp period_description("month"), do: "this month"
end
