defmodule DaProductAppWeb.ParameterManagementLive.Index do
  use DaProductAppWeb, :live_view
  alias DaProductApp.ParameterManagement
  alias DaProductApp.TerminalManagement

  def mount(_params, _session, socket) do
    {:ok,
     socket
     |> assign(:categories, list_categories())
     |> assign(:templates, list_templates())
     |> assign(:selected_category, nil)
     |> assign(:selected_template, nil)
     |> assign(:selected_template_for_params, nil)
     |> assign(:selected_template_info, nil)
     |> assign(:parameters, [])
     |> assign(:template_values, [])
     |> assign(:template_parameter_counts, get_template_parameter_counts())
     |> assign(:show_parameter_form, false)
     |> assign(:show_template_form, false)
     |> assign(:form_type, :new)
     |> assign(:editing_parameter, nil)
     |> assign(:current_page, "parameters")
     |> assign(:editing_template, nil)}
  end

  def handle_event("select_category", %{"category_id" => category_id}, socket) do
    category_id = String.to_integer(category_id)
    parameters = ParameterManagement.list_parameters_by_category(category_id)

    {:noreply,
     socket
     |> assign(:selected_category, category_id)
     |> assign(:parameters, parameters)}
  end

  def handle_event("select_template", %{"template_id" => template_id}, socket) do
    template_id = String.to_integer(template_id)
    template_values = ParameterManagement.list_template_values_by_template(template_id)

    {:noreply,
     socket
     |> assign(:selected_template, template_id)
     |> assign(:template_values, template_values)}
  end

  def handle_event("select_template_for_parameters", %{"template_id" => template_id}, socket) do
    template_id = String.to_integer(template_id)
    template = ParameterManagement.get_parameter_template!(template_id)
    template_values = ParameterManagement.list_template_values_by_template(template_id)

    {:noreply,
     socket
     |> assign(:selected_template_for_params, template_id)
     |> assign(:selected_template_info, template)
     |> assign(:template_values, template_values)}
  end

  def handle_event("show_parameter_form", %{"type" => "new"}, socket) do
    {:noreply,
     socket
     |> assign(:show_parameter_form, true)
     |> assign(:form_type, :new)
     |> assign(:editing_parameter, nil)}
  end

  def handle_event("show_parameter_form", %{"type" => "edit", "id" => id}, socket) do
    parameter = ParameterManagement.get_parameter_definition!(String.to_integer(id))

    {:noreply,
     socket
     |> assign(:show_parameter_form, true)
     |> assign(:form_type, :edit)
     |> assign(:editing_parameter, parameter)}
  end

  def handle_event("show_template_form", %{"type" => "new"}, socket) do
    {:noreply,
     socket
     |> assign(:show_template_form, true)
     |> assign(:form_type, :new)
     |> assign(:editing_template, nil)}
  end

  def handle_event("show_template_form", %{"type" => "edit", "id" => id}, socket) do
    template = ParameterManagement.get_parameter_template!(String.to_integer(id))

    {:noreply,
     socket
     |> assign(:show_template_form, true)
     |> assign(:form_type, :edit)
     |> assign(:editing_template, template)}
  end

  def handle_event("hide_forms", _params, socket) do
    {:noreply,
     socket
     |> assign(:show_parameter_form, false)
     |> assign(:show_template_form, false)}
  end

  def handle_event("delete_parameter", %{"id" => id}, socket) do
    parameter = ParameterManagement.get_parameter_definition!(String.to_integer(id))

    case ParameterManagement.delete_parameter_definition(parameter) do
      {:ok, _} ->
        parameters =
          if socket.assigns.selected_category do
            ParameterManagement.list_parameters_by_category(socket.assigns.selected_category)
          else
            []
          end

        {:noreply,
         socket
         |> put_flash(:info, "Parameter deleted successfully")
         |> assign(:parameters, parameters)}

      {:error, _changeset} ->
        {:noreply, put_flash(socket, :error, "Could not delete parameter")}
    end
  end

  def handle_event("delete_template", %{"id" => id}, socket) do
    template = ParameterManagement.get_parameter_template!(String.to_integer(id))

    case ParameterManagement.delete_parameter_template(template) do
      {:ok, _} ->
        {:noreply,
         socket
         |> put_flash(:info, "Template deleted successfully")
         |> assign(:templates, list_templates())}

      {:error, _changeset} ->
        {:noreply, put_flash(socket, :error, "Could not delete template")}
    end
  end

  def handle_info(:hide_forms, socket) do
    {:noreply,
     socket
     |> assign(:show_parameter_form, false)
     |> assign(:show_template_form, false)}
  end

  def handle_info({:parameter_saved, message}, socket) do
    parameters =
      if socket.assigns.selected_category do
        ParameterManagement.list_parameters_by_category(socket.assigns.selected_category)
      else
        []
      end

    {:noreply,
     socket
     |> put_flash(:info, message)
     |> assign(:show_parameter_form, false)
     |> assign(:parameters, parameters)}
  end

  def handle_info({:template_saved, message}, socket) do
    {:noreply,
     socket
     |> put_flash(:info, message)
     |> assign(:show_template_form, false)
     |> assign(:templates, list_templates())
     |> assign(:template_parameter_counts, get_template_parameter_counts())}
  end

  defp list_categories do
    ParameterManagement.list_parameter_categories()
  end

  defp list_templates do
    ParameterManagement.list_parameter_templates()
  end

  defp get_template_parameter_counts do
    templates = ParameterManagement.list_parameter_templates()

    Enum.reduce(templates, %{}, fn template, acc ->
      count = ParameterManagement.count_template_values_by_template(template.id)
      Map.put(acc, template.id, count)
    end)
  end

  defp get_template_value(template_values, parameter_id) do
    case Enum.find(template_values, &(&1.parameter_definition_id == parameter_id)) do
      nil -> nil
      %{value: value, is_overridden: is_overridden} -> {value, is_overridden}
    end
  end

  @impl true
  def handle_event("close_panel", _params, socket) do
    changeset =
      ParameterManagement.change_parameter_definition(%ParameterManagement.ParameterDefinition{})

    form = to_form(changeset)

    {:noreply,
     assign(socket,
       show_parameter_form: false,
       show_template_form: false,
       editing_parameter: nil,
       parameter_changeset: changeset,
       parameter_form: form
     )}
  end

  @impl true
  def handle_event("close_slide_over", _params, socket) do
    changeset =
      ParameterManagement.change_parameter_template(%ParameterManagement.ParameterTemplate{})

    form = to_form(changeset)

    {:noreply,
     assign(socket,
       show_parameter_form: false,
       show_template_form: false,
       editing_template: nil,
       template_changeset: changeset,
       template_form: form
     )}
  end
end
