defmodule DaProductAppWeb.ParameterManagementLive.ParameterFormComponent do
  use DaProductAppWeb, :live_component
  alias DaProductApp.ParameterManagement

  embed_templates "parameter_form_component/*"

  def update(assigns, socket) do
    parameter = assigns.parameter || %ParameterManagement.ParameterDefinition{}
    changeset = ParameterManagement.change_parameter_definition(parameter)
    form = to_form(changeset)

    {:ok,
     socket
     |> assign(assigns)
     |> assign(:changeset, changeset)
     |> assign(:form, form)
     |> assign(:parameter, parameter)}
  end

  def handle_event("validate", %{"parameter_definition" => params}, socket) do
    changeset =
      socket.assigns.parameter
      |> ParameterManagement.change_parameter_definition(params)
      |> Map.put(:action, :validate)

    form = to_form(changeset)

    {:noreply, assign(socket, changeset: changeset, form: form)}
  end

  def handle_event("save", %{"parameter_definition" => params}, socket) do
    case socket.assigns.form_type do
      :new -> create_parameter(socket, params)
      :edit -> update_parameter(socket, params)
    end
  end

  def handle_event("cancel", _params, socket) do
    send(self(), :hide_forms)
    {:noreply, socket}
  end

  defp create_parameter(socket, params) do
    case ParameterManagement.create_parameter_definition(params) do
      {:ok, _parameter} ->
        send(self(), {:parameter_saved, "Parameter created successfully"})
        {:noreply, socket}

      {:error, changeset} ->
        {:noreply, assign(socket, changeset: changeset, form: to_form(changeset))}
    end
  end

  defp update_parameter(socket, params) do
    case ParameterManagement.update_parameter_definition(socket.assigns.parameter, params) do
      {:ok, _parameter} ->
        send(self(), {:parameter_saved, "Parameter updated successfully"})
        {:noreply, socket}

      {:error, changeset} ->
        {:noreply, assign(socket, changeset: changeset, form: to_form(changeset))}
    end
  end

  def render(assigns), do: parameter_form(assigns)

  defp category_options(categories) do
    Enum.map(categories, fn category -> {category.name, category.id} end)
  end

  defp data_type_options do
    [
      {"String", "string"},
      {"Integer", "integer"},
      {"Boolean", "boolean"},
      {"Select", "select"},
      {"Multi Select", "multi_select"}
    ]
  end

  @impl true
  def handle_event("close_panel", _params, socket) do
    changeset =
      ParameterManagement.change_parameter_definition(%ParameterManagement.ParameterDefinition{})

    form = to_form(changeset)

    {:noreply,
     assign(socket,
       show_parameter_form: false,
       show_template_form: false,
       editing_parameter: nil,
       parameter_changeset: changeset,
       parameter_form: form
     )}
  end

  @impl true
  def handle_event("close_slide_over", _params, socket) do
    changeset =
      ParameterManagement.change_parameter_template(%ParameterManagement.ParameterTemplate{})

    form = to_form(changeset)

    {:noreply,
     assign(socket,
       show_parameter_form: false,
       show_template_form: false,
       editing_template: nil,
       template_changeset: changeset,
       template_form: form
     )}
  end
end
