defmodule DaProductAppWeb.SettlementLive.Index do
  use DaProductAppWeb, :live_view
  alias DaProductApp.Settlements.SettlementManagement

  @impl true
  def mount(_params, _session, socket) do
    summaries = SettlementManagement.list_settlement_summaries()

    {:ok,
     assign(socket,
       summaries: summaries,
       selected: nil,
       transactions: [],
       filter_status: nil,
       filter_provider: nil,
       current_page: "settlement",
       filter_date: nil
     )}
  end

  @impl true
  def handle_event(
        "show_transactions",
        %{"merchant_id" => merchant_id, "provider_id" => provider_id, "date" => date},
        socket
      ) do
    settlement =
      Enum.find(socket.assigns.summaries, fn s ->
        s.merchant_id == merchant_id && s.provider_id == String.to_integer(provider_id) &&
          to_string(s.date) == date
      end)

    transactions =
      if settlement do
        SettlementManagement.list_transactions_for_settlement(settlement.id)
      else
        []
      end

    {:noreply, assign(socket, selected: settlement, transactions: transactions)}
  end

  @impl true
  def handle_event(
        "filter",
        %{"status" => status, "provider" => provider, "date" => date},
        socket
      ) do
    summaries = SettlementManagement.list_settlement_summaries()

    filtered =
      summaries
      |> Enum.filter(fn s ->
        (status == "" or s.status == status) and
          (provider == "" or to_string(s.provider_id) == provider) and
          (date == "" or to_string(s.date) == date)
      end)

    {:noreply,
     assign(socket,
       summaries: filtered,
       filter_status: status,
       filter_provider: provider,
       filter_date: date
     )}
  end

  @impl true
  def handle_event("export_csv", _params, socket) do
    csv =
      [
        ["Merchant", "Provider", "Date", "Status", "Total Amount"]
        | Enum.map(socket.assigns.summaries, fn s ->
            [s.merchant_name, s.provider_name, s.date, s.status, s.total_amount]
          end)
      ]
      |> CSV.encode()
      |> Enum.join()

    {:noreply,
     push_event(socket, "download_csv", %{csv: csv, filename: "settlement_summaries.csv"})}
  end

  @impl true
  def handle_event("export_transactions_csv", _params, socket) do
    csv =
      [
        [
          "Transaction ID",
          "Terminal ID",
          "Amount",
          "Status",
          "Time",
          "MDR",
          "Tax",
          "Net Received"
        ]
        | Enum.map(socket.assigns.transactions, fn t ->
            [
              t.transaction_id,
              t.terminal_id,
              t.transaction_amount,
              t.transaction_status,
              t.transaction_time,
              t.mdr_charge,
              t.tax_on_mdr,
              t.net_received_amount
            ]
          end)
      ]
      |> CSV.encode()
      |> Enum.join()

    {:noreply,
     push_event(socket, "download_csv", %{csv: csv, filename: "settlement_transactions.csv"})}
  end
end
