defmodule DaProductAppWeb.TerminalGroupLive.Index do
  use DaProductAppWeb, :live_view
  require Logger
  alias DaProductApp.TerminalManagement

  @impl true
  def mount(_params, _session, socket) do
    if connected?(socket) do
      Phoenix.PubSub.subscribe(DaProductApp.PubSub, "tms:groups")
      Phoenix.PubSub.subscribe(DaProductApp.PubSub, "terminal:events")
    end

    groups = TerminalManagement.list_terminal_groups()
    statistics = TerminalManagement.get_group_statistics()

    {:ok,
     assign(socket,
       groups: groups,
       statistics: statistics,
       current_page: "terminal_groups",
       selected_group: nil,
       show_new_group_slider: false,
       show_edit_group_slider: false,
       show_rules_slider: false,
       show_assign_terminals_slider: false,
       new_group_form: %{},
       edit_group_form: %{},
       new_rule_form: %{},
       group_rules: [],
       group_terminals: [],
       available_terminals: [],
       filters: %{"group_type" => "all", "active_only" => true},
       errors: %{},
       applying_rules: false,
       last_event: nil,
       auto_events_enabled: true
     )}
  end

  @impl true
  def handle_params(params, _url, socket) do
    {:noreply, apply_action(socket, socket.assigns.live_action, params)}
  end

  defp apply_action(socket, :index, _params) do
    socket
    |> assign(:page_title, "Terminal Groups")
  end

  defp apply_action(socket, :show, %{"id" => id}) do
    case TerminalManagement.get_terminal_group(id) do
      {:ok, group} ->
        rules = TerminalManagement.list_group_rules(group.id)
        terminals = TerminalManagement.get_group_terminals(group.id)

        socket
        |> assign(:page_title, "Group: #{group.name}")
        |> assign(:selected_group, group)
        |> assign(:group_rules, rules)
        |> assign(:group_terminals, terminals)

      {:error, :not_found} ->
        socket
        |> put_flash(:error, "Group not found")
        |> redirect(to: ~p"/terminals/groups")
    end
  end

  @impl true
  def handle_event("filter", %{"filters" => filters}, socket) do
    groups = TerminalManagement.list_terminal_groups(filters)

    {:noreply,
     assign(socket,
       groups: groups,
       filters: filters
     )}
  end

  def handle_event("show_new_group", _params, socket) do
    {:noreply,
     assign(socket,
       show_new_group_slider: true,
       new_group_form: %{
         "name" => "",
         "description" => "",
         "group_type" => "custom",
         "color" => "#3B82F6",
         "icon" => "hero-squares-2x2"
       },
       errors: %{}
     )}
  end

  def handle_event("hide_new_group", _params, socket) do
    {:noreply,
     assign(socket,
       show_new_group_slider: false,
       new_group_form: %{},
       errors: %{}
     )}
  end

  def handle_event("update_new_group_form", %{"group" => group_params}, socket) do
    {:noreply, assign(socket, new_group_form: group_params)}
  end

  def handle_event("create_group", %{"group" => group_params}, socket) do
    # Replace with actual user
    attrs = Map.put(group_params, "created_by", "current_user")

    case TerminalManagement.create_terminal_group(attrs) do
      {:ok, _group} ->
        groups = TerminalManagement.list_terminal_groups(socket.assigns.filters)
        statistics = TerminalManagement.get_group_statistics()

        {:noreply,
         socket
         |> assign(groups: groups, statistics: statistics)
         |> assign(show_new_group_slider: false, new_group_form: %{}, errors: %{})
         |> put_flash(:info, "Group created successfully")}

      {:error, changeset} ->
        errors =
          changeset.errors
          |> Enum.into(%{}, fn {field, {message, _}} -> {Atom.to_string(field), message} end)

        {:noreply, assign(socket, errors: errors)}
    end
  end

  def handle_event("edit_group", %{"id" => id}, socket) do
    case TerminalManagement.get_terminal_group(id) do
      {:ok, group} ->
        form = %{
          "name" => group.name,
          "description" => group.description || "",
          "color" => group.color,
          "icon" => group.icon
        }

        {:noreply,
         assign(socket,
           selected_group: group,
           show_edit_group_slider: true,
           edit_group_form: form,
           errors: %{}
         )}

      {:error, :not_found} ->
        {:noreply, put_flash(socket, :error, "Group not found")}
    end
  end

  def handle_event("update_edit_group_form", %{"group" => group_params}, socket) do
    {:noreply, assign(socket, edit_group_form: group_params)}
  end

  def handle_event("update_group", %{"group" => group_params}, socket) do
    group = socket.assigns.selected_group

    case TerminalManagement.update_terminal_group(group, group_params) do
      {:ok, _updated_group} ->
        groups = TerminalManagement.list_terminal_groups(socket.assigns.filters)

        {:noreply,
         socket
         |> assign(groups: groups)
         |> assign(show_edit_group_slider: false, edit_group_form: %{}, errors: %{})
         |> put_flash(:info, "Group updated successfully")}

      {:error, changeset} ->
        errors =
          changeset.errors
          |> Enum.into(%{}, fn {field, {message, _}} -> {Atom.to_string(field), message} end)

        {:noreply, assign(socket, errors: errors)}
    end
  end

  def handle_event("delete_group", %{"id" => id}, socket) do
    case TerminalManagement.get_terminal_group(id) do
      {:ok, group} ->
        case TerminalManagement.delete_terminal_group(group) do
          {:ok, _} ->
            groups = TerminalManagement.list_terminal_groups(socket.assigns.filters)
            statistics = TerminalManagement.get_group_statistics()

            {:noreply,
             socket
             |> assign(groups: groups, statistics: statistics)
             |> put_flash(:info, "Group deleted successfully")}

          {:error, :cannot_delete_system_group} ->
            {:noreply, put_flash(socket, :error, "Cannot delete system groups")}

          {:error, _} ->
            {:noreply, put_flash(socket, :error, "Failed to delete group")}
        end

      {:error, :not_found} ->
        {:noreply, put_flash(socket, :error, "Group not found")}
    end
  end

  def handle_event("show_rules", %{"id" => id}, socket) do
    case TerminalManagement.get_terminal_group(id) do
      {:ok, group} ->
        rules = TerminalManagement.list_group_rules(group.id)

        {:noreply,
         assign(socket,
           selected_group: group,
           group_rules: rules,
           show_rules_slider: true,
           new_rule_form: %{
             "rule_name" => "",
             "rule_type" => "field_match",
             "field_name" => "vendor",
             "operator" => "equals",
             "value" => "",
             "priority" => "100"
           }
         )}

      {:error, :not_found} ->
        {:noreply, put_flash(socket, :error, "Group not found")}
    end
  end

  def handle_event("hide_rules", _params, socket) do
    {:noreply,
     assign(socket,
       show_rules_slider: false,
       selected_group: nil,
       group_rules: [],
       new_rule_form: %{}
     )}
  end

  def handle_event("update_rule_form", %{"rule" => rule_params}, socket) do
    {:noreply, assign(socket, new_rule_form: rule_params)}
  end

  def handle_event("apply_rule_template", %{"template" => template}, socket) do
    form =
      case template do
        "vendor_ingenico" ->
          %{
            "rule_name" => "Ingenico Terminals",
            "rule_type" => "field_match",
            "field_name" => "vendor",
            "operator" => "equals",
            "value" => "Ingenico",
            "priority" => "100"
          }

        "status_active" ->
          %{
            "rule_name" => "Active Terminals",
            "rule_type" => "field_match",
            "field_name" => "status",
            "operator" => "equals",
            "value" => "active",
            "priority" => "50"
          }

        "high_tier" ->
          %{
            "rule_name" => "High Tier Terminals",
            "rule_type" => "field_match",
            "field_name" => "tier",
            "operator" => "in",
            "value" => "premium,enterprise,high",
            "priority" => "75"
          }

        "production_deployment" ->
          %{
            "rule_name" => "Production Deployment",
            "rule_type" => "field_match",
            "field_name" => "deployment_type",
            "operator" => "equals",
            "value" => "production",
            "priority" => "25"
          }

        _ ->
          socket.assigns.new_rule_form
      end

    {:noreply, assign(socket, new_rule_form: form)}
  end

  def handle_event("create_rule", %{"rule" => rule_params}, socket) do
    group = socket.assigns.selected_group
    attrs = Map.put(rule_params, "group_id", group.id)

    # Validate rule before creating
    case validate_rule_params(rule_params) do
      {:ok, _} ->
        case TerminalManagement.create_group_rule(attrs) do
          {:ok, _rule} ->
            rules = TerminalManagement.list_group_rules(group.id)
            statistics = TerminalManagement.get_group_statistics()
            groups = TerminalManagement.list_terminal_groups(socket.assigns.filters)

            {:noreply,
             socket
             |> assign(group_rules: rules, statistics: statistics, groups: groups)
             |> assign(
               new_rule_form: %{
                 "rule_name" => "",
                 "rule_type" => "field_match",
                 "field_name" => "vendor",
                 "operator" => "equals",
                 "value" => "",
                 "priority" => "100"
               }
             )
             |> put_flash(:info, "Rule created and applied successfully")}

          {:error, changeset} ->
            errors =
              changeset.errors
              |> Enum.into(%{}, fn {field, {message, _}} -> {Atom.to_string(field), message} end)

            {:noreply, assign(socket, errors: errors)}
        end

      {:error, message} ->
        {:noreply, put_flash(socket, :error, message)}
    end
  end

  def handle_event("delete_rule", %{"id" => _rule_id}, socket) do
    # Implementation for deleting rules
    {:noreply, put_flash(socket, :info, "Rule deletion not implemented yet")}
  end

  def handle_event("apply_all_rules", _params, socket) do
    # Set loading state and then process in the background
    send(self(), :process_apply_rules)
    {:noreply, assign(socket, applying_rules: true)}
  end

  def handle_info(:process_apply_rules, socket) do
    start_time = System.monotonic_time(:millisecond)
    Logger.info("Starting rule application process for all terminals")

    case TerminalManagement.apply_all_rules() do
      {:ok, count} ->
        end_time = System.monotonic_time(:millisecond)
        duration = end_time - start_time

        Logger.info("Successfully applied #{count} rules to all terminals in #{duration}ms")

        # Refresh both statistics and groups list to update terminal counts
        statistics = TerminalManagement.get_group_statistics()
        groups = TerminalManagement.list_terminal_groups(socket.assigns.filters)

        {:noreply,
         socket
         |> assign(statistics: statistics, groups: groups, applying_rules: false)
         |> put_flash(:info, "Applied #{count} rules to all terminals (#{duration}ms)")}

      {:error, reason} ->
        Logger.error("Failed to apply rules: #{inspect(reason)}")

        {:noreply,
         socket
         |> assign(applying_rules: false)
         |> put_flash(:error, "Failed to apply rules - check logs for details")}
    end
  end

  def handle_info({:event_processed, event}, socket) do
    # Real-time update when rules are automatically applied
    groups = TerminalManagement.list_terminal_groups(socket.assigns.filters)
    statistics = TerminalManagement.get_group_statistics()

    {:noreply,
     socket
     |> assign(groups: groups, statistics: statistics, last_event: event)
     |> put_flash(:info, "✨ Auto-applied rules for #{event.type} event")}
  end

  def handle_info({:group_change, _event}, socket) do
    # Refresh data when groups change
    groups = TerminalManagement.list_terminal_groups(socket.assigns.filters)
    statistics = TerminalManagement.get_group_statistics()

    {:noreply, assign(socket, groups: groups, statistics: statistics)}
  end

  def handle_event("refresh_counts", _params, socket) do
    # Preserve filters when refreshing
    filters = socket.assigns.filters
    groups = TerminalManagement.list_terminal_groups(filters)
    statistics = TerminalManagement.get_group_statistics()

    {:noreply,
     socket
     |> assign(groups: groups, statistics: statistics, filters: filters)
     |> put_flash(:info, "Terminal counts refreshed")}
  end

  def handle_event("close_slide_over", _params, socket) do
    {:noreply,
     assign(socket,
       show_new_group_slider: false,
       show_edit_group_slider: false,
       show_rules_slider: false,
       show_assign_terminals_slider: false,
       new_group_form: %{},
       edit_group_form: %{},
       new_rule_form: %{},
       errors: %{},
       applying_rules: false
     )}
  end

  @impl true
  def render(assigns) do
    ~H"""
    <div class="p-6">
      <!-- Header -->
      <div class="mb-6">
        <div class="flex justify-between items-center">
          <div>
            <h1 class="text-2xl font-bold text-gray-900">Terminal Groups</h1>
            <p class="text-gray-600">Organize terminals into logical groups with automated rules</p>
          </div>

          <div class="flex space-x-3">
            <button
              phx-click="apply_all_rules"
              disabled={@applying_rules}
              class={"inline-flex items-center px-4 py-2 text-white text-sm font-medium rounded-md #{if @applying_rules, do: "bg-indigo-400 cursor-not-allowed", else: "bg-indigo-600 hover:bg-indigo-700"}"}
            >
              <%= if @applying_rules do %>
                <svg
                  class="animate-spin -ml-1 mr-2 h-4 w-4 text-white"
                  xmlns="http://www.w3.org/2000/svg"
                  fill="none"
                  viewBox="0 0 24 24"
                >
                  <circle
                    class="opacity-25"
                    cx="12"
                    cy="12"
                    r="10"
                    stroke="currentColor"
                    stroke-width="4"
                  >
                  </circle>
                  <path
                    class="opacity-75"
                    fill="currentColor"
                    d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                  >
                  </path>
                </svg>
                Applying Rules...
              <% else %>
                <.icon name="hero-arrow-path" class="w-4 h-4 mr-2" /> Apply All Rules
              <% end %>
            </button>

            <button
              phx-click="show_new_group"
              class="inline-flex items-center px-4 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700"
            >
              <.icon name="hero-plus" class="w-4 h-4 mr-2" /> New Group
            </button>

            <button
              phx-click="refresh_counts"
              class="inline-flex items-center px-4 py-2 bg-gray-600 text-white text-sm font-medium rounded-md hover:bg-gray-700"
              title="Refresh terminal counts"
            >
              <.icon name="hero-arrow-path" class="w-4 h-4 mr-2" /> Refresh
            </button>
          </div>
        </div>
        
    <!-- Statistics Cards -->
        <div class="mt-6 grid grid-cols-1 md:grid-cols-4 gap-4">
          <div class="bg-white p-4 rounded-lg shadow">
            <div class="text-sm text-gray-500">Total Groups</div>
            <div class="text-2xl font-bold text-gray-900">{@statistics.groups.total}</div>
          </div>

          <div class="bg-white p-4 rounded-lg shadow">
            <div class="text-sm text-gray-500">Active Rules</div>
            <div class="text-2xl font-bold text-green-600">{@statistics.rules.active}</div>
          </div>

          <div class="bg-white p-4 rounded-lg shadow">
            <div class="text-sm text-gray-500">Active Assignments</div>
            <div class="text-2xl font-bold text-blue-600">{@statistics.assignments.active}</div>
          </div>

          <div class="bg-white p-4 rounded-lg shadow">
            <div class="text-sm text-gray-500">Rule-based</div>
            <div class="text-2xl font-bold text-purple-600">
              {Map.get(@statistics.assignments.by_type, "rule_based", 0)}
            </div>
          </div>
        </div>
      </div>
      
    <!-- Filters -->
      <div class="mb-6 bg-white p-4 rounded-lg shadow">
        <form phx-change="filter">
          <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div>
              <label class="block text-sm font-medium text-gray-700 mb-1">Group Type</label>
              <select
                name="filters[group_type]"
                class="w-full border border-gray-300 rounded-md px-3 py-2"
              >
                <option value="all">All Types</option>
                <option value="custom" selected={@filters["group_type"] == "custom"}>Custom</option>
                <option value="auto" selected={@filters["group_type"] == "auto"}>Automatic</option>
                <option value="system" selected={@filters["group_type"] == "system"}>System</option>
              </select>
            </div>

            <div>
              <label class="block text-sm font-medium text-gray-700 mb-1">Status</label>
              <select
                name="filters[active_only]"
                class="w-full border border-gray-300 rounded-md px-3 py-2"
              >
                <option value={true} selected={@filters["active_only"]}>Active Only</option>
                <option value={false} selected={!@filters["active_only"]}>All Groups</option>
              </select>
            </div>

            <div>
              <label class="block text-sm font-medium text-gray-700 mb-1">Search</label>
              <input
                type="text"
                name="filters[search]"
                value={@filters["search"] || ""}
                placeholder="Search groups..."
                class="w-full border border-gray-300 rounded-md px-3 py-2"
              />
            </div>
          </div>
        </form>
      </div>
      
    <!-- Groups List -->
      <div class="bg-white shadow rounded-lg">
        <div class="overflow-x-auto">
          <table class="min-w-full divide-y divide-gray-200">
            <thead class="bg-gray-50">
              <tr>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Group
                </th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Type
                </th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Terminals
                </th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Rules
                </th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Status
                </th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Actions
                </th>
              </tr>
            </thead>
            <tbody class="bg-white divide-y divide-gray-200">
              <%= for group <- @groups do %>
                <tr class="hover:bg-gray-50">
                  <td class="px-6 py-4 whitespace-nowrap">
                    <div class="flex items-center">
                      <div
                        class="w-8 h-8 rounded-full flex items-center justify-center mr-3"
                        style={"background-color: #{group.color}20"}
                      >
                        <.icon name={group.icon} class="w-4 h-4" style={"color: #{group.color}"} />
                      </div>
                      <div>
                        <div class="text-sm font-medium text-gray-900">{group.name}</div>
                        <div class="text-sm text-gray-500">{group.description}</div>
                      </div>
                    </div>
                  </td>
                  <td class="px-6 py-4 whitespace-nowrap">
                    <span class={"inline-flex px-2 py-1 text-xs font-semibold rounded-full #{group_type_class(group.group_type)}"}>
                      {String.capitalize(group.group_type)}
                    </span>
                  </td>
                  <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    {group.terminal_count || 0}
                  </td>
                  <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    {length(group.group_rules || [])}
                  </td>
                  <td class="px-6 py-4 whitespace-nowrap">
                    <span class={"inline-flex px-2 py-1 text-xs font-semibold rounded-full #{if group.is_active, do: "bg-green-100 text-green-800", else: "bg-red-100 text-red-800"}"}>
                      {if group.is_active, do: "Active", else: "Inactive"}
                    </span>
                  </td>
                  <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                    <div class="flex space-x-2">
                      <button
                        phx-click="show_rules"
                        phx-value-id={group.id}
                        class="text-indigo-600 hover:text-indigo-900"
                      >
                        Rules
                      </button>

                      <%= unless group.group_type == "system" do %>
                        <button
                          phx-click="edit_group"
                          phx-value-id={group.id}
                          class="text-blue-600 hover:text-blue-900"
                        >
                          Edit
                        </button>

                        <button
                          phx-click="delete_group"
                          phx-value-id={group.id}
                          class="text-red-600 hover:text-red-900"
                          onclick="return confirm('Are you sure you want to delete this group?')"
                        >
                          Delete
                        </button>
                      <% end %>
                    </div>
                  </td>
                </tr>
              <% end %>
            </tbody>
          </table>
        </div>
      </div>
      
    <!-- New Group Slide-over -->
      <%= if @show_new_group_slider do %>
        <.slide_over origin="right" title="Create New Group">
          <div class="h-full flex flex-col">
            <div class="flex-1 overflow-y-auto p-6">
              <form phx-submit="create_group" phx-change="update_new_group_form" class="space-y-6">
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-2">
                    Name <span class="text-red-500">*</span>
                  </label>
                  <input
                    type="text"
                    name="group[name]"
                    value={@new_group_form["name"] || ""}
                    placeholder="Enter group name"
                    required
                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  />
                  <%= if @errors["name"] do %>
                    <p class="text-red-500 text-xs mt-1">{@errors["name"]}</p>
                  <% end %>
                </div>

                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-2">Description</label>
                  <textarea
                    name="group[description]"
                    value={@new_group_form["description"] || ""}
                    placeholder="Enter group description"
                    rows="3"
                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  ><%= @new_group_form["description"] || "" %></textarea>
                </div>

                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-2">Group Type</label>
                  <select
                    name="group[group_type]"
                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  >
                    <option value="custom" selected={@new_group_form["group_type"] == "custom"}>
                      Custom
                    </option>
                    <option value="auto" selected={@new_group_form["group_type"] == "auto"}>
                      Automatic
                    </option>
                  </select>
                </div>

                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-2">Color</label>
                  <input
                    type="color"
                    name="group[color]"
                    value={@new_group_form["color"] || "#3B82F6"}
                    class="w-full h-10 border border-gray-300 rounded-md"
                  />
                </div>

                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-2">Icon</label>
                  <select
                    name="group[icon]"
                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  >
                    <option
                      value="hero-squares-2x2"
                      selected={@new_group_form["icon"] == "hero-squares-2x2"}
                    >
                      Squares
                    </option>
                    <option
                      value="hero-computer-desktop"
                      selected={@new_group_form["icon"] == "hero-computer-desktop"}
                    >
                      Desktop
                    </option>
                    <option
                      value="hero-device-phone-mobile"
                      selected={@new_group_form["icon"] == "hero-device-phone-mobile"}
                    >
                      Mobile
                    </option>
                    <option value="hero-server" selected={@new_group_form["icon"] == "hero-server"}>
                      Server
                    </option>
                  </select>
                </div>

                <div class="flex justify-end space-x-3 pt-6 border-t">
                  <button
                    type="button"
                    phx-click="hide_new_group"
                    class="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50"
                  >
                    Cancel
                  </button>
                  <button
                    type="submit"
                    class="px-4 py-2 text-sm font-medium text-white bg-blue-600 border border-transparent rounded-md hover:bg-blue-700"
                  >
                    Create Group
                  </button>
                </div>
              </form>
            </div>
          </div>
        </.slide_over>
      <% end %>
      
    <!-- Edit Group Slide-over -->
      <%= if @show_edit_group_slider && @selected_group do %>
        <.slide_over origin="right" title="Edit Group">
          <div class="h-full flex flex-col">
            <div class="flex-1 overflow-y-auto p-6">
              <form phx-submit="update_group" phx-change="update_edit_group_form" class="space-y-6">
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-2">
                    Name <span class="text-red-500">*</span>
                  </label>
                  <input
                    type="text"
                    name="group[name]"
                    value={@edit_group_form["name"] || ""}
                    placeholder="Enter group name"
                    required
                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  />
                  <%= if @errors["name"] do %>
                    <p class="text-red-500 text-xs mt-1">{@errors["name"]}</p>
                  <% end %>
                </div>

                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-2">Description</label>
                  <textarea
                    name="group[description]"
                    value={@edit_group_form["description"] || ""}
                    placeholder="Enter group description"
                    rows="3"
                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  ><%= @edit_group_form["description"] || "" %></textarea>
                </div>

                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-2">Color</label>
                  <input
                    type="color"
                    name="group[color]"
                    value={@edit_group_form["color"] || "#3B82F6"}
                    class="w-full h-10 border border-gray-300 rounded-md"
                  />
                </div>

                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-2">Icon</label>
                  <select
                    name="group[icon]"
                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  >
                    <option
                      value="hero-squares-2x2"
                      selected={@edit_group_form["icon"] == "hero-squares-2x2"}
                    >
                      Squares
                    </option>
                    <option
                      value="hero-computer-desktop"
                      selected={@edit_group_form["icon"] == "hero-computer-desktop"}
                    >
                      Desktop
                    </option>
                    <option
                      value="hero-device-phone-mobile"
                      selected={@edit_group_form["icon"] == "hero-device-phone-mobile"}
                    >
                      Mobile
                    </option>
                    <option value="hero-server" selected={@edit_group_form["icon"] == "hero-server"}>
                      Server
                    </option>
                  </select>
                </div>

                <div class="flex justify-end space-x-3 pt-6 border-t">
                  <button
                    type="button"
                    phx-click="close_slide_over"
                    class="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50"
                  >
                    Cancel
                  </button>
                  <button
                    type="submit"
                    class="px-4 py-2 text-sm font-medium text-white bg-blue-600 border border-transparent rounded-md hover:bg-blue-700"
                  >
                    Update Group
                  </button>
                </div>
              </form>
            </div>
          </div>
        </.slide_over>
      <% end %>
      
    <!-- Rules Management Slide-over -->
      <%= if @show_rules_slider && @selected_group do %>
        <.slide_over origin="right" title={"Rules for #{@selected_group.name}"}>
          <div class="h-full flex flex-col">
            <div class="flex-1 overflow-y-auto p-6">
              
    <!-- Existing Rules -->
              <div class="mb-8">
                <h4 class="text-lg font-medium text-gray-900 mb-4">Existing Rules</h4>
                <%= if length(@group_rules) > 0 do %>
                  <div class="space-y-3">
                    <%= for rule <- @group_rules do %>
                      <div class="bg-gray-50 border border-gray-200 rounded-lg p-4">
                        <div class="flex justify-between items-start">
                          <div>
                            <h5 class="font-medium text-gray-900">{rule.rule_name}</h5>
                            <p class="text-sm text-gray-600">
                              {String.capitalize(rule.rule_type)} - {rule.field_name} {rule.operator} "{rule.value}"
                            </p>
                            <p class="text-xs text-gray-500">Priority: {rule.priority}</p>
                          </div>
                          <button
                            phx-click="delete_rule"
                            phx-value-id={rule.id}
                            class="text-red-600 hover:text-red-900 text-sm"
                          >
                            Delete
                          </button>
                        </div>
                      </div>
                    <% end %>
                  </div>
                <% else %>
                  <p class="text-gray-500 text-center py-8">No rules defined yet</p>
                <% end %>
              </div>
              
    <!-- Create New Rule Form -->
              <%= if @selected_group.group_type in ["auto", "custom"] do %>
                <div class="border-t pt-6">
                  <h4 class="text-lg font-medium text-gray-900 mb-4">Add New Rule</h4>
                  
    <!-- Rule Templates -->
                  <div class="mb-6">
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                      Quick Templates
                    </label>
                    <div class="grid grid-cols-2 gap-2 mb-4">
                      <button
                        type="button"
                        phx-click="apply_rule_template"
                        phx-value-template="vendor_ingenico"
                        class="text-left px-3 py-2 text-sm bg-blue-50 border border-blue-200 rounded-md hover:bg-blue-100"
                      >
                        Ingenico Terminals
                      </button>
                      <button
                        type="button"
                        phx-click="apply_rule_template"
                        phx-value-template="status_active"
                        class="text-left px-3 py-2 text-sm bg-green-50 border border-green-200 rounded-md hover:bg-green-100"
                      >
                        Active Status
                      </button>
                      <button
                        type="button"
                        phx-click="apply_rule_template"
                        phx-value-template="high_tier"
                        class="text-left px-3 py-2 text-sm bg-purple-50 border border-purple-200 rounded-md hover:bg-purple-100"
                      >
                        High Tier Terminals
                      </button>
                      <button
                        type="button"
                        phx-click="apply_rule_template"
                        phx-value-template="production_deployment"
                        class="text-left px-3 py-2 text-sm bg-orange-50 border border-orange-200 rounded-md hover:bg-orange-100"
                      >
                        Production Only
                      </button>
                    </div>
                  </div>

                  <form phx-submit="create_rule" phx-change="update_rule_form" class="space-y-4">
                    <div>
                      <label class="block text-sm font-medium text-gray-700 mb-1">Rule Name</label>
                      <input
                        type="text"
                        name="rule[rule_name]"
                        value={@new_rule_form["rule_name"] || ""}
                        placeholder="Enter rule name"
                        class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>

                    <div>
                      <label class="block text-sm font-medium text-gray-700 mb-1">Rule Type</label>
                      <select
                        name="rule[rule_type]"
                        class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      >
                        <option
                          value="field_match"
                          selected={@new_rule_form["rule_type"] == "field_match"}
                        >
                          Field Match
                        </option>
                        <option
                          value="merchant_type"
                          selected={@new_rule_form["rule_type"] == "merchant_type"}
                        >
                          Merchant Type
                        </option>
                        <option value="location" selected={@new_rule_form["rule_type"] == "location"}>
                          Location
                        </option>
                      </select>
                    </div>

                    <div>
                      <label class="block text-sm font-medium text-gray-700 mb-1">Field Name</label>
                      <select
                        name="rule[field_name]"
                        class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      >
                        <option value="vendor" selected={@new_rule_form["field_name"] == "vendor"}>
                          Vendor
                        </option>
                        <option value="model" selected={@new_rule_form["field_name"] == "model"}>
                          Model
                        </option>
                        <option value="status" selected={@new_rule_form["field_name"] == "status"}>
                          Status
                        </option>
                        <option value="area" selected={@new_rule_form["field_name"] == "area"}>
                          Area/Region
                        </option>
                        <option
                          value="location_code"
                          selected={@new_rule_form["field_name"] == "location_code"}
                        >
                          Location Code
                        </option>
                        <option
                          value="merchant_id"
                          selected={@new_rule_form["field_name"] == "merchant_id"}
                        >
                          Merchant ID
                        </option>
                        <option
                          value="deployment_type"
                          selected={@new_rule_form["field_name"] == "deployment_type"}
                        >
                          Deployment Type
                        </option>
                        <option value="tier" selected={@new_rule_form["field_name"] == "tier"}>
                          Terminal Tier
                        </option>
                        <option
                          value="app_version"
                          selected={@new_rule_form["field_name"] == "app_version"}
                        >
                          App Version
                        </option>
                        <option
                          value="system_version"
                          selected={@new_rule_form["field_name"] == "system_version"}
                        >
                          System Version
                        </option>
                        <option
                          value="hardware_version"
                          selected={@new_rule_form["field_name"] == "hardware_version"}
                        >
                          Hardware Version
                        </option>
                      </select>
                    </div>

                    <div>
                      <label class="block text-sm font-medium text-gray-700 mb-1">Operator</label>
                      <select
                        name="rule[operator]"
                        class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      >
                        <option value="equals" selected={@new_rule_form["operator"] == "equals"}>
                          Equals
                        </option>
                        <option
                          value="not_equals"
                          selected={@new_rule_form["operator"] == "not_equals"}
                        >
                          Not Equals
                        </option>
                        <option value="contains" selected={@new_rule_form["operator"] == "contains"}>
                          Contains
                        </option>
                        <option
                          value="starts_with"
                          selected={@new_rule_form["operator"] == "starts_with"}
                        >
                          Starts With
                        </option>
                        <option value="ends_with" selected={@new_rule_form["operator"] == "ends_with"}>
                          Ends With
                        </option>
                        <option value="in" selected={@new_rule_form["operator"] == "in"}>
                          In List (comma-separated)
                        </option>
                        <option value="not_in" selected={@new_rule_form["operator"] == "not_in"}>
                          Not In List
                        </option>
                        <option value="regex" selected={@new_rule_form["operator"] == "regex"}>
                          Regex Pattern
                        </option>
                        <option value="range" selected={@new_rule_form["operator"] == "range"}>
                          Numeric Range (min,max)
                        </option>
                      </select>
                    </div>

                    <div>
                      <label class="block text-sm font-medium text-gray-700 mb-1">Value</label>
                      <input
                        type="text"
                        name="rule[value]"
                        value={@new_rule_form["value"] || ""}
                        placeholder={get_value_placeholder(@new_rule_form["operator"] || "equals")}
                        class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                      <p class="text-xs text-gray-500 mt-1">
                        {get_value_help_text(@new_rule_form["operator"] || "equals")}
                      </p>
                    </div>

                    <div>
                      <label class="block text-sm font-medium text-gray-700 mb-1">Priority</label>
                      <input
                        type="number"
                        name="rule[priority]"
                        value={@new_rule_form["priority"] || "100"}
                        min="1"
                        max="1000"
                        class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>

                    <div class="flex justify-end space-x-3 pt-4">
                      <button
                        type="button"
                        phx-click="hide_rules"
                        class="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50"
                      >
                        Close
                      </button>
                      <button
                        type="submit"
                        class="px-4 py-2 text-sm font-medium text-white bg-green-600 border border-transparent rounded-md hover:bg-green-700"
                      >
                        Add Rule
                      </button>
                    </div>
                  </form>
                </div>
              <% end %>
            </div>
          </div>
        </.slide_over>
      <% end %>
    </div>
    """
  end

  defp group_type_class("system"), do: "bg-gray-100 text-gray-800"
  defp group_type_class("auto"), do: "bg-blue-100 text-blue-800"
  defp group_type_class("custom"), do: "bg-green-100 text-green-800"
  defp group_type_class(_), do: "bg-gray-100 text-gray-800"

  defp get_value_placeholder(operator) do
    case operator do
      "equals" -> "e.g., Ingenico"
      "not_equals" -> "e.g., PAX"
      "contains" -> "e.g., POS"
      "starts_with" -> "e.g., NYC"
      "ends_with" -> "e.g., -DEV"
      "in" -> "e.g., Ingenico,PAX,Verifone"
      "not_in" -> "e.g., test,demo,staging"
      "regex" -> "e.g., ^[A-Z]{3}[0-9]{3}$"
      "range" -> "e.g., 1,100"
      _ -> "Enter value to match"
    end
  end

  defp get_value_help_text(operator) do
    case operator do
      "equals" -> "Exact match (case-sensitive)"
      "not_equals" -> "Must not equal this value"
      "contains" -> "Field must contain this text"
      "starts_with" -> "Field must start with this text"
      "ends_with" -> "Field must end with this text"
      "in" -> "Comma-separated list of values"
      "not_in" -> "Must not be in this comma-separated list"
      "regex" -> "Regular expression pattern"
      "range" -> "Numeric range: minimum,maximum"
      _ -> ""
    end
  end

  defp validate_rule_params(rule_params) do
    cond do
      is_nil(rule_params["rule_name"]) or String.trim(rule_params["rule_name"]) == "" ->
        {:error, "Rule name is required"}

      is_nil(rule_params["value"]) or String.trim(rule_params["value"]) == "" ->
        {:error, "Rule value is required"}

      rule_params["operator"] == "range" and not valid_range?(rule_params["value"]) ->
        {:error, "Range must be in format: min,max (e.g., 1,100)"}

      rule_params["operator"] == "regex" and not valid_regex?(rule_params["value"]) ->
        {:error, "Invalid regular expression pattern"}

      rule_params["operator"] in ["in", "not_in"] and not valid_list?(rule_params["value"]) ->
        {:error, "List values must be comma-separated (e.g., value1,value2,value3)"}

      true ->
        {:ok, rule_params}
    end
  end

  defp valid_range?(value) do
    case String.split(value, ",") do
      [min_str, max_str] ->
        case {Integer.parse(String.trim(min_str)), Integer.parse(String.trim(max_str))} do
          {{min_val, ""}, {max_val, ""}} when min_val <= max_val -> true
          _ -> false
        end

      _ ->
        false
    end
  end

  defp valid_regex?(value) do
    case Regex.compile(value) do
      {:ok, _} -> true
      {:error, _} -> false
    end
  end

  defp valid_list?(value) do
    values = String.split(value, ",") |> Enum.map(&String.trim/1) |> Enum.reject(&(&1 == ""))
    length(values) > 0
  end
end
