defmodule DaProductAppWeb.TerminalLive.Index do
  use DaProductAppWeb, :live_view
  require Logger
  alias DaProductApp.TerminalManagement

  # ETS-based caching for terminals list
  @ets_table :terminal_cache

  defp ensure_ets_table do
    if :ets.whereis(@ets_table) == :undefined do
      :ets.new(@ets_table, [:named_table, :public, read_concurrency: true])
    end
  end

  defp fetch_terminals(filters) do
    # Use the new group-aware terminal listing function when view=groups
    case Map.get(filters, "view") do
      "groups" ->
        # Return terminals grouped by groups for group view
        TerminalManagement.list_terminals_with_groups(filters)

      _ ->
        # Use existing filter-based listing for normal terminal view
        DaProductApp.TerminalManagement.list_terminals_with_filters(filters)
    end
  end

  @impl true
  def mount(_params, _session, socket) do
    if connected?(socket), do: Phoenix.PubSub.subscribe(DaProductApp.PubSub, "tms:terminals")
    ensure_ets_table()

    filters = %{
      "status" => "all",
      "device_sn" => nil,
      "area" => nil,
      "vendor" => nil,
      "group" => nil,
      "model" => nil
    }

    terminals = fetch_terminals(filters)
    app_packages = DaProductApp.TerminalManagement.list_app_packages()

    # Get terminal groups for dropdown
    terminal_groups = TerminalManagement.list_terminal_groups(%{"active_only" => true})

    # Get dynamic filter options from high-performance cache
    filter_options = TerminalManagement.get_filter_options()

    # Push initial data to AG Grid after mounting
    if connected?(socket) do
      Process.send_after(self(), :push_initial_data, 100)
    end

    {:ok,
     assign(socket,
       terminals: terminals,
       current_page: "terminals",
       filters: filters,
       selected_terminal: nil,
       terminal_logs: [],
       show_panel: false,
       show_new_terminal: false,
       new_terminal_form: %{},
       new_terminal_errors: %{},
       page: 1,
       per_page: 10,
       total: length(terminals),
       config_form: %{},
       app_packages: app_packages,
       tab: "details",
       edit_mode: false,
       terminal_form: %{},
       terminal_groups: terminal_groups,
       filter_options: filter_options,
       # Remote control state
       remote_session_id: nil,
       remote_connected: false,
       remote_logging: false,
       remote_mode: nil,
       remote_logs: [],
       remote_log_count: 0,
       save_success: false
     )}
  end

  @impl true
  def handle_params(%{"serial_number" => serial}, _uri, socket) do
    terminals = fetch_terminals(socket.assigns.filters || %{"status" => "all"})
    terminal = Enum.find(terminals, &(&1.serial_number == serial))
    logs = if terminal, do: TerminalManagement.list_status_logs(terminal.id), else: []

    {:noreply,
     assign(socket,
       terminals: terminals,
       selected_terminal: terminal,
       terminal_logs: logs,
       show_new_terminal: false
     )}
  end

  @impl true
  def handle_params(params, _uri, %{assigns: %{live_action: :new}} = socket) do
    # Extract view parameter and update filters
    view = Map.get(params, "view", "all")
    filters = Map.put(socket.assigns.filters || %{"status" => "all"}, "view", view)

    terminals = fetch_terminals(filters)

    new_terminal_form = %{
      "serial_number" => "",
      "oid" => "",
      "area" => "",
      "vendor" => "",
      "model" => "",
      "status" => "offline",
      "remarks" => "",
      "imei" => "",
      "imei2" => "",
      "group" => "",
      "cpuid" => "",
      "mac" => "",
      "app_version" => "",
      "data_version" => "",
      "system_version" => ""
    }

    {:noreply,
     assign(socket,
       terminals: terminals,
       selected_terminal: nil,
       terminal_logs: [],
       show_new_terminal: true,
       new_terminal_form: new_terminal_form,
       new_terminal_errors: %{},
       filters: filters
     )}
  end

  @impl true
  def handle_params(params, _uri, socket) do
    # Extract view parameter and update filters
    view = Map.get(params, "view", "all")
    filters = Map.put(socket.assigns.filters || %{"status" => "all"}, "view", view)

    terminals = fetch_terminals(filters)

    {:noreply,
     assign(socket,
       terminals: terminals,
       selected_terminal: nil,
       terminal_logs: [],
       show_new_terminal: false,
       filters: filters
     )}
  end

  @impl true
  def handle_event("show_terminal_details", %{"serial_number" => serial}, socket) do
    # Find the terminal from the existing terminals list to maintain the same structure
    terminal = Enum.find(socket.assigns.terminals, &(&1.serial_number == serial))

    logs =
      if terminal, do: DaProductApp.TerminalManagement.list_status_logs(terminal.id), else: []

    # Initialize the form with terminal data, safely accessing fields that might not exist
    terminal_form =
      if terminal do
        %{
          "serial_number" => terminal.serial_number || "",
          "oid" => terminal.oid || "",
          "area" => Map.get(terminal, :area) || "",
          "vendor" => Map.get(terminal, :vendor) || "",
          "model" => Map.get(terminal, :model) || "",
          "status" => terminal.status || "",
          # Note: database field is 'remark', form uses 'remarks'
          "remarks" => Map.get(terminal, :remark) || "",
          "imei" => Map.get(terminal, :imei) || "",
          "group" => Map.get(terminal, :group) || ""
        }
      else
        %{}
      end

    {:noreply,
     assign(socket,
       selected_terminal: terminal,
       terminal_logs: logs,
       tab: "details",
       edit_mode: false,
       terminal_form: terminal_form
     )}
  end

  @impl true
  def handle_event("set_tab", %{"tab" => tab}, socket) do
    cond do
      tab == "location" ->
        location =
          case socket.assigns.selected_terminal do
            nil -> nil
            terminal -> TerminalManagement.get_latest_terminal_location(terminal.id)
          end

        {:noreply, assign(socket, tab: "location", location: location)}

      tab == "history" ->
        socket =
          assign_new(socket, :history_filters, fn -> %{} end)
          |> assign_new(:history_rows, fn -> [] end)

        {:noreply, assign(socket, tab: tab)}

      true ->
        {:noreply, assign(socket, tab: tab)}
    end
  end

  @impl true
  def handle_event("close_panel", _params, socket) do
    {:noreply, assign(socket, selected_terminal: nil, edit_mode: false)}
  end

  @impl true
  def handle_event("close_slide_over", _params, socket) do
    {:noreply, push_patch(socket, to: "/terminals")}
  end

  @impl true
  def handle_event("close_new_terminal", _params, socket) do
    {:noreply, push_patch(socket, to: "/terminals")}
  end

  @impl true
  def handle_event("update_new_terminal_form", %{"new_terminal_form" => form_params}, socket) do
    new_terminal_form = Map.merge(socket.assigns.new_terminal_form, form_params)
    {:noreply, assign(socket, new_terminal_form: new_terminal_form, new_terminal_errors: %{})}
  end

  @impl true
  def handle_event("update_new_terminal_form", params, socket) do
    # Handle direct parameter updates (fallback)
    new_terminal_form = Map.merge(socket.assigns.new_terminal_form, params)
    {:noreply, assign(socket, new_terminal_form: new_terminal_form, new_terminal_errors: %{})}
  end

  @impl true
  def handle_event("create_terminal", _params, socket) do
    # Convert form data to appropriate types and prepare for insertion
    terminal_attrs =
      socket.assigns.new_terminal_form
      |> Enum.reject(fn {_k, v} -> is_nil(v) or v == "" end)
      |> Map.new()
      |> Enum.into(%{}, fn {k, v} ->
        # Convert string keys to atoms and handle special cases
        case k do
          # Map form field 'remarks' to database field 'remark'
          "remarks" -> {:remark, v}
          _ -> {String.to_atom(k), v}
        end
      end)

    case DaProductApp.TerminalManagement.create_terminal(terminal_attrs) do
      {:ok, terminal} ->
        # Refresh the terminals list to include the new terminal with current filters
        terminals = fetch_terminals(socket.assigns.filters)

        # Push the new terminal to AG Grid
        Phoenix.LiveView.push_event(socket, "add_terminal_row", %{row: terminal})

        {:noreply,
         socket
         |> put_flash(:info, "Terminal created successfully")
         |> assign(:terminals, terminals)
         |> push_patch(to: "/terminals")}

      {:error, changeset} ->
        errors =
          changeset.errors
          |> Enum.into(%{}, fn {field, {message, _}} ->
            {to_string(field), message}
          end)

        {:noreply, assign(socket, new_terminal_errors: errors)}
    end
  end

  @impl true
  def handle_event("toggle_edit_mode", _params, socket) do
    {:noreply, assign(socket, edit_mode: !socket.assigns.edit_mode)}
  end

  @impl true
  def handle_event("update_terminal_form", %{"terminal_form" => form_params}, socket) do
    terminal_form = Map.merge(socket.assigns.terminal_form, form_params)
    {:noreply, assign(socket, terminal_form: terminal_form)}
  end

  @impl true
  def handle_event("update_terminal_form", params, socket) do
    # Handle direct parameter updates (fallback)
    terminal_form = Map.merge(socket.assigns.terminal_form, params)
    {:noreply, assign(socket, terminal_form: terminal_form)}
  end

  @impl true
  def handle_event("save_terminal", _params, socket) do
    case socket.assigns.selected_terminal do
      nil ->
        {:noreply, put_flash(socket, :error, "No terminal selected")}

      terminal ->
        # Extract only the editable fields from the form and remove empty values
        editable_fields = ["area", "vendor", "model", "group", "imei", "remarks", "oid"]

        terminal_attrs =
          socket.assigns.terminal_form
          |> Map.take(editable_fields)
          |> Enum.reject(fn {_k, v} -> is_nil(v) or v == "" end)
          |> Map.new()
          |> Enum.into(%{}, fn {k, v} ->
            # Map form field 'remarks' to database field 'remark'
            case k do
              "remarks" -> {:remark, v}
              _ -> {String.to_atom(k), v}
            end
          end)

        # Use the overloaded update_terminal function that handles maps
        case DaProductApp.TerminalManagement.update_terminal(terminal, terminal_attrs) do
          {:ok, updated_terminal} ->
            # Update the terminals list with the updated terminal using current filters
            updated_terminals = fetch_terminals(socket.assigns.filters)

            # Find the updated terminal in the new list format
            updated_selected = Enum.find(updated_terminals, &(&1.id == updated_terminal.id))

            # Push the update to AG Grid
            if updated_selected do
              Phoenix.LiveView.push_event(socket, "update_terminal_row", %{row: updated_selected})
            end

            {:noreply,
             assign(socket,
               terminals: updated_terminals,
               selected_terminal: updated_selected || terminal,
               edit_mode: false
             )
             |> put_flash(:info, "Terminal updated successfully")}

          {:error, changeset} ->
            error_msg =
              changeset.errors
              |> Enum.map(fn {field, {msg, _}} -> "#{field}: #{msg}" end)
              |> Enum.join(", ")

            {:noreply, put_flash(socket, :error, "Failed to update terminal: #{error_msg}")}

          {:error, :not_found} ->
            {:noreply, put_flash(socket, :error, "Terminal not found")}
        end
    end
  end

  @impl true
  def handle_event("cancel_edit", _params, socket) do
    # Reset form to original terminal data
    terminal_form =
      if socket.assigns.selected_terminal do
        terminal = socket.assigns.selected_terminal

        %{
          "serial_number" => terminal.serial_number || "",
          "oid" => terminal.oid || "",
          "area" => terminal.area || "",
          "vendor" => terminal.vendor || "",
          "model" => terminal.model || "",
          "status" => terminal.status || "",
          "remark" => terminal.remark || "",
          "imei" => terminal.imei || "",
          "group" => terminal.group || ""
        }
      else
        %{}
      end

    {:noreply, assign(socket, edit_mode: false, terminal_form: terminal_form)}
  end

  @impl true
  def handle_info(:push_initial_data, socket) do
    # Push initial terminal data to AG Grid after connection is established
    Phoenix.LiveView.push_event(socket, "update_terminals_data", %{rows: socket.assigns.terminals})

    {:noreply, socket}
  end

  @impl true
  def handle_info({:terminal_status_updated, _sn}, socket) do
    terminals = fetch_terminals(socket.assigns.filters)
    Logger.info("Updated terminals: #{inspect(terminals)}")

    selected_terminal =
      case socket.assigns.selected_terminal do
        nil ->
          nil

        %{serial_number: serial} ->
          Enum.find(terminals, &(&1.serial_number == serial))
      end

    terminal_logs =
      if selected_terminal,
        do: TerminalManagement.list_status_logs(selected_terminal.id),
        else: []

    # Find changed/inserted terminals and push only those updates to AG Grid
    old_terminals = Map.new(socket.assigns.terminals, &{&1.serial_number, &1})

    Enum.each(terminals, fn t ->
      old = Map.get(old_terminals, t.serial_number)

      if is_nil(old) or old != t do
        Phoenix.LiveView.push_event(socket, "update_terminal_row", %{row: t})
      end
    end)

    {:noreply,
     assign(socket,
       terminals: terminals,
       selected_terminal: selected_terminal,
       terminal_logs: terminal_logs
     )}
  end

  @impl true
  def handle_info({:terminal_status_updated, sn, %{itemkey: key, value: value}}, socket) do
    # Find and update the terminal in assigns.terminals
    terminals =
      Enum.map(socket.assigns.terminals, fn t ->
        if t.serial_number == sn do
          Map.put(t, String.to_atom(key), value)
        else
          t
        end
      end)

    # Push event to AG Grid for live update
    Phoenix.LiveView.push_event(socket, "update_terminal_metrics", %{
      serial_number: sn,
      metrics: %{key => value}
    })

    {:noreply, assign(socket, terminals: terminals)}
  end

  # Remote Log PubSub Handlers

  @impl true
  def handle_info({:remote_log_status, status}, socket) do
    {:noreply,
     assign(socket,
       remote_connected: status.connected,
       remote_logging: status.logging,
       remote_mode: status.mode,
       save_success: Map.get(socket.assigns, :save_success, false)
     )}
  end

  @impl true
  def handle_info({:remote_log_update, update}, socket) do
    {:noreply,
     assign(socket,
       remote_logs: update.logs,
       remote_log_count: update.count,
       save_success: Map.get(socket.assigns, :save_success, false)
     )}
  end

  @impl true
  def handle_event("filter", params, socket) do
    filters = Map.merge(socket.assigns.filters, params)
    terminals = TerminalManagement.list_terminals_with_filters(filters)

    # Track filter usage for intelligent sorting
    Enum.each(params, fn {filter_type, filter_value} ->
      if filter_value && filter_value != "" && filter_type in ["area", "vendor", "model"] do
        TerminalManagement.track_filter_usage(String.to_atom(filter_type), filter_value)
      end
    end)

    # Push filtered data to AG Grid
    Phoenix.LiveView.push_event(socket, "update_terminals_data", %{rows: terminals})

    {:noreply,
     assign(socket, terminals: terminals, filters: filters, page: 1, total: length(terminals))}
  end

  @impl true
  def handle_event("refresh_filters", _params, socket) do
    case TerminalManagement.refresh_filter_cache() do
      {:ok, _options} ->
        # Get updated filter options
        filter_options = TerminalManagement.get_filter_options()

        socket =
          socket
          |> assign(:filter_options, filter_options)
          |> put_flash(:info, "Filter options refreshed successfully!")

        {:noreply, socket}

      {:error, _reason} ->
        socket = put_flash(socket, :error, "Failed to refresh filter options. Please try again.")
        {:noreply, socket}
    end
  end

  @impl true
  def handle_event("set_status_tab", %{"status" => status}, socket) do
    filters = Map.put(socket.assigns.filters, "status", status)
    terminals = TerminalManagement.list_terminals_with_filters(filters)

    # Push filtered data to AG Grid
    Phoenix.LiveView.push_event(socket, "update_terminals_data", %{rows: terminals})

    {:noreply,
     assign(socket, terminals: terminals, filters: filters, page: 1, total: length(terminals))}
  end

  @impl true
  def handle_event("paginate", %{"page" => page}, socket) do
    {:noreply, assign(socket, page: String.to_integer(page))}
  end

  @impl true
  def handle_event("history_filter", params, socket) do
    filters = Map.merge(socket.assigns.history_filters || %{}, params)
    history_rows = list_terminal_history(socket.assigns.selected_terminal, filters)
    {:noreply, assign(socket, history_filters: filters, history_rows: history_rows)}
  end

  @impl true
  def handle_event("update_config_form", params, socket) do
    {:noreply, assign(socket, config_form: params)}
  end

  @impl true
  def handle_event("push_config", params, socket) do
    serial =
      params["terminal_id"] ||
        (socket.assigns.selected_terminal && socket.assigns.selected_terminal.serial_number)

    # Use the OtaService for consistent configuration pushing
    merchant_config = %{
      "merchant_id" => params["merchant_id"] || "900890089008000",
      "terminal_id" => serial,
      "mqtt_ip" => params["mqtt_ip"] || "testapp.ariticapp.com",
      "mqtt_port" => parse_int(params["mqtt_port"], 1883),
      "http_ip" => params["http_ip"] || "demo.ctrmv.com",
      "http_port" => parse_int(params["http_port"], 4001),
      "product_key" => params["product_key"] || "pFppbioOCKlo5c8E",
      "product_secret" => params["product_secret"] || "sj2AJl102397fQAV",
      "username" => params["username"] || "user001",
      "keepalive_time" => parse_int(params["keepalive_time"], 300),
      "play_language" => parse_int(params["play_language"], 1),
      "heartbeat_interval" => parse_int(params["heartbeat_interval"], 300)
    }

    case DaProductApp.TerminalManagement.OtaService.send_merchant_config_update(
           serial,
           merchant_config
         ) do
      {:ok, message} ->
        {:noreply, put_flash(socket, :info, message)}

      {:error, reason} ->
        {:noreply, put_flash(socket, :error, "Failed to push configuration: #{reason}")}
    end
  end

  @impl true
  def handle_event("push_app_package", params, socket) do
    serial = socket.assigns.selected_terminal && socket.assigns.selected_terminal.serial_number
    package_id = params["package_id"]

    case DaProductApp.TerminalManagement.AppPackageService.deploy_package_to_device(
           serial,
           package_id,
           nil
         ) do
      {:ok, message} ->
        {:noreply, put_flash(socket, :info, message)}

      {:error, reason} ->
        {:noreply, put_flash(socket, :error, "Failed to deploy package: #{reason}")}
    end
  end

  @impl true
  def handle_event("push_parameters", params, socket) do
    serial = socket.assigns.selected_terminal && socket.assigns.selected_terminal.serial_number

    case DaProductApp.TerminalManagement.ParameterPushService.send_terminal_parameters(
           serial,
           params["parameters"]
         ) do
      {:ok, message} ->
        {:noreply, put_flash(socket, :info, message)}

      {:error, reason} ->
        {:noreply, put_flash(socket, :error, "Failed to push parameters: #{reason}")}
    end
  end

  @impl true
  def handle_event("send_file_download", params, socket) do
    serial = socket.assigns.selected_terminal && socket.assigns.selected_terminal.serial_number

    case DaProductApp.TerminalManagement.FileDownloadService.send_file_download_command(
           serial,
           params["download_params"]
         ) do
      {:ok, message} ->
        {:noreply, put_flash(socket, :info, message)}

      {:error, reason} ->
        {:noreply, put_flash(socket, :error, "Failed to send file download: #{reason}")}
    end
  end

  # Remote Control Event Handlers

  @impl true
  def handle_event("remote_connect", _params, socket) do
    case socket.assigns.selected_terminal do
      nil ->
        {:noreply, put_flash(socket, :error, "No terminal selected")}

      terminal ->
        case DaProductApp.TerminalManagement.RemoteLogService.start_log_session(
               terminal.serial_number,
               socket.assigns[:current_user_id] || "user"
             ) do
          {:ok, session_id, _pid} ->
            # Subscribe to session updates
            Phoenix.PubSub.subscribe(DaProductApp.PubSub, "remote_log:#{session_id}")

            {:noreply,
             assign(socket,
               remote_session_id: session_id,
               remote_connected: true,
               save_success: false
             )
             |> put_flash(:info, "Connected to terminal #{terminal.serial_number}")}

          {:error, reason} ->
            {:noreply, put_flash(socket, :error, "Failed to connect: #{reason}")}
        end
    end
  end

  @impl true
  def handle_event("remote_disconnect", _params, socket) do
    if socket.assigns.remote_session_id do
      DaProductApp.TerminalManagement.RemoteLogService.stop_log_session(
        socket.assigns.remote_session_id
      )

      {:noreply,
       assign(socket,
         remote_session_id: nil,
         remote_connected: false,
         remote_logging: false,
         remote_mode: nil,
         remote_logs: [],
         remote_log_count: 0,
         save_success: false
       )
       |> put_flash(:info, "Disconnected from terminal")}
    else
      {:noreply, socket}
    end
  end

  @impl true
  def handle_event("remote_start_logging", %{"mode" => mode} = params, socket) do
    if socket.assigns.remote_session_id do
      opts =
        case mode do
          "WITH_DELAY" ->
            %{
              last_lines_count: parse_int(params["last_lines_count"], 10),
              frequency_send: parse_int(params["frequency_send"], 15),
              log_level: params["log_level"] || "ALL"
            }

          _ ->
            %{}
        end

      case DaProductApp.TerminalManagement.RemoteLogService.start_logging(
             socket.assigns.remote_session_id,
             mode,
             opts
           ) do
        {:ok, _request_id} ->
          {:noreply,
           assign(socket,
             remote_logging: true,
             remote_mode: mode,
             save_success: false
           )
           |> put_flash(:info, "Started logging in #{mode} mode")}

        {:error, reason} ->
          {:noreply, put_flash(socket, :error, "Failed to start logging: #{reason}")}
      end
    else
      {:noreply, put_flash(socket, :error, "Not connected to terminal")}
    end
  end

  @impl true
  def handle_event("remote_stop_logging", _params, socket) do
    if socket.assigns.remote_session_id do
      case DaProductApp.TerminalManagement.RemoteLogService.stop_logging(
             socket.assigns.remote_session_id
           ) do
        :ok ->
          {:noreply,
           assign(socket,
             remote_logging: false,
             remote_mode: nil,
             save_success: false
           )
           |> put_flash(:info, "Stopped logging")}

        {:error, reason} ->
          {:noreply, put_flash(socket, :error, "Failed to stop logging: #{reason}")}
      end
    else
      {:noreply, socket}
    end
  end

  @impl true
  def handle_event("remote_clear_logs", _params, socket) do
    if socket.assigns.remote_session_id do
      DaProductApp.TerminalManagement.RemoteLogService.clear_logs(
        socket.assigns.remote_session_id
      )

      {:noreply,
       assign(socket, remote_logs: [], remote_log_count: 0, save_success: false)
       |> put_flash(:info, "Logs cleared")}
    else
      {:noreply, socket}
    end
  end

  @impl true
  def handle_event("remote_save_logs", _params, socket) do
    if socket.assigns.remote_session_id && length(socket.assigns.remote_logs) > 0 do
      # Generate log file content
      timestamp = DateTime.utc_now() |> DateTime.to_string()
      terminal_serial = socket.assigns.selected_terminal.serial_number

      log_content =
        [
          "Terminal Debug Logs",
          "Terminal: #{terminal_serial}",
          "Generated: #{timestamp}",
          "Total Lines: #{socket.assigns.remote_log_count}",
          "=" |> String.duplicate(50),
          ""
        ] ++
          Enum.map(socket.assigns.remote_logs, fn log ->
            "#{log.timestamp} | #{log.message}"
          end)

      log_text = Enum.join(log_content, "\n")
      filename = "terminal_#{terminal_serial}_#{DateTime.utc_now() |> DateTime.to_date()}.txt"

      # Push download event to client (without flash message to prevent panel closing)
      # Set temporary success state that will auto-clear
      Process.send_after(self(), :clear_save_success, 2000)

      {:noreply,
       assign(socket, save_success: true)
       |> push_event("download_file", %{
         filename: filename,
         content: log_text,
         type: "text/plain"
       })}
    else
      {:noreply, socket}
    end
  end

  @impl true
  def handle_info(:clear_save_success, socket) do
    {:noreply, assign(socket, save_success: false)}
  end

  defp parse_int(nil, default), do: default
  defp parse_int(<<>>, default), do: default

  defp parse_int(val, default) when is_binary(val) do
    case Integer.parse(val) do
      {int, _} -> int
      _ -> default
    end
  end

  defp parse_int(val, _default) when is_integer(val), do: val

  defp list_terminal_history(nil, _filters), do: []

  defp list_terminal_history(terminal, filters) do
    import Ecto.Query

    base_query =
      from l in DaProductApp.TerminalManagement.TmsTerminalStatusLog,
        where: l.terminal_id == ^terminal.id,
        preload: [:status_items],
        order_by: [desc: l.inserted_at]

    query =
      if filters["app_version"] && filters["app_version"] != "" do
        from l in base_query,
          join: i in assoc(l, :status_items),
          where:
            i.itemkey == "app_version" and ilike(i.value, ^("%" <> filters["app_version"] <> "%")),
          preload: [status_items: i]
      else
        base_query
      end

    query =
      if filters["data_version"] && filters["data_version"] != "" do
        from l in query,
          join: i in assoc(l, :status_items),
          where:
            i.itemkey == "data_version" and
              ilike(i.value, ^("%" <> filters["data_version"] <> "%")),
          preload: [status_items: i]
      else
        query
      end

    logs = DaProductApp.Repo.all(query)
    # Flatten and map to table rows
    Enum.map(logs, fn log ->
      row = %{
        imei: get_status_item(log, "imei"),
        app_version: get_status_item(log, "app_version"),
        data_version: get_status_item(log, "data_version"),
        system_version: get_status_item(log, "system_version"),
        secure_firmware: get_status_item(log, "secure_firmware"),
        boot_version: get_status_item(log, "boot_version"),
        pubkey: get_status_item(log, "pubkey"),
        appkey: get_status_item(log, "appkey"),
        battery: get_status_item(log, "battery"),
        ip: get_status_item(log, "ip"),
        area: get_status_item(log, "area"),
        login_time: log.upload_time
      }

      row
    end)
  end

  defp get_status_item(log, key) do
    log.status_items
    |> Enum.find_value("", fn i -> if i.itemkey == key, do: i.value, else: nil end)
  end

  defp format_timestamp(timestamp) when is_binary(timestamp) do
    case DateTime.from_iso8601(timestamp) do
      {:ok, dt, _} ->
        dt
        |> DateTime.to_time()
        |> Time.to_string()
        # HH:MM:SS
        |> String.slice(0, 8)

      _ ->
        timestamp |> String.slice(0, 8)
    end
  end

  defp format_timestamp(_), do: "00:00:00"

  defp history_tab(assigns) do
    ~H"""
    <div class="p-2">
      <form phx-change="history_filter" phx-submit="history_filter">
        <div class="flex gap-4 mb-4">
          <.input
            name="app_version"
            value={@history_filters["app_version"]}
            placeholder="App version"
            phx-debounce="300"
          />
          <.input
            name="data_version"
            value={@history_filters["data_version"]}
            placeholder="Data version"
            phx-debounce="300"
          />
          <button type="submit" class="btn btn-primary">Search</button>
          <button
            type="button"
            class="btn btn-secondary"
            phx-click="history_filter"
            phx-value-reset="true"
          >
            Reset
          </button>
          <button type="button" class="btn btn-success" phx-click="history_export">Export</button>
        </div>
      </form>
      <div class="ag-table-container">
        <div
          id="historyGridContainer"
          phx-hook="TerminalHistoryAgGridHook"
          phx-update="ignore"
          data-rows={Jason.encode!(@history_rows)}
          class="ag-theme-alpine w-full h-[300px]"
        >
        </div>
      </div>
    </div>
    """
  end

  defp location_tab(assigns) do
    location = assigns[:location] || %{}
    # Default to Bangalore if no location is available
    default_lat = 12.9716
    default_lng = 77.5946
    has_location = location[:lat] && location[:lng]
    lat = (has_location && location[:lat]) || default_lat
    lng = (has_location && location[:lng]) || default_lng

    assigns = assign(assigns, :location, location)
    assigns = assign(assigns, :has_location, has_location)
    assigns = assign(assigns, :lat, lat)
    assigns = assign(assigns, :lng, lng)

    ~H"""
    <div class="p-2">
      <h3 class="text-lg font-semibold mb-2 text-blue-700">Location Info</h3>
      <div class="mb-2">
        <span class="font-semibold">Address:</span> {(@has_location && (@location[:address] || "N/A")) ||
          "Default: Bangalore"}<br />
        <span class="font-semibold">Coordinates:</span> {@lat}, {@lng}<br />
        <span class="font-semibold">Timestamp:</span> {(@has_location && @location[:timestamp]) ||
          "N/A"}
      </div>
      <div
        id="terminal-location-map"
        phx-hook="TerminalLocationMapHook"
        data-lat={@lat}
        data-lng={@lng}
        class="w-full h-64 rounded shadow border"
      >
      </div>
      <%= unless @has_location do %>
        <div class="text-gray-500 mt-2">
          No location data available for this terminal. Showing default (Bangalore).
        </div>
      <% end %>
    </div>
    """
  end

  defp apk_tab(assigns) do
    ~H"""
    <div class="p-2">
      <h3 class="text-lg font-semibold mb-2 text-blue-700">APK List</h3>
      <div class="text-gray-500">APK list and management UI will be shown here.</div>
    </div>
    """
  end

  defp traffic_tab(assigns) do
    ~H"""
    <div class="p-2">
      <h3 class="text-lg font-semibold mb-2 text-blue-700">App Traffic</h3>
      <div class="text-gray-500">App traffic statistics and charts will be shown here.</div>
    </div>
    """
  end

  # defp remote_tab(assigns) do
  #   render("remote_tab.html", assigns)
  # end
  # end

  defp remote_tab(assigns) do
    ~H"""
    <div
      id="remote-control-container"
      class="h-full flex flex-col p-6 space-y-6"
      phx-hook="RemoteControl"
    >
      <!-- Header Section -->
      <div class="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-lg p-6 border border-blue-200">
        <div class="flex items-center justify-between">
          <div>
            <h3 class="text-xl font-semibold text-blue-900 flex items-center">
              <svg
                class="w-6 h-6 mr-3 text-blue-600"
                fill="none"
                stroke="currentColor"
                viewBox="0 0 24 24"
              >
                <path
                  stroke-linecap="round"
                  stroke-linejoin="round"
                  stroke-width="2"
                  d="M12 18h.01M8 21h8a2 2 0 002-2V5a2 2 0 00-2-2H8a2 2 0 00-2 2v14a2 2 0 002 2z"
                >
                </path>
              </svg>
              Remote Debug Log Viewer
            </h3>
            <p class="text-blue-700 text-sm mt-1">Stream live debug logs from terminal over MQTT</p>
          </div>
          
    <!-- Connection Status -->
          <div class="flex items-center space-x-4">
            <div class="flex items-center">
              <div class={[
                "w-3 h-3 rounded-full mr-2",
                if(@remote_connected, do: "bg-green-500", else: "bg-gray-400")
              ]}>
              </div>
              <span class={[
                "text-sm font-medium",
                if(@remote_connected, do: "text-green-700", else: "text-gray-600")
              ]}>
                {if @remote_connected, do: "Connected", else: "Disconnected"}
              </span>
            </div>

            <%= if @remote_connected do %>
              <button
                phx-click="remote_disconnect"
                class="inline-flex items-center px-3 py-1.5 bg-red-600 hover:bg-red-700 text-white text-sm font-medium rounded-md transition-colors duration-200"
              >
                <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path
                    stroke-linecap="round"
                    stroke-linejoin="round"
                    stroke-width="2"
                    d="M6 18L18 6M6 6l12 12"
                  >
                  </path>
                </svg>
                Disconnect
              </button>
            <% else %>
              <button
                phx-click="remote_connect"
                class="inline-flex items-center px-3 py-1.5 bg-blue-600 hover:bg-blue-700 text-white text-sm font-medium rounded-md transition-colors duration-200"
              >
                <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path
                    stroke-linecap="round"
                    stroke-linejoin="round"
                    stroke-width="2"
                    d="M8.111 16.404a5.5 5.5 0 017.778 0M12 20h.01m-7.08-7.071c3.904-3.905 10.236-3.905 14.141 0M1.394 9.393c5.857-5.857 15.355-5.857 21.213 0"
                  >
                  </path>
                </svg>
                Connect
              </button>
            <% end %>
          </div>
        </div>
        
    <!-- Device Info -->
        <%= if @selected_terminal do %>
          <div class="mt-4 pt-4 border-t border-blue-200">
            <div class="grid grid-cols-2 gap-4 text-sm">
              <div>
                <span class="text-blue-600 font-medium">Serial Number:</span>
                <span class="ml-2 text-blue-900">{@selected_terminal.serial_number}</span>
              </div>
              <div>
                <span class="text-blue-600 font-medium">Product Key:</span>
                <span class="ml-2 text-blue-900 font-mono">pFppbioOCKlo5c8E</span>
              </div>
            </div>
          </div>
        <% end %>
      </div>
      
    <!-- Logging Controls -->
      <%= if @remote_connected do %>
        <div class="bg-white rounded-lg border border-gray-200 p-6">
          <h4 class="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <svg
              class="w-5 h-5 mr-2 text-purple-600"
              fill="none"
              stroke="currentColor"
              viewBox="0 0 24 24"
            >
              <path
                stroke-linecap="round"
                stroke-linejoin="round"
                stroke-width="2"
                d="M12 6V4m0 2a2 2 0 100 4m0-4a2 2 0 110 4m-6 8a2 2 0 100-4m0 4a2 2 0 100 4m0-4v2m0-6V4m6 6v10m6-2a2 2 0 100-4m0 4a2 2 0 100 4m0-4v2m0-6V4"
              >
              </path>
            </svg>
            Logging Mode
          </h4>

          <form phx-submit="remote_start_logging" class="space-y-4">
            <!-- Mode Selection -->
            <div class="space-y-3">
              <div class="flex items-center space-x-6">
                <label class="flex items-center">
                  <input
                    type="radio"
                    name="mode"
                    value="REALTIME"
                    checked={@remote_mode == "REALTIME" || !@remote_mode}
                    class="w-4 h-4 text-purple-600 border-gray-300 focus:ring-purple-500"
                  />
                  <span class="ml-2 text-sm font-medium text-gray-700">Realtime</span>
                  <span class="ml-2 text-xs text-gray-500">(Live streaming)</span>
                </label>

                <label class="flex items-center">
                  <input
                    type="radio"
                    name="mode"
                    value="WITH_DELAY"
                    checked={@remote_mode == "WITH_DELAY"}
                    class="w-4 h-4 text-purple-600 border-gray-300 focus:ring-purple-500"
                  />
                  <span class="ml-2 text-sm font-medium text-gray-700">With Delay</span>
                  <span class="ml-2 text-xs text-gray-500">(Batched with interval)</span>
                </label>
              </div>
              
    <!-- Delay Mode Options -->
              <div
                id="delay-options"
                class="pl-6 space-y-3 bg-gray-50 rounded-md p-4"
                style="display: none;"
              >
                <div class="grid grid-cols-3 gap-4">
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                      Last Lines Count
                    </label>
                    <input
                      type="number"
                      name="last_lines_count"
                      value="10"
                      min="1"
                      max="100"
                      class="w-full px-3 py-1.5 text-sm border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-purple-500"
                    />
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                      Frequency (seconds)
                    </label>
                    <input
                      type="number"
                      name="frequency_send"
                      value="15"
                      min="5"
                      max="300"
                      class="w-full px-3 py-1.5 text-sm border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-purple-500"
                    />
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Log Level</label>
                    <select
                      name="log_level"
                      class="w-full px-3 py-1.5 text-sm border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-purple-500"
                    >
                      <option value="ALL">All Levels</option>
                      <option value="ERROR">Error</option>
                      <option value="WARN">Warning</option>
                      <option value="INFO">Info</option>
                      <option value="DEBUG">Debug</option>
                    </select>
                  </div>
                </div>
              </div>
            </div>
            
    <!-- Action Buttons -->
            <div class="flex items-center justify-between pt-4 border-t border-gray-200">
              <div class="flex items-center space-x-3">
                <%= if @remote_logging do %>
                  <button
                    type="button"
                    phx-click="remote_stop_logging"
                    class="inline-flex items-center px-4 py-2 bg-red-600 hover:bg-red-700 text-white font-medium rounded-md transition-colors duration-200"
                  >
                    <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path
                        stroke-linecap="round"
                        stroke-linejoin="round"
                        stroke-width="2"
                        d="M6 18L18 6M6 6l12 12"
                      >
                      </path>
                    </svg>
                    Stop
                  </button>
                  <div class="flex items-center text-sm text-green-600">
                    <div class="w-2 h-2 bg-green-500 rounded-full mr-2 animate-pulse"></div>
                    Logging in {@remote_mode || "REALTIME"} mode
                  </div>
                <% else %>
                  <button
                    type="submit"
                    class="inline-flex items-center px-4 py-2 bg-green-600 hover:bg-green-700 text-white font-medium rounded-md transition-colors duration-200"
                  >
                    <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path
                        stroke-linecap="round"
                        stroke-linejoin="round"
                        stroke-width="2"
                        d="M14.828 14.828a4 4 0 01-5.656 0M9 10h1m4 0h1m-6 4h8m2 0V8a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2h12a2 2 0 002-2z"
                      >
                      </path>
                    </svg>
                    Start Logging
                  </button>
                <% end %>
              </div>

              <div class="flex items-center space-x-2">
                <button
                  type="button"
                  phx-click="remote_clear_logs"
                  class="inline-flex items-center px-3 py-1.5 bg-gray-600 hover:bg-gray-700 text-white text-sm font-medium rounded-md transition-colors duration-200"
                >
                  <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path
                      stroke-linecap="round"
                      stroke-linejoin="round"
                      stroke-width="2"
                      d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"
                    >
                    </path>
                  </svg>
                  Clear
                </button>
                <button
                  type="button"
                  phx-click="remote_save_logs"
                  disabled={@remote_log_count == 0}
                  class={[
                    "inline-flex items-center px-3 py-1.5 text-sm font-medium rounded-md transition-colors duration-200",
                    cond do
                      Map.get(assigns, :save_success, false) -> "bg-green-600 text-white"
                      @remote_log_count > 0 -> "bg-blue-600 hover:bg-blue-700 text-white"
                      true -> "bg-gray-300 text-gray-500 cursor-not-allowed"
                    end
                  ]}
                >
                  <%= if Map.get(assigns, :save_success, false) do %>
                    <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path
                        stroke-linecap="round"
                        stroke-linejoin="round"
                        stroke-width="2"
                        d="M5 13l4 4L19 7"
                      >
                      </path>
                    </svg>
                    Saved!
                  <% else %>
                    <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path
                        stroke-linecap="round"
                        stroke-linejoin="round"
                        stroke-width="2"
                        d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"
                      >
                      </path>
                    </svg>
                    Save ({@remote_log_count})
                  <% end %>
                </button>
              </div>
            </div>
          </form>
        </div>
      <% end %>
      
    <!-- Log Viewer -->
      <div class="flex-1 bg-white rounded-lg border border-gray-200 overflow-hidden flex flex-col">
        <div class="px-6 py-3 bg-gray-50 border-b border-gray-200 flex items-center justify-between">
          <div class="flex items-center">
            <h4 class="text-lg font-semibold text-gray-900">Debug Logs</h4>
            <span class="ml-3 px-2 py-1 bg-blue-100 text-blue-800 text-xs font-medium rounded-full">
              {@remote_log_count} / 1000 lines
            </span>
          </div>

          <%= if @remote_logging do %>
            <div class="flex items-center text-sm text-green-600">
              <div class="w-2 h-2 bg-green-500 rounded-full mr-2 animate-pulse"></div>
              Live streaming...
            </div>
          <% end %>
        </div>

        <div class="flex-1 overflow-y-auto">
          <%= if @remote_log_count > 0 do %>
            <div
              id="log-viewer"
              class="p-4 font-mono text-sm space-y-1"
              phx-hook="LogViewer"
              style="max-height: 400px; overflow-y: auto;"
            >
              <%= for log <- Enum.reverse(@remote_logs) do %>
                <div class="flex border-b border-gray-100 pb-1 mb-1 last:border-b-0">
                  <div class="text-gray-500 mr-4 text-xs whitespace-nowrap flex-shrink-0 w-24">
                    {case DateTime.from_iso8601(log.timestamp) do
                      {:ok, dt, _} ->
                        dt |> DateTime.to_time() |> Time.to_string() |> String.slice(0, 8)

                      _ ->
                        log.timestamp |> String.slice(0, 8)
                    end}
                  </div>
                  <div class="text-gray-900 flex-1 break-words">
                    {log.message}
                  </div>
                </div>
              <% end %>
            </div>
          <% else %>
            <div class="flex-1 flex items-center justify-center text-gray-500">
              <div class="text-center">
                <svg
                  class="w-12 h-12 mx-auto mb-4 text-gray-400"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    stroke-linecap="round"
                    stroke-linejoin="round"
                    stroke-width="2"
                    d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"
                  >
                  </path>
                </svg>
                <p class="text-lg font-medium">No logs available</p>
                <p class="text-sm mt-1">
                  <%= if @remote_connected do %>
                    Start logging to view debug messages from the terminal
                  <% else %>
                    Connect to the terminal to start viewing logs
                  <% end %>
                </p>
              </div>
            </div>
          <% end %>
        </div>
      </div>
    </div>

    <script>
      // Toggle delay options visibility
      document.addEventListener('change', function(e) {
      if (e.target.name === 'mode') {
        const delayOptions = document.getElementById('delay-options');
        if (e.target.value === 'WITH_DELAY') {
          delayOptions.style.display = 'block';
        } else {
          delayOptions.style.display = 'none';
        }
      }
      });
    </script>
    """
  end
end
