defmodule DaProductAppWeb.TerminalLocationsLive do
  use DaProductAppWeb, :live_view
  alias DaProductApp.TerminalManagement

  @impl true
  def mount(_params, _session, socket) do
    # Subscribe to real-time updates
    if connected?(socket) do
      TerminalManagement.subscribe_to_device_updates()
    end

    socket =
      socket
      |> assign(:page_title, "Terminal Locations")
      |> assign(:current_page, "terminal_locations")
      |> assign(:loading, true)
      # street, satellite, terrain
      |> assign(:map_view, "street")
      |> assign(:show_clusters, true)
      |> assign(:show_heatmap, false)
      |> assign_initial_filters()
      |> load_location_data()

    {:ok, socket}
  end

  @impl true
  def handle_params(params, _url, socket) do
    socket =
      socket
      |> update_filters_from_params(params)
      |> load_location_data()

    {:noreply, socket}
  end

  @impl true
  def handle_event("update_filter", %{"filter" => filter_data}, socket) do
    socket =
      socket
      |> update_filters(filter_data)
      |> push_patch_with_filters()

    {:noreply, socket}
  end

  @impl true
  def handle_event("change_map_view", %{"view" => view}, socket) do
    {:noreply, assign(socket, :map_view, view)}
  end

  @impl true
  def handle_event("toggle_clusters", _params, socket) do
    {:noreply, assign(socket, :show_clusters, !socket.assigns.show_clusters)}
  end

  @impl true
  def handle_event("toggle_heatmap", _params, socket) do
    {:noreply, assign(socket, :show_heatmap, !socket.assigns.show_heatmap)}
  end

  @impl true
  def handle_event("clear_filters", _params, socket) do
    socket =
      socket
      |> assign_initial_filters()
      |> load_location_data()
      |> push_patch(to: ~p"/terminals/locations")

    {:noreply, socket}
  end

  @impl true
  def handle_event("export_locations", %{"format" => format}, socket) do
    case export_location_data(socket.assigns.filtered_devices, format) do
      {:ok, _} ->
        socket = put_flash(socket, :info, "Location data exported successfully")
        {:noreply, socket}

      {:error, reason} ->
        socket = put_flash(socket, :error, "Export failed: #{reason}")
        {:noreply, socket}
    end
  end

  @impl true
  def handle_event("create_geofence", params, socket) do
    case create_geofence_from_params(params) do
      {:ok, geofence} ->
        socket =
          socket
          |> put_flash(:info, "Geofence '#{geofence.name}' created successfully")
          |> load_location_data()

        {:noreply, socket}

      {:error, reason} ->
        socket = put_flash(socket, :error, "Failed to create geofence: #{reason}")
        {:noreply, socket}
    end
  end

  @impl true
  def handle_info({:device_update, _device_data}, socket) do
    socket = load_location_data(socket)
    {:noreply, socket}
  end

  defp assign_initial_filters(socket) do
    socket
    |> assign(:filters, %{
      # Default to UAE
      country: "AE",
      # Emirate/State
      region: "",
      city: "",
      status: "",
      device_model: "",
      vendor: "",
      device_type: "",
      date_from: Date.add(Date.utc_today(), -30),
      date_to: Date.utc_today(),
      last_activity_days: 30
    })
  end

  defp update_filters_from_params(socket, params) do
    filters = socket.assigns.filters

    updated_filters =
      params
      |> Enum.reduce(filters, fn {key, value}, acc ->
        case key do
          "country" -> Map.put(acc, :country, value)
          "region" -> Map.put(acc, :region, value)
          "city" -> Map.put(acc, :city, value)
          "status" -> Map.put(acc, :status, value)
          "device_model" -> Map.put(acc, :device_model, value)
          "vendor" -> Map.put(acc, :vendor, value)
          "device_type" -> Map.put(acc, :device_type, value)
          "date_from" -> Map.put(acc, :date_from, parse_date(value))
          "date_to" -> Map.put(acc, :date_to, parse_date(value))
          "last_activity_days" -> Map.put(acc, :last_activity_days, parse_int(value, 30))
          _ -> acc
        end
      end)

    assign(socket, :filters, updated_filters)
  end

  defp update_filters(socket, filter_data) do
    current_filters = socket.assigns.filters
    updated_filters = Map.merge(current_filters, atomize_keys(filter_data))
    assign(socket, :filters, updated_filters)
  end

  defp load_location_data(socket) do
    filters = socket.assigns.filters

    # Get all devices with location data
    all_devices = TerminalManagement.list_terminals_with_latest_status()
    IO.inspect(length(all_devices), label: "Total devices from DB")

    # Apply filters and get location data
    filtered_devices =
      all_devices
      |> apply_filters(filters)
      |> Enum.map(&enrich_with_location/1)
      |> Enum.filter(&has_valid_location?/1)

    IO.inspect(length(filtered_devices), label: "Filtered devices with location")

    # Generate location analytics
    location_stats = generate_location_analytics(filtered_devices)

    # Get geographic options for filters
    geographic_options = get_geographic_options(filters.country)

    socket
    |> assign(:all_devices, all_devices)
    |> assign(:filtered_devices, filtered_devices)
    |> assign(:location_stats, location_stats)
    |> assign(:geographic_options, geographic_options)
    |> assign(:loading, false)
  end

  defp apply_filters(devices, filters) do
    devices
    |> filter_by_country(filters.country)
    |> filter_by_region(filters.region)
    |> filter_by_status(filters.status)
    |> filter_by_device_model(filters.device_model)
    |> filter_by_vendor(filters.vendor)
    |> filter_by_date_range(filters.date_from, filters.date_to)
    |> filter_by_last_activity(filters.last_activity_days)
  end

  defp enrich_with_location(device) do
    device_id = device[:id] || device.id

    case TerminalManagement.get_latest_terminal_location(device_id) do
      nil ->
        # If no location data, add default/mock coordinates for testing
        Map.merge(device, %{
          # Random around Dubai
          lat: 25.2048 + (:rand.uniform() - 0.5) * 0.1,
          lng: 55.2708 + (:rand.uniform() - 0.5) * 0.1,
          address: "Location not available",
          location_timestamp: DateTime.utc_now()
        })

      location ->
        Map.merge(device, %{
          lat: parse_float(location.lat),
          lng: parse_float(location.lng),
          address: location.address || "Address not available",
          location_timestamp: location.timestamp || DateTime.utc_now()
        })
    end
  end

  defp has_valid_location?(device) do
    lat = device[:lat] || device.lat
    lng = device[:lng] || device.lng

    # Check if we have valid coordinates
    valid_coords =
      lat != nil and lng != nil and
        is_number(lat) and is_number(lng) and
        lat >= -90 and lat <= 90 and
        lng >= -180 and lng <= 180

    if not valid_coords do
      IO.inspect(device, label: "Device with invalid location")
    end

    valid_coords
  end

  defp generate_location_analytics(devices) do
    total_devices = length(devices)

    online_devices =
      Enum.count(devices, fn device ->
        status = device[:status] || device.status
        status in ["online", "Online", "connected"]
      end)

    # Group by emirates/regions
    by_region = Enum.group_by(devices, &extract_region_from_address/1)

    # Calculate coverage metrics
    coverage_areas = length(Map.keys(by_region))

    avg_devices_per_area =
      if coverage_areas > 0, do: Float.round(total_devices / coverage_areas, 2), else: 0.0

    # Device density analysis
    density_analysis = calculate_device_density(devices)

    %{
      total_devices: total_devices,
      online_devices: online_devices,
      offline_devices: total_devices - online_devices,
      coverage_areas: coverage_areas,
      avg_devices_per_area: avg_devices_per_area,
      by_region: by_region,
      density_analysis: density_analysis,
      last_updated: DateTime.utc_now()
    }
  end

  defp get_geographic_options("AE") do
    %{
      country_name: "United Arab Emirates",
      regions: [
        %{code: "AZ", name: "Abu Dhabi", cities: ["Abu Dhabi", "Al Ain", "Madinat Zayed"]},
        %{code: "DU", name: "Dubai", cities: ["Dubai", "Hatta"]},
        %{code: "SH", name: "Sharjah", cities: ["Sharjah", "Khor Fakkan", "Kalba"]},
        %{code: "AJ", name: "Ajman", cities: ["Ajman"]},
        %{code: "UQ", name: "Umm Al Quwain", cities: ["Umm Al Quwain"]},
        %{code: "RK", name: "Ras Al Khaimah", cities: ["Ras Al Khaimah"]},
        %{code: "FU", name: "Fujairah", cities: ["Fujairah", "Dibba Al-Fujairah"]}
      ]
    }
  end

  defp get_geographic_options("IN") do
    %{
      country_name: "India",
      regions: [
        %{code: "KA", name: "Karnataka", cities: ["Bangalore", "Mysore", "Mangalore"]},
        %{code: "MH", name: "Maharashtra", cities: ["Mumbai", "Pune", "Nagpur"]},
        %{code: "DL", name: "Delhi", cities: ["New Delhi", "Delhi"]}
        # Add more states as needed
      ]
    }
  end

  defp get_geographic_options(_), do: %{country_name: "Unknown", regions: []}

  # Filter functions
  defp filter_by_country(devices, ""), do: devices

  defp filter_by_country(devices, country) do
    Enum.filter(devices, fn device ->
      # This would need to be implemented based on how country is stored
      # For now, we'll use a simple area-based filter
      area = device[:area] || device.area
      area && String.contains?(String.upcase(area || ""), country)
    end)
  end

  defp filter_by_region(devices, ""), do: devices

  defp filter_by_region(devices, region) do
    Enum.filter(devices, fn device ->
      area = device[:area] || device.area
      area && String.contains?(String.downcase(area || ""), String.downcase(region))
    end)
  end

  defp filter_by_status(devices, ""), do: devices

  defp filter_by_status(devices, status) do
    Enum.filter(devices, fn device ->
      device_status = device[:status] || device.status
      device_status == status
    end)
  end

  defp filter_by_device_model(devices, ""), do: devices

  defp filter_by_device_model(devices, model) do
    Enum.filter(devices, fn device ->
      device_model = device[:model] || device.model
      device_model == model
    end)
  end

  defp filter_by_vendor(devices, ""), do: devices

  defp filter_by_vendor(devices, vendor) do
    Enum.filter(devices, fn device ->
      device_vendor = device[:vendor] || device.vendor
      device_vendor == vendor
    end)
  end

  defp filter_by_date_range(devices, nil, nil), do: devices

  defp filter_by_date_range(devices, from_date, to_date) do
    Enum.filter(devices, fn device ->
      device_date =
        case device[:inserted_at] do
          %DateTime{} = dt -> DateTime.to_date(dt)
          nil -> nil
          _ -> nil
        end

      if device_date do
        Date.compare(device_date, from_date || ~D[1900-01-01]) != :lt &&
          Date.compare(device_date, to_date || Date.utc_today()) != :gt
      else
        # Include devices without dates when filtering by date
        true
      end
    end)
  end

  defp filter_by_last_activity(devices, days) when is_integer(days) and days > 0 do
    cutoff_date = DateTime.add(DateTime.utc_now(), -days * 24 * 60 * 60, :second)

    Enum.filter(devices, fn device ->
      case device[:last_seen_at] do
        %DateTime{} = last_seen -> DateTime.compare(last_seen, cutoff_date) != :lt
        # Include devices without last_seen_at data
        nil -> true
        _ -> true
      end
    end)
  end

  defp filter_by_last_activity(devices, _), do: devices

  # Helper functions
  defp parse_float(nil), do: nil
  defp parse_float(value) when is_number(value), do: value

  defp parse_float(value) when is_binary(value) do
    case Float.parse(value) do
      {float_val, _} -> float_val
      :error -> nil
    end
  end

  defp parse_float(_), do: nil

  defp parse_date(nil), do: nil

  defp parse_date(date_string) when is_binary(date_string) do
    case Date.from_iso8601(date_string) do
      {:ok, date} -> date
      _ -> nil
    end
  end

  defp parse_int(value, default) when is_binary(value) do
    case Integer.parse(value) do
      {int_val, _} -> int_val
      :error -> default
    end
  end

  defp parse_int(value, _default) when is_integer(value), do: value
  defp parse_int(_, default), do: default

  defp atomize_keys(map) when is_map(map) do
    Map.new(map, fn {k, v} -> {String.to_atom(k), v} end)
  end

  defp extract_region_from_address(device) do
    # Extract region/emirate from address or area
    area = device[:area] || device.area
    address = device[:address] || device.address
    area || address || "Unknown"
  end

  defp calculate_device_density(devices) do
    # Simple density calculation - would be more sophisticated in real implementation
    %{
      high_density_areas: 0,
      medium_density_areas: 0,
      low_density_areas: 0,
      coverage_gaps: []
    }
  end

  defp push_patch_with_filters(socket) do
    filters = socket.assigns.filters
    query_params = build_query_params(filters)
    push_patch(socket, to: ~p"/terminals/locations?#{query_params}")
  end

  defp build_query_params(filters) do
    filters
    |> Enum.filter(fn {_k, v} -> v != nil and v != "" end)
    |> Enum.map(fn {k, v} -> {to_string(k), to_string(v)} end)
    |> URI.encode_query()
  end

  defp export_location_data(devices, format) do
    # Implementation for exporting location data
    {:ok, "exported"}
  end

  defp create_geofence_from_params(params) do
    # Implementation for creating geofences
    {:ok, %{name: params["name"]}}
  end

  @impl true
  def render(assigns) do
    ~H"""
    <div class="min-h-screen bg-gradient-to-br from-slate-50 to-blue-50">
      <!-- Header Section -->
      <div class="bg-white shadow-sm border-b">
        <div class="px-6 py-6">
          <div class="flex justify-between items-center">
            <div>
              <h1 class="text-3xl font-bold text-gray-900">🗺️ Terminal Locations</h1>
              <p class="text-gray-600 mt-2">Geographic distribution and location management</p>
            </div>
            
    <!-- Action Buttons -->
            <div class="flex items-center space-x-4">
              <button
                phx-click="export_locations"
                phx-value-format="csv"
                class="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 transition-colors flex items-center"
              >
                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path
                    stroke-linecap="round"
                    stroke-linejoin="round"
                    stroke-width="2"
                    d="M12 10v6m0 0l-3-3m3 3l3-3M3 17V7a2 2 0 012-2h6l2 2h6a2 2 0 012 2v10a2 2 0 01-2 2H5a2 2 0 01-2-2z"
                  />
                </svg>
                Export
              </button>

              <button
                phx-click="clear_filters"
                class="bg-gray-600 text-white px-4 py-2 rounded-lg hover:bg-gray-700 transition-colors"
              >
                Clear Filters
              </button>
            </div>
          </div>
          
    <!-- Loading Indicator -->
          <%= if @loading do %>
            <div class="mt-4 flex items-center text-blue-600">
              <svg
                class="animate-spin -ml-1 mr-3 h-5 w-5 text-blue-600"
                fill="none"
                viewBox="0 0 24 24"
              >
                <circle
                  class="opacity-25"
                  cx="12"
                  cy="12"
                  r="10"
                  stroke="currentColor"
                  stroke-width="4"
                >
                </circle>
                <path
                  class="opacity-75"
                  fill="currentColor"
                  d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                >
                </path>
              </svg>
              Loading location data...
            </div>
          <% end %>
        </div>
      </div>

      <div class="flex">
        <!-- Filters Sidebar -->
        {render_filters_sidebar(assigns)}
        
    <!-- Main Content -->
        <div class="flex-1 p-6">
          <%= if !@loading do %>
            <!-- Location Statistics -->
            {render_location_stats(assigns)}
            
    <!-- Interactive Map -->
            {render_location_map(assigns)}
          <% end %>
        </div>
      </div>
    </div>
    """
  end

  defp render_filters_sidebar(assigns) do
    ~H"""
    <div class="w-80 bg-white shadow-sm border-r h-screen overflow-y-auto">
      <div class="p-6">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">🔍 Filters</h3>

        <form phx-change="update_filter" class="space-y-4">
          <!-- Geographic Filters -->
          <div class="border-b border-gray-200 pb-4">
            <h4 class="font-medium text-gray-900 mb-3">📍 Geographic</h4>

            <div class="space-y-3">
              <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Country</label>
                <select
                  name="filter[country]"
                  value={@filters.country}
                  class="w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                >
                  <option value="">All Countries</option>
                  <option value="AE">🇦🇪 United Arab Emirates</option>
                  <option value="IN">🇮🇳 India</option>
                  <option value="SA">🇸🇦 Saudi Arabia</option>
                </select>
              </div>

              <%= if @filters.country != "" do %>
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-1">
                    {if @filters.country == "AE", do: "Emirate", else: "Region"}
                  </label>
                  <select
                    name="filter[region]"
                    value={@filters.region}
                    class="w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                  >
                    <option value="">All Regions</option>
                    <%= for region <- @geographic_options.regions do %>
                      <option value={region.code}>{region.name}</option>
                    <% end %>
                  </select>
                </div>
              <% end %>
            </div>
          </div>
          
    <!-- Device Filters -->
          <div class="border-b border-gray-200 pb-4">
            <h4 class="font-medium text-gray-900 mb-3">📱 Device</h4>

            <div class="space-y-3">
              <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Status</label>
                <select
                  name="filter[status]"
                  value={@filters.status}
                  class="w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                >
                  <option value="">All Status</option>
                  <option value="online">🟢 Online</option>
                  <option value="offline">🔴 Offline</option>
                  <option value="maintenance">🟡 Maintenance</option>
                  <option value="error">🚨 Error</option>
                </select>
              </div>

              <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Vendor</label>
                <select
                  name="filter[vendor]"
                  value={@filters.vendor}
                  class="w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                >
                  <option value="">All Vendors</option>
                  <option value="ingenico">Ingenico</option>
                  <option value="verifone">Verifone</option>
                  <option value="pax">PAX</option>
                  <option value="newland">Newland</option>
                </select>
              </div>

              <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Model</label>
                <input
                  type="text"
                  name="filter[device_model]"
                  value={@filters.device_model}
                  placeholder="Enter device model"
                  class="w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                />
              </div>
            </div>
          </div>
          
    <!-- Date Filters -->
          <div class="border-b border-gray-200 pb-4">
            <h4 class="font-medium text-gray-900 mb-3">📅 Date Range</h4>

            <div class="space-y-3">
              <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">From Date</label>
                <input
                  type="date"
                  name="filter[date_from]"
                  value={@filters.date_from}
                  class="w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                />
              </div>

              <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">To Date</label>
                <input
                  type="date"
                  name="filter[date_to]"
                  value={@filters.date_to}
                  class="w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                />
              </div>

              <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">
                  Last Activity (Days)
                </label>
                <select
                  name="filter[last_activity_days]"
                  value={@filters.last_activity_days}
                  class="w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                >
                  <option value="7">Last 7 days</option>
                  <option value="30">Last 30 days</option>
                  <option value="90">Last 90 days</option>
                  <option value="365">Last year</option>
                </select>
              </div>
            </div>
          </div>
        </form>
      </div>
    </div>
    """
  end

  defp render_location_stats(assigns) do
    ~H"""
    <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-6">
      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <div class="flex items-center justify-between">
          <div>
            <p class="text-sm font-medium text-gray-600">Total Devices</p>
            <p class="text-3xl font-bold text-gray-900">{@location_stats.total_devices}</p>
          </div>
          <div class="bg-blue-100 rounded-full p-3">
            <svg class="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path
                stroke-linecap="round"
                stroke-linejoin="round"
                stroke-width="2"
                d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"
              />
              <path
                stroke-linecap="round"
                stroke-linejoin="round"
                stroke-width="2"
                d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"
              />
            </svg>
          </div>
        </div>
      </div>

      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <div class="flex items-center justify-between">
          <div>
            <p class="text-sm font-medium text-gray-600">Online Devices</p>
            <p class="text-3xl font-bold text-green-600">{@location_stats.online_devices}</p>
          </div>
          <div class="bg-green-100 rounded-full p-3">
            <svg class="w-6 h-6 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path
                stroke-linecap="round"
                stroke-linejoin="round"
                stroke-width="2"
                d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"
              />
            </svg>
          </div>
        </div>
      </div>

      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <div class="flex items-center justify-between">
          <div>
            <p class="text-sm font-medium text-gray-600">Coverage Areas</p>
            <p class="text-3xl font-bold text-purple-600">{@location_stats.coverage_areas}</p>
          </div>
          <div class="bg-purple-100 rounded-full p-3">
            <svg class="w-6 h-6 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path
                stroke-linecap="round"
                stroke-linejoin="round"
                stroke-width="2"
                d="M9 20l-5.447-2.724A1 1 0 013 16.382V5.618a1 1 0 011.447-.894L9 7m0 13l6-3m-6 3V7m6 10l4.553 2.276A1 1 0 0021 18.382V7.618a1 1 0 00-.553-.894L15 4m0 13V4m0 0L9 7"
              />
            </svg>
          </div>
        </div>
      </div>

      <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <div class="flex items-center justify-between">
          <div>
            <p class="text-sm font-medium text-gray-600">Avg per Area</p>
            <p class="text-3xl font-bold text-orange-600">{@location_stats.avg_devices_per_area}</p>
          </div>
          <div class="bg-orange-100 rounded-full p-3">
            <svg class="w-6 h-6 text-orange-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path
                stroke-linecap="round"
                stroke-linejoin="round"
                stroke-width="2"
                d="M7 12l3-3 3 3 4-4M8 21l4-4 4 4M3 4h18M4 4h16v12a1 1 0 01-1 1H5a1 1 0 01-1-1V4z"
              />
            </svg>
          </div>
        </div>
      </div>
    </div>
    """
  end

  defp render_location_map(assigns) do
    ~H"""
    <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
      <div class="flex items-center justify-between mb-4">
        <h3 class="text-lg font-semibold text-gray-900">🗺️ Device Locations</h3>
        
    <!-- Map Controls -->
        <div class="flex items-center space-x-4">
          <div class="flex items-center space-x-2">
            <label class="text-sm font-medium text-gray-700">View:</label>
            <select
              phx-change="change_map_view"
              name="view"
              value={@map_view}
              class="rounded border-gray-300 text-sm focus:border-blue-500 focus:ring-blue-500"
            >
              <option value="street">Street</option>
              <option value="satellite">Satellite</option>
              <option value="terrain">Terrain</option>
            </select>
          </div>

          <button
            type="button"
            phx-click="toggle_clusters"
            class={"px-3 py-1 rounded text-sm transition-colors " <> if(@show_clusters, do: "bg-blue-600 text-white", else: "bg-gray-200 text-gray-700 hover:bg-gray-300")}
          >
            Clusters
          </button>

          <button
            type="button"
            phx-click="toggle_heatmap"
            class={"px-3 py-1 rounded text-sm transition-colors " <> if(@show_heatmap, do: "bg-red-600 text-white", else: "bg-gray-200 text-gray-700 hover:bg-gray-300")}
          >
            Heatmap
          </button>
        </div>
      </div>

      <div
        id="terminal-locations-map"
        phx-hook="TerminalLocationsMapHook"
        data-devices={Jason.encode!(@filtered_devices)}
        data-map-view={@map_view}
        data-show-clusters={@show_clusters}
        data-show-heatmap={@show_heatmap}
        class="w-full h-[600px] rounded-lg shadow border"
      >
      </div>

      <div class="mt-4 flex items-center justify-between">
        <div class="flex items-center space-x-6 text-sm">
          <div class="flex items-center space-x-2">
            <div class="w-3 h-3 rounded-full bg-green-500"></div>
            <span class="text-gray-600">Online ({@location_stats.online_devices})</span>
          </div>
          <div class="flex items-center space-x-2">
            <div class="w-3 h-3 rounded-full bg-red-500"></div>
            <span class="text-gray-600">Offline ({@location_stats.offline_devices})</span>
          </div>
          <div class="flex items-center space-x-2">
            <div class="w-3 h-3 rounded-full bg-orange-500"></div>
            <span class="text-gray-600">Maintenance</span>
          </div>
        </div>

        <div class="text-sm text-gray-600">
          Showing {length(@filtered_devices)} of {@location_stats.total_devices} devices
        </div>
      </div>
    </div>
    """
  end
end
