defmodule DaProductAppWeb.UserManagementLive do
  use DaProductAppWeb, :live_view

  alias DaProductApp.Users
  alias DaProductApp.Users.User

  @impl true
  def mount(_params, _session, socket) do
    # Check if user has permission to manage users
    if authorized?(socket.assigns.current_user) do
      socket =
        socket
        |> assign(:users, list_users())
        |> assign(:modules, Users.list_modules())
        |> assign(:permissions, Users.list_permissions())

      {:ok, socket}
    else
      {:ok, socket |> put_flash(:error, "You don't have permission to access this page.")}
    end
  end

  @impl true
  def handle_event("delete_user", %{"id" => id}, socket) do
    if authorized?(socket.assigns.current_user) do
      user = Users.get_user!(id)
      {:ok, _} = Users.delete_user(user)

      socket =
        socket
        |> assign(:users, list_users())
        |> put_flash(:info, "User deleted successfully")

      {:noreply, socket}
    else
      {:noreply, socket |> put_flash(:error, "You don't have permission to perform this action.")}
    end
  end

  @impl true
  def handle_event("toggle_user_role", %{"id" => id, "role" => role}, socket) do
    if authorized?(socket.assigns.current_user) do
      user = Users.get_user!(id)
      allowed_roles = ["user", "admin", "superuser"]

      if role in allowed_roles do
        new_role = String.to_atom(role)

        case Users.update_user(user, %{role: new_role}) do
          {:ok, _user} ->
            socket =
              socket
              |> assign(:users, list_users())
              |> put_flash(:info, "User role updated successfully")

            {:noreply, socket}

          {:error, _changeset} ->
            {:noreply, socket |> put_flash(:error, "Failed to update user role")}
        end
      else
        {:noreply, socket |> put_flash(:error, "Invalid role provided.")}
      end
    else
      {:noreply, socket |> put_flash(:error, "You don't have permission to perform this action.")}
    end
  end

  @impl true
  def render(assigns) do
    ~H"""
    <div class="mx-auto max-w-6xl">
      <.header class="text-center">
        User Management
        <:subtitle>
          Manage users, roles, and permissions
        </:subtitle>
      </.header>

      <div class="mt-8">
        <div class="bg-white shadow overflow-hidden sm:rounded-lg">
          <div class="px-4 py-5 sm:px-6">
            <h3 class="text-lg leading-6 font-medium text-gray-900">
              All Users
            </h3>
            <p class="mt-1 max-w-2xl text-sm text-gray-500">
              List of all registered users and their roles
            </p>
          </div>
          <div class="border-t border-gray-200">
            <table class="min-w-full divide-y divide-gray-200">
              <thead class="bg-gray-50">
                <tr>
                  <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Name
                  </th>
                  <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Email
                  </th>
                  <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Role
                  </th>
                  <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Status
                  </th>
                  <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Actions
                  </th>
                </tr>
              </thead>
              <tbody class="bg-white divide-y divide-gray-200">
                <tr :for={user <- @users} class="hover:bg-gray-50">
                  <td class="px-6 py-4 whitespace-nowrap">
                    <div class="flex items-center">
                      <div class="text-sm font-medium text-gray-900">
                        {user.name || "#{user.first_name} #{user.last_name}"}
                      </div>
                    </div>
                  </td>
                  <td class="px-6 py-4 whitespace-nowrap">
                    <div class="text-sm text-gray-900">{user.email}</div>
                  </td>
                  <td class="px-6 py-4 whitespace-nowrap">
                    <select
                      class="text-sm border-gray-300 rounded-md"
                      phx-change="toggle_user_role"
                      phx-value-id={user.id}
                    >
                      <option value="user" selected={user.role == :user}>User</option>
                      <option value="admin" selected={user.role == :admin}>Admin</option>
                      <option value="superuser" selected={user.role == :superuser}>Super User</option>
                    </select>
                  </td>
                  <td class="px-6 py-4 whitespace-nowrap">
                    <span class={"inline-flex px-2 py-1 text-xs font-semibold rounded-full " <> 
                      if user.confirmed_at, do: "bg-green-100 text-green-800", else: "bg-yellow-100 text-yellow-800"}>
                      {if user.confirmed_at, do: "Confirmed", else: "Pending"}
                    </span>
                  </td>
                  <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                    <.button
                      phx-click="delete_user"
                      phx-value-id={user.id}
                      data-confirm="Are you sure you want to delete this user?"
                      class="text-red-600 hover:text-red-900 text-sm"
                    >
                      Delete
                    </.button>
                  </td>
                </tr>
              </tbody>
            </table>
          </div>
        </div>
      </div>

      <div class="mt-8">
        <div class="bg-white shadow overflow-hidden sm:rounded-lg">
          <div class="px-4 py-5 sm:px-6">
            <h3 class="text-lg leading-6 font-medium text-gray-900">
              System Modules
            </h3>
            <p class="mt-1 max-w-2xl text-sm text-gray-500">
              Available system modules and their permissions
            </p>
          </div>
          <div class="border-t border-gray-200">
            <dl>
              <div
                :for={module <- @modules}
                class="bg-gray-50 px-4 py-5 sm:grid sm:grid-cols-3 sm:gap-4 sm:px-6"
              >
                <dt class="text-sm font-medium text-gray-500">
                  {module.name}
                </dt>
                <dd class="mt-1 text-sm text-gray-900 sm:mt-0 sm:col-span-2">
                  {module.description}
                </dd>
              </div>
            </dl>
          </div>
        </div>
      </div>
    </div>
    """
  end

  defp authorized?(user) do
    Users.user_has_permission?(user, "manage_users") || user.role in [:admin, :superuser]
  end

  defp list_users do
    Users.list_users()
  end
end
