defmodule DaProductAppWeb.Router do
  use DaProductAppWeb, :router

  import DaProductAppWeb.UserAuth

  pipeline :browser do
    plug :accepts, ["html"]
    plug :fetch_session
    plug :fetch_live_flash
    plug :put_root_layout, html: {DaProductAppWeb.Layouts, :root}
    plug :protect_from_forgery

    # plug :put_secure_browser_headers, %{"content-security-policy" => "default-src 'self'; img-src * data:;"}
    # add library which we want to add and make the list of it so that security check can be managed
    plug :put_secure_browser_headers, %{
      "content-security-policy" =>
        "default-src 'self'; " <>
          "script-src 'self' 'unsafe-inline' https://testapp.ariticapp.com; " <>
          "style-src 'self' 'unsafe-inline' https://fonts.googleapis.com  ag-grid-community.min.js ag-grid-enterprise.js ag-charts-community.js ag-charts-enterprise.js https://unpkg.com/leaflet/dist/leaflet.css; " <>
          "img-src * data:; " <>
          "font-src 'self' https://fonts.gstatic.com data:; " <>
          "connect-src 'self' wss://ariticapp.com;"
    }

    plug :fetch_current_user
  end

  pipeline :api do
    plug :accepts, ["json"]
  end

  pipeline :non_csrf do
    plug :accepts, ["html", "json"]
    plug :fetch_session
    plug :put_secure_browser_headers
    # Do NOT include :protect_from_forgery, so CSRF is not enforced here
  end

  scope "/", DaProductAppWeb do
    pipe_through :browser

    get "/", PageController, :home
    live "/form", FormLive, :index
    live "/live", PageLive, :index
    live "/live/modal/:size", PageLive, :modal
    live "/live/slide_over/:origin", SbomComponentLive, :slide_over
    live "/live/pagination/:page", PageLive, :pagination
  end

  scope "/", DaProductAppWeb do
    pipe_through :browser

    get "/", PageController, :home
    # resources "/software", SoftwareController, only: [:index, :show]
  end

  scope "/", DaProductAppWeb do
    pipe_through [:browser, :require_authenticated_user]

    live_session :default, on_mount: [{DaProductAppWeb.UserAuth, :mount_current_user}] do
      live "/dashboard", DashboardLive, :index
      live "/analytics", AnalyticsLive, :index
      live "/alerts", AlertsLive, :index
      live "/audit-compliance", AuditComplianceLive, :index
      live "/sbomcomponent", SbomComponentLive, :index
      live "/sbomcomponent/:origin", SbomComponentLive, :slide_over
      live "/workflow", WorkflowLive
      live "/software", SoftwareLive
      live "/software/:id", SoftwareLive.Show
      live "/terminals", TerminalLive.Index, :index
      live "/terminals/new", TerminalLive.Index, :new
      live "/terminals/:serial_number", TerminalLive.Index, :slide_over
      live "/terminals_all/locations", TerminalLocationsLive, :index

      # Terminal Groups Management
      live "/terminals/groups", TerminalGroupLive.Index, :index
      live "/terminals/groups/:id", TerminalGroupLive.Index, :show

      live "/admin/settlements", SettlementLive.Index, :index
      live "/admin/settlements/files", SettlementLive.FileTracker, :index

      live "/ota/packages", ApplicationPackageLive.Index, :index
      live "/ota/jobs", ApplicationUpgradeLive.Index, :index
      live "/ota/status", ApplicationUpgradeStatusLive.Index, :index

      live "/apppackage", ApplicationPackageLive.Index, :index
      live "/appupgrade", ApplicationUpgradeLive.Index, :index
      live "/appupgrade/status", ApplicationUpgradeStatusLive.Index, :index

      # Parameter Management
      live "/admin/parameters", ParameterManagementLive.Dashboard, :index
      live "/admin/parameters/templates", ParameterManagementLive.Index, :templates
      live "/admin/parameters/logs", ParameterManagementLive.Logs, :index
      # Device Configuration Management
      live "/admin/device-configs", DeviceConfigLive.Index, :index
      live "/admin/device-configs/new", DeviceConfigLive.Index, :new
    end

    # live "/software/:id", SoftwareLive.Show, as: :software_show

    # resources "/software", SoftwareController, only: [:index, :show]
    # resources "/sbom_component", ComponentController, only: [:index]
  end

  scope "/api/v1", DaProductAppWeb do
    pipe_through :api

    get "/settlements/settlement/:merchant_id/:date",
        SettlementController,
        :get_by_merchant_and_date
  end

  # Settlement CSV parsing endpoint for AlipayPlus and TMS
  scope "/api/v1/settlements/settlement", DaProductAppWeb do
    pipe_through :api

    get "/:filename", SettlementController, :parse_settlement_csv_by_filename
  end

  # Enable LiveDashboard and Swoosh mailbox preview in development
  if Application.compile_env(:da_product_app, :dev_routes) do
    # If you want to use the LiveDashboard in production, you should put
    # it behind authentication and allow only admins to access it.
    # If your application does not have an admins-only section yet,
    # you can use Plug.BasicAuth to set up some basic authentication
    # as long as you are also using SSL (which you should anyway).
    import Phoenix.LiveDashboard.Router

    scope "/dev" do
      pipe_through :browser

      live_dashboard "/dashboard", metrics: DaProductAppWeb.Telemetry
      forward "/mailbox", Plug.Swoosh.MailboxPreview
    end
  end

  ## Authentication routes

  scope "/", DaProductAppWeb do
    pipe_through [:browser, :redirect_if_user_is_authenticated]

    live_session :redirect_if_user_is_authenticated,
      on_mount: [{DaProductAppWeb.UserAuth, :redirect_if_user_is_authenticated}] do
      live "/users/register", UserRegistrationLive, :new
      live "/users/log_in", UserLoginLive, :new
      live "/users/reset_password", UserForgotPasswordLive, :new
      live "/users/reset_password/:token", UserResetPasswordLive, :edit
      live "/transactions/:id", TransactionLive.Show, :show
    end

    post "/users/log_in", UserSessionController, :create
  end

  scope "/", DaProductAppWeb do
    pipe_through [:browser, :require_authenticated_user]

    live_session :require_authenticated_user,
      on_mount: [{DaProductAppWeb.UserAuth, :ensure_authenticated}] do
      live "/users/settings", UserSettingsLive, :edit
      live "/users/settings/confirm_email/:token", UserSettingsLive, :confirm_email
      live "/admin/users", UserManagementLive, :index
    end
  end

  scope "/", DaProductAppWeb do
    pipe_through [:browser]

    delete "/users/log_out", UserSessionController, :delete

    live_session :current_user,
      on_mount: [{DaProductAppWeb.UserAuth, :mount_current_user}] do
      live "/users/confirm/:token", UserConfirmationLive, :edit
      live "/users/confirm", UserConfirmationInstructionsLive, :new
    end
  end

  scope "/", DaProductAppWeb do
    pipe_through :non_csrf

    post "/transaction_post", TransactionPostController, :new
  end

  scope "/api", DaProductAppWeb do
    post "/qr", QRController, :generate
  end

  scope "/api", DaProductAppWeb do
    post "/device", DeviceController, :initiate
  end

  # Merchant Settlement APIs
  scope "/api/v1/merchant", DaProductAppWeb do
    pipe_through :api

    # Settlement endpoints
    get "/settlements", SettlementController, :index
    get "/settlement/summary", SettlementController, :summary
    get "/settlement/:settlement_id", SettlementController, :show
    get "/settlement/:settlement_id/transactions", SettlementController, :transactions
    get "/settlement/:settlement_id/csv", SettlementController, :download_csv
    get "/settlement/:settlement_id/pdf", SettlementController, :download_pdf
    get "/settlement/:settlement_id/download", SettlementController, :download

    # AlipayPlus specific endpoints
    post "/settlement/alipayplus/process", SettlementController, :process_alipayplus_file

    post "/settlement/alipayplus/schedule_retrieval",
         SettlementController,
         :schedule_alipayplus_retrieval

    # Dispute endpoints
    post "/dispute", DisputeController, :create
    get "/disputes", DisputeController, :index
    get "/dispute/:dispute_id", DisputeController, :show
    get "/dispute/:dispute_id/download", DisputeController, :download
  end

  # EOD Settlement File Generation API
  scope "/api/v1/settlements", DaProductAppWeb do
    pipe_through :api

    post "/settlement/eod/date", SettlementController, :generate_eod_file
    post "/settlement/eod/:date", SettlementController, :generate_eod_file
    get "/settlement/eod/job/:job_id", SettlementController, :get_job_status

    # New transaction-level settlement file endpoints
    get "/settlement/:bank_user_id/:date",
        SettlementController,
        :generate_merchant_settlement_files

    get "/settlement/:bank_user_id/:date/csv",
        SettlementController,
        :generate_merchant_settlement_csv

    get "/settlement/:bank_user_id/:date/json",
        SettlementController,
        :generate_merchant_settlement_json

    # YCS Settlement Creation (when provider-wise batch number is disabled)
    post "/ycs/create", SettlementController, :create_ycs_settlement
  end

  # TMS Management APIs
  scope "/api", DaProductAppWeb do
    pipe_through :api

    # Dashboard endpoints
    get "/dashboard/overview", DashboardController, :overview
    get "/dashboard/update_stats", DashboardController, :update_stats
    get "/dashboard/online_rate", DashboardController, :online_rate
    get "/dashboard/chart_data", DashboardController, :chart_data

    # OTA Configuration endpoints
    post "/ota/configuration", OtaConfigurationController, :create
    get "/ota/configuration/:device_serial", OtaConfigurationController, :show
    get "/ota/configurations", OtaConfigurationController, :index
    post "/ota/configuration/:id/retry", OtaConfigurationController, :retry

    # Parameter Push endpoints
    post "/parameter/push/:device_serial", ParameterPushController, :push_parameters
    post "/parameter/schedule/:device_serial", ParameterPushController, :schedule_parameter_push
    post "/parameter/batch/push", ParameterPushController, :batch_push_parameters
    post "/parameter/batch/schedule", ParameterPushController, :schedule_batch_parameter_push
    post "/parameter/merchant/:device_serial", ParameterPushController, :push_merchant_config

    post "/parameter/merchant/schedule/:device_serial",
         ParameterPushController,
         :schedule_merchant_config_push

    # Enhanced Parameter endpoints (similar to app package pattern)
    post "/parameter/template/apply/:device_serial",
         ParameterController,
         :apply_template_to_terminal

    post "/parameter/template/batch/apply", ParameterController, :batch_apply_template
    get "/parameter/status/:device_serial", ParameterController, :get_parameter_push_status
    get "/parameter/statistics", ParameterController, :get_parameter_push_statistics
    post "/parameter/compatibility/check", ParameterController, :check_template_compatibility

    # File Download endpoints
    post "/download/send/:device_serial", FileDownloadController, :send_download_command
    post "/download/schedule/:device_serial", FileDownloadController, :schedule_download_command
    post "/download/batch/send", FileDownloadController, :batch_send_download_command
    post "/download/logo/:device_serial", FileDownloadController, :send_logo_download
    post "/download/logo/schedule/:device_serial", FileDownloadController, :schedule_logo_download
    post "/download/firmware/:device_serial", FileDownloadController, :send_firmware_download

    post "/download/application/:device_serial",
         FileDownloadController,
         :send_application_download

    # App Package Deployment endpoints
    post "/package/deploy/:device_serial", AppPackageController, :deploy_package
    post "/package/schedule/:device_serial", AppPackageController, :schedule_package_deployment
    post "/package/batch/deploy", AppPackageController, :batch_deploy_package
    post "/package/rollback/:device_serial", AppPackageController, :rollback_package

    post "/package/rollback/schedule/:device_serial",
         AppPackageController,
         :schedule_package_rollback

    get "/package/status/:device_serial", AppPackageController, :get_deployment_status
    get "/package/statistics", AppPackageController, :get_deployment_statistics
    post "/package/compatibility/check", AppPackageController, :check_package_compatibility

    # Device Monitor endpoints
    get "/device/:device_serial/monitor", DeviceMonitorController, :show
    post "/device/monitor/check_offline", DeviceMonitorController, :check_offline
    get "/device/monitor/stats", DeviceMonitorController, :stats
  end

  scope "/api", DaProductAppWeb do
    post "/settlement_webhook", SettlementWebhookController, :receive_settlement
    get "/merchant_settlement_summary/:merchant_id", MerchantSettlementSummaryController, :show
  end

  # Aani Settlement Summary API
  scope "/api/v1/aani", DaProductAppWeb do
    pipe_through :api
    post "/settlement-summary", AaniSettlementController, :settlement_summary
  end

  # Parameter Management APIs
  resources "/parameters", ParameterController, except: [:new, :edit]

  # Parameter Templates - separate resource
  get "/parameter_templates", ParameterController, :list_templates
  post "/parameter_templates", ParameterController, :create_template
  get "/parameter_templates/:id", ParameterController, :show_template
  put "/parameter_templates/:id", ParameterController, :update_template
  delete "/parameter_templates/:id", ParameterController, :delete_template

  # Terminal Parameter Operations
  get "/terminals/:serial_number/parameters", ParameterController, :get_terminal_parameters

  post "/terminals/:serial_number/parameters/push",
       ParameterController,
       :push_parameters_to_terminal

  post "/terminals/:serial_number/parameters/template/:template_id",
       ParameterController,
       :apply_template_to_terminal

  put "/terminals/:serial_number/parameters/:parameter_key",
      ParameterController,
      :set_terminal_parameter

  # Parameter Push Logs
  get "/terminals/:serial_number/parameter_logs", ParameterController, :get_push_logs
  get "/parameter_logs/:id", ParameterController, :get_push_log
end
