# Script for populating the database. You can run it as:
#
#     mix run priv/repo/seeds.exs
#
# Inside the script, you can read and write to any of your
# repositories directly:
#
#     DaProductApp.Repo.insert!(%DaProductApp.SomeSchema{})
#
# We recommend using the bang functions (`insert!`, `update!`
# and so on) as they will fail if something goes wrong.

alias DaProductApp.Repo
alias DaProductApp.Users.{User, Module, Permission}
alias DaProductApp.Merchants.Merchant
alias DaProductApp.Providers.Provider
alias DaProductApp.TerminalManagement.{TmsTerminal, TerminalGroup}
alias DaProductApp.ParameterManagement.ParameterCategory

# Create default modules
user_module =
  Repo.insert!(%Module{
    name: "User Management",
    description: "User and authentication management",
    active: true
  })

terminal_module =
  Repo.insert!(%Module{
    name: "Terminal Management",
    description: "Terminal device management",
    active: true
  })

transaction_module =
  Repo.insert!(%Module{
    name: "Transaction Management",
    description: "Payment transaction management",
    active: true
  })

settlement_module =
  Repo.insert!(%Module{
    name: "Settlement Management",
    description: "Payment settlement and reconciliation",
    active: true
  })

# Create default permissions
Repo.insert!(%Permission{
  name: "user:read",
  description: "View user information",
  module_id: user_module.id,
  active: true
})

Repo.insert!(%Permission{
  name: "user:write",
  description: "Create and edit users",
  module_id: user_module.id,
  active: true
})

Repo.insert!(%Permission{
  name: "terminal:read",
  description: "View terminal information",
  module_id: terminal_module.id,
  active: true
})

Repo.insert!(%Permission{
  name: "terminal:write",
  description: "Create and edit terminals",
  module_id: terminal_module.id,
  active: true
})

Repo.insert!(%Permission{
  name: "transaction:read",
  description: "View transaction information",
  module_id: transaction_module.id,
  active: true
})

Repo.insert!(%Permission{
  name: "settlement:read",
  description: "View settlement information",
  module_id: settlement_module.id,
  active: true
})

# Create default admin user
admin_user =
  Repo.insert!(%User{
    email: "admin@example.com",
    hashed_password: Bcrypt.hash_pwd_salt("admin123"),
    confirmed_at: DateTime.utc_now(),
    role: :admin,
    name: "System Administrator",
    first_name: "Admin",
    last_name: "User"
  })

# Create sample providers
alipay_provider =
  Repo.insert!(%Provider{
    name: "Alipay",
    status: "active",
    production_url: "https://api.alipay.com",
    production_mode: "sandbox",
    description: "Alipay payment gateway"
  })

wechat_provider =
  Repo.insert!(%Provider{
    name: "WeChat Pay",
    status: "active",
    production_url: "https://api.wechatpay.com",
    production_mode: "sandbox",
    description: "WeChat Pay payment gateway"
  })

# Create sample merchants
merchant1 =
  Repo.insert!(%Merchant{
    id: Ecto.UUID.generate(),
    name: "Sample Restaurant",
    email: "restaurant@example.com",
    phone: "+1234567890",
    status: "active",
    merchant_id: "MERCH001"
  })

merchant2 =
  Repo.insert!(%Merchant{
    id: Ecto.UUID.generate(),
    name: "Sample Retail Store",
    email: "retail@example.com",
    phone: "+1234567891",
    status: "active",
    merchant_id: "MERCH002"
  })

# Create sample terminal groups
group1 =
  Repo.insert!(%TerminalGroup{
    name: "Restaurant Terminals",
    description: "Terminals deployed in restaurant locations",
    is_active: true
  })

group2 =
  Repo.insert!(%TerminalGroup{
    name: "Retail Terminals",
    description: "Terminals deployed in retail locations",
    is_active: true
  })

# Create sample terminals
Repo.insert!(%TmsTerminal{
  serial_number: "T001234567890",
  status: "online",
  model: "POS-X1",
  vendor: "TechCorp",
  area: "North Zone",
  group: "Restaurant Terminals",
  merchant_id: "MERCH001",
  app_version: "1.0.0",
  data_version: "1.0.0",
  system_version: "1.0.0"
})

Repo.insert!(%TmsTerminal{
  serial_number: "T001234567891",
  status: "online",
  model: "POS-X2",
  vendor: "TechCorp",
  area: "South Zone",
  group: "Retail Terminals",
  merchant_id: "MERCH002",
  app_version: "1.0.0",
  data_version: "1.0.0",
  system_version: "1.0.0"
})

# Create parameter categories
Repo.insert!(%ParameterCategory{
  name: "Network Configuration",
  description: "Network and connectivity settings",
  is_active: true
})

Repo.insert!(%ParameterCategory{
  name: "Display Settings",
  description: "Screen and UI configuration",
  is_active: true
})

Repo.insert!(%ParameterCategory{
  name: "Payment Settings",
  description: "Payment processing configuration",
  is_active: true
})

IO.puts("Database seeded successfully!")
IO.puts("Admin user created: admin@example.com / admin123")
IO.puts("Created #{Repo.aggregate(User, :count, :id)} users")
IO.puts("Created #{Repo.aggregate(Merchant, :count, :id)} merchants")
IO.puts("Created #{Repo.aggregate(Provider, :count, :id)} providers")
IO.puts("Created #{Repo.aggregate(TmsTerminal, :count, :id)} terminals")
