defmodule DaProductApp.BankUserIdEnhancementsTest do
  use DaProductApp.DataCase
  alias DaProductApp.{Settlements, Disputes}
  import DaProductApp.DisputesFixtures
  import DaProductApp.SettlementsFixtures

  describe "bank_user_id filtering for settlements" do
    @tag :bank_user_id_enhancement
    test "can filter settlements by bank_user_id" do
      # Create settlements with different bank_user_ids
      settlement1 = settlement_fixture(%{bank_user_id: "BANK_USER_001"})
      settlement2 = settlement_fixture(%{bank_user_id: "BANK_USER_002"})
      settlement3 = settlement_fixture(%{bank_user_id: "BANK_USER_001"})

      # Filter by bank_user_id
      settlements = Settlements.list_settlements(%{"bank_user_id" => "BANK_USER_001"})

      assert length(settlements) == 2
      assert Enum.all?(settlements, &(&1.bank_user_id == "BANK_USER_001"))
    end

    @tag :bank_user_id_enhancement
    test "includes bank_user_id in settlement response" do
      settlement = settlement_fixture(%{bank_user_id: "BANK_USER_123"})

      retrieved = Settlements.get_settlement_by_settlement_id(settlement.settlement_id)

      assert retrieved.bank_user_id == "BANK_USER_123"
    end
  end

  describe "bank_user_id filtering for disputes" do
    test "can filter disputes by bank_user_id" do
      dispute1 = dispute_fixture(%{bank_user_id: "BANK_USER_001", txn_id: "TXN_001"})
      dispute2 = dispute_fixture(%{bank_user_id: "BANK_USER_002", txn_id: "TXN_002"})
      dispute3 = dispute_fixture(%{bank_user_id: "BANK_USER_001", txn_id: "TXN_003"})

      # Filter by bank_user_id
      disputes = Disputes.list_disputes(%{"bank_user_id" => "BANK_USER_001"})

      assert length(disputes) == 2
      assert Enum.all?(disputes, &(&1.bank_user_id == "BANK_USER_001"))
    end

    test "includes bank_user_id in dispute response" do
      dispute = dispute_fixture(%{bank_user_id: "BANK_USER_456", txn_id: "TXN_456"})

      retrieved = Disputes.get_dispute_by_dispute_id(dispute.dispute_id)

      assert retrieved.bank_user_id == "BANK_USER_456"
    end

    test "can create dispute with bank_user_id" do
      attrs = %{
        "txn_id" => "TXN_NEW",
        "reason" => "Test dispute",
        "contact_email" => "test@example.com",
        "bank_user_id" => "BANK_USER_NEW"
      }

      {:ok, dispute} = Disputes.create_dispute(attrs)

      assert dispute.bank_user_id == "BANK_USER_NEW"
    end
  end
end
