defmodule DaProductApp.DisputesTest do
  use DaProductApp.DataCase

  alias DaProductApp.Disputes

  describe "disputes" do
    alias DaProductApp.Disputes.Dispute

    import DaProductApp.DisputesFixtures

    @invalid_attrs %{dispute_id: nil, txn_id: nil, reason: nil, contact_email: nil}

    test "list_disputes/0 returns all disputes" do
      dispute = dispute_fixture()
      assert Disputes.list_disputes() == [dispute]
    end

    test "get_dispute_by_dispute_id/1 returns the dispute with given dispute_id" do
      dispute = dispute_fixture()
      assert Disputes.get_dispute_by_dispute_id(dispute.dispute_id) == dispute
    end

    test "create_dispute/1 with valid data creates a dispute" do
      valid_attrs = %{
        txn_id: "TXN1003",
        reason: "Transaction not credited",
        merchant_comment: "Please verify settlement mismatch",
        contact_email: "merchant@example.com"
      }

      assert {:ok, %Dispute{} = dispute} = Disputes.create_dispute(valid_attrs)
      assert dispute.txn_id == "TXN1003"
      assert dispute.reason == "Transaction not credited"
      assert dispute.merchant_comment == "Please verify settlement mismatch"
      assert dispute.contact_email == "merchant@example.com"
      assert dispute.status == "open"
      assert String.starts_with?(dispute.dispute_id, "DSP")
    end

    test "create_dispute/1 with invalid data returns error changeset" do
      assert {:error, %Ecto.Changeset{}} = Disputes.create_dispute(@invalid_attrs)
    end

    test "create_dispute/1 with invalid email returns error changeset" do
      invalid_attrs = %{
        txn_id: "TXN1003",
        reason: "Transaction not credited",
        contact_email: "invalid-email"
      }

      assert {:error, %Ecto.Changeset{}} = Disputes.create_dispute(invalid_attrs)
    end

    test "update_dispute/2 with valid data updates the dispute" do
      dispute = dispute_fixture()

      update_attrs = %{
        status: "resolved",
        ops_response: "Issue has been resolved"
      }

      assert {:ok, %Dispute{} = dispute} = Disputes.update_dispute(dispute, update_attrs)
      assert dispute.status == "resolved"
      assert dispute.ops_response == "Issue has been resolved"
    end

    test "update_dispute/2 with invalid data returns error changeset" do
      dispute = dispute_fixture()
      assert {:error, %Ecto.Changeset{}} = Disputes.update_dispute(dispute, @invalid_attrs)
      assert dispute == Disputes.get_dispute_by_dispute_id(dispute.dispute_id)
    end

    test "delete_dispute/1 deletes the dispute" do
      dispute = dispute_fixture()
      assert {:ok, %Dispute{}} = Disputes.delete_dispute(dispute)
      assert Disputes.get_dispute_by_dispute_id(dispute.dispute_id) == nil
    end

    test "change_dispute/1 returns a dispute changeset" do
      dispute = dispute_fixture()
      assert %Ecto.Changeset{} = Disputes.change_dispute(dispute)
    end
  end
end
