defmodule DaProductApp.Settlements.AlipayPlus.CsvParserTest do
  use ExUnit.Case, async: true

  alias DaProductApp.Settlements.AlipayPlus.CsvParser

  describe "parse_filename/1" do
    test "parses valid AlipayPlus filename correctly" do
      filename = "settlement_1000012345_AED_202000000000000000_AGR5678_001.csv"
      
      assert {:ok, metadata} = CsvParser.parse_filename(filename)
      assert metadata.participant_id == "1000012345"
      assert metadata.settlement_currency == "AED"
      assert metadata.settlement_batch_id == "202000000000000000"
      assert metadata.participant_agreement_id == "AGR5678"
      assert metadata.sequence == "001"
    end

    test "returns error for invalid filename format" do
      filename = "invalid_filename.csv"
      
      assert {:error, error_msg} = CsvParser.parse_filename(filename)
      assert String.contains?(error_msg, "Invalid filename format")
    end
  end

  describe "convert_amount_from_smallest_unit/2" do
    test "converts AED amount correctly" do
      result = CsvParser.convert_amount_from_smallest_unit("1960", "AED")
      assert Decimal.equal?(result, Decimal.new("19.60"))
    end

    test "converts JPY amount correctly (no decimal places)" do
      result = CsvParser.convert_amount_from_smallest_unit("1960", "JPY")
      assert Decimal.equal?(result, Decimal.new("1960"))
    end

    test "converts USD amount correctly" do
      result = CsvParser.convert_amount_from_smallest_unit("12345", "USD")
      assert Decimal.equal?(result, Decimal.new("123.45"))
    end
  end

  describe "parse_file/2" do
    test "parses valid AlipayPlus CSV file successfully" do
      filename = "settlement_1000012345_AED_202000000000000000_AGR5678_001.csv"
      
      csv_content = """
      settleDate,valueDate,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      20220421,20220421,CREDIT,AED,1960,AED,1960,
      clearingBatchId,clearingDate,totalCount,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      202000000000000000,20220419,6,CREDIT,AED,1960,AED,1960,
      """
      
      assert {:ok, parsed_data} = CsvParser.parse_file(filename, csv_content)
      
      # Verify filename parsing
      assert parsed_data.participant_id == "1000012345"
      assert parsed_data.settlement_currency == "AED"
      assert parsed_data.settlement_batch_id == "202000000000000000"
      
      # Verify summary section
      summary = parsed_data.summary
      assert summary.settle_date == ~D[2022-04-21]
      assert summary.value_date == ~D[2022-04-21]
      assert summary.fund_direction == "CREDIT"
      assert summary.settlement_currency == "AED"
      assert Decimal.equal?(summary.net_settlement_amount_value, Decimal.new("19.60"))
      
      # Verify detail section
      assert length(parsed_data.details) == 1
      detail = hd(parsed_data.details)
      assert detail.clearing_batch_id == "202000000000000000"
      assert detail.clearing_date == ~D[2022-04-19]
      assert detail.total_count == 6
      assert detail.fund_direction == "CREDIT"
      assert Decimal.equal?(detail.net_settlement_amount_value, Decimal.new("19.60"))
    end

    test "handles multiple detail rows correctly" do
      filename = "settlement_1000012345_AED_202000000000000000_AGR5678_001.csv"
      
      csv_content = """
      settleDate,valueDate,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      20220421,20220421,CREDIT,AED,3920,AED,3920,
      clearingBatchId,clearingDate,totalCount,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      202000000000000001,20220419,3,CREDIT,AED,1960,AED,1960,
      202000000000000002,20220419,3,CREDIT,AED,1960,AED,1960,
      """
      
      assert {:ok, parsed_data} = CsvParser.parse_file(filename, csv_content)
      
      # Verify summary shows total amount
      assert Decimal.equal?(parsed_data.summary.net_settlement_amount_value, Decimal.new("39.20"))
      
      # Verify both detail rows are parsed
      assert length(parsed_data.details) == 2
      
      [detail1, detail2] = parsed_data.details
      assert detail1.clearing_batch_id == "202000000000000001"
      assert detail2.clearing_batch_id == "202000000000000002"
      assert detail1.total_count == 3
      assert detail2.total_count == 3
    end

    test "handles optional fields correctly" do
      filename = "settlement_1000012345_AED_202000000000000000_AGR5678_001.csv"
      
      csv_content = """
      settleDate,valueDate,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      20220421,20220421,CREDIT,AED,1960,,,"Optional info"
      clearingBatchId,clearingDate,totalCount,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      202000000000000000,20220419,6,CREDIT,AED,1960,,,
      """
      
      assert {:ok, parsed_data} = CsvParser.parse_file(filename, csv_content)
      
      # Verify optional fields are handled
      summary = parsed_data.summary
      assert summary.transaction_currency == nil
      assert summary.net_transaction_amount_value == nil
      assert summary.extend_info == "Optional info"
      
      detail = hd(parsed_data.details)
      assert detail.transaction_currency == nil
      assert detail.net_transaction_amount_value == nil
      assert detail.extend_info == nil
    end

    test "returns error for invalid CSV structure" do
      filename = "settlement_1000012345_AED_202000000000000000_AGR5678_001.csv"
      
      csv_content = """
      invalid,header
      some,data
      """
      
      assert {:error, error_msg} = CsvParser.parse_file(filename, csv_content)
      assert String.contains?(error_msg, "Invalid summary section header")
    end

    test "returns error for malformed filename" do
      filename = "invalid_filename.csv"
      
      csv_content = """
      settleDate,valueDate,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      20220421,20220421,CREDIT,AED,1960,AED,1960,
      """
      
      assert {:error, error_msg} = CsvParser.parse_file(filename, csv_content)
      assert String.contains?(error_msg, "Invalid filename format")
    end
  end

  describe "validate_csv_structure/1" do
    test "validates correct CSV structure" do
      lines = [
        "settleDate,valueDate,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo",
        "20220421,20220421,CREDIT,AED,1960,AED,1960,",
        "clearingBatchId,clearingDate,totalCount,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo",
        "202000000000000000,20220419,6,CREDIT,AED,1960,AED,1960,"
      ]
      
      assert {:ok, ^lines} = CsvParser.validate_csv_structure(lines)
    end

    test "rejects CSV with insufficient lines" do
      lines = ["single,line"]
      
      assert {:error, error_msg} = CsvParser.validate_csv_structure(lines)
      assert String.contains?(error_msg, "must contain at least summary and detail sections")
    end

    test "rejects CSV with invalid summary header" do
      lines = [
        "invalid,header",
        "some,data",
        "clearingBatchId,clearingDate,totalCount,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo",
        "202000000000000000,20220419,6,CREDIT,AED,1960,AED,1960,"
      ]
      
      assert {:error, error_msg} = CsvParser.validate_csv_structure(lines)
      assert String.contains?(error_msg, "Invalid summary section header")
    end
  end
end