defmodule DaProductApp.Settlements.AlipayPlus.ProcessorTest do
  use DaProductApp.DataCase

  alias DaProductApp.Settlements.AlipayPlus.Processor
  alias DaProductApp.Settlements.AlipayPlus.CsvParser
  alias DaProductApp.Settlements.Settlement
  alias DaProductApp.Repo

  describe "generate_settlement_id/1" do
    test "generates settlement ID from parsed data" do
      parsed_data = %{
        settlement_batch_id: "202000000000000000",
        summary: %{
          settle_date: ~D[2022-04-21]
        }
      }
      
      settlement_id = Processor.generate_settlement_id(parsed_data)
      assert settlement_id == "SETT20220421-202000000000000000"
    end

    test "handles different dates correctly" do
      parsed_data = %{
        settlement_batch_id: "202000000000000000",
        summary: %{
          settle_date: ~D[2023-12-31]
        }
      }
      
      settlement_id = Processor.generate_settlement_id(parsed_data)
      assert settlement_id == "SETT20231231-202000000000000000"
    end
  end

  describe "create_or_update_settlement/1" do
    test "creates new settlement when none exists" do
      parsed_data = create_valid_parsed_data()
      
      assert {:ok, settlement} = Processor.create_or_update_settlement(parsed_data)
      
      assert settlement.settlement_id == "SETT20220421-202000000000000000"
      assert settlement.date == ~D[2022-04-21]
      assert settlement.status == "settled"
      assert Decimal.equal?(settlement.amount, Decimal.new("19.60"))
      assert settlement.participant_id == "1000012345"
      assert settlement.settlement_batch_id == "202000000000000000"
      assert settlement.participant_agreement_id == "AGR5678"
      assert settlement.value_date == ~D[2022-04-21]
      assert settlement.fund_direction == "CREDIT"
    end

    test "updates existing settlement when it already exists" do
      # First create a settlement
      parsed_data = create_valid_parsed_data()
      {:ok, original_settlement} = Processor.create_or_update_settlement(parsed_data)
      
      # Update the parsed data with new amount
      updated_parsed_data = put_in(parsed_data.summary.net_settlement_amount_value, Decimal.new("39.20"))
      
      assert {:ok, updated_settlement} = Processor.create_or_update_settlement(updated_parsed_data)
      
      # Should be the same record, but with updated amount
      assert updated_settlement.id == original_settlement.id
      assert updated_settlement.settlement_id == original_settlement.settlement_id
      assert Decimal.equal?(updated_settlement.amount, Decimal.new("39.20"))
    end

    test "sets status to pending for DEBIT fund direction" do
      parsed_data = 
        create_valid_parsed_data()
        |> put_in([:summary, :fund_direction], "DEBIT")
      
      assert {:ok, settlement} = Processor.create_or_update_settlement(parsed_data)
      assert settlement.status == "pending"
    end

    test "sets status to settled for CREDIT fund direction" do
      parsed_data = 
        create_valid_parsed_data()
        |> put_in([:summary, :fund_direction], "CREDIT")
      
      assert {:ok, settlement} = Processor.create_or_update_settlement(parsed_data)
      assert settlement.status == "settled"
    end
  end

  describe "validate_settlement_reconciliation/2" do
    test "returns balanced when amounts match" do
      settlement = %Settlement{
        amount: Decimal.new("19.60"),
        settlement_id: "SETT20220421-202000000000000000"
      }
      
      parsed_data = create_valid_parsed_data()
      
      assert {:ok, :balanced} = Processor.validate_settlement_reconciliation(settlement, parsed_data)
    end

    test "returns unbalanced with difference when amounts don't match" do
      settlement = %Settlement{
        amount: Decimal.new("25.00"),
        settlement_id: "SETT20220421-202000000000000000"
      }
      
      parsed_data = create_valid_parsed_data() # Amount is 19.60
      
      assert {:ok, :unbalanced, mismatch_info} = Processor.validate_settlement_reconciliation(settlement, parsed_data)
      
      assert Decimal.equal?(mismatch_info.settlement_amount, Decimal.new("25.00"))
      assert Decimal.equal?(mismatch_info.clearing_total, Decimal.new("19.60"))
      assert Decimal.equal?(mismatch_info.difference, Decimal.new("5.40"))
    end

    test "handles multiple clearing cycles correctly" do
      settlement = %Settlement{
        amount: Decimal.new("39.20"),
        settlement_id: "SETT20220421-202000000000000000"
      }
      
      parsed_data = create_parsed_data_with_multiple_details()
      
      assert {:ok, :balanced} = Processor.validate_settlement_reconciliation(settlement, parsed_data)
    end
  end

  describe "process_settlement_file/3" do
    test "processes valid settlement file successfully" do
      filename = "settlement_1000012345_AED_202000000000000000_AGR5678_001.csv"
      content = create_valid_csv_content()
      
      assert {:ok, result} = Processor.process_settlement_file(filename, content, %{})
      
      assert result.status == :success
      assert result.settlement_id == "SETT20220421-202000000000000000"
      assert result.transactions_updated >= 0
      assert result.errors == []
      
      # Verify settlement was created in database
      settlement = Repo.get_by(Settlement, settlement_id: result.settlement_id)
      assert settlement != nil
      assert settlement.participant_id == "1000012345"
    end

    test "handles processing errors gracefully" do
      filename = "invalid_file.csv"
      content = "invalid,csv,content"
      
      assert {:error, _reason} = Processor.process_settlement_file(filename, content, %{})
      
      # Verify no settlement was created due to transaction rollback
      settlement = Repo.get_by(Settlement, filename: filename)
      assert settlement == nil
    end
  end

  # Helper functions

  defp create_valid_parsed_data do
    %{
      filename: "settlement_1000012345_AED_202000000000000000_AGR5678_001.csv",
      participant_id: "1000012345",
      settlement_currency: "AED",
      settlement_batch_id: "202000000000000000",
      participant_agreement_id: "AGR5678",
      sequence: "001",
      summary: %{
        settle_date: ~D[2022-04-21],
        value_date: ~D[2022-04-21],
        fund_direction: "CREDIT",
        settlement_currency: "AED",
        net_settlement_amount_value: Decimal.new("19.60"),
        transaction_currency: "AED",
        net_transaction_amount_value: Decimal.new("19.60"),
        extend_info: nil
      },
      details: [
        %{
          clearing_batch_id: "202000000000000000",
          clearing_date: ~D[2022-04-19],
          total_count: 6,
          fund_direction: "CREDIT",
          settlement_currency: "AED",
          net_settlement_amount_value: Decimal.new("19.60"),
          transaction_currency: "AED",
          net_transaction_amount_value: Decimal.new("19.60"),
          extend_info: nil
        }
      ]
    }
  end

  defp create_parsed_data_with_multiple_details do
    %{
      filename: "settlement_1000012345_AED_202000000000000000_AGR5678_001.csv",
      participant_id: "1000012345",
      settlement_currency: "AED",
      settlement_batch_id: "202000000000000000",
      participant_agreement_id: "AGR5678",
      sequence: "001",
      summary: %{
        settle_date: ~D[2022-04-21],
        value_date: ~D[2022-04-21],
        fund_direction: "CREDIT",
        settlement_currency: "AED",
        net_settlement_amount_value: Decimal.new("39.20"),
        transaction_currency: "AED",
        net_transaction_amount_value: Decimal.new("39.20"),
        extend_info: nil
      },
      details: [
        %{
          clearing_batch_id: "202000000000000001",
          clearing_date: ~D[2022-04-19],
          total_count: 3,
          fund_direction: "CREDIT",
          settlement_currency: "AED",
          net_settlement_amount_value: Decimal.new("19.60"),
          transaction_currency: "AED",
          net_transaction_amount_value: Decimal.new("19.60"),
          extend_info: nil
        },
        %{
          clearing_batch_id: "202000000000000002",
          clearing_date: ~D[2022-04-19],
          total_count: 3,
          fund_direction: "CREDIT",
          settlement_currency: "AED",
          net_settlement_amount_value: Decimal.new("19.60"),
          transaction_currency: "AED",
          net_transaction_amount_value: Decimal.new("19.60"),
          extend_info: nil
        }
      ]
    }
  end

  defp create_valid_csv_content do
    """
    settleDate,valueDate,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
    20220421,20220421,CREDIT,AED,1960,AED,1960,
    clearingBatchId,clearingDate,totalCount,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
    202000000000000000,20220419,6,CREDIT,AED,1960,AED,1960,
    """
  end
end