defmodule DaProductApp.Settlements.AlipayPlusIntegrationTest do
  use DaProductApp.DataCase
  
  alias DaProductApp.Settlements
  alias DaProductApp.Settlements.Settlement
  alias DaProductApp.Settlements.AlipayPlus.CsvParser
  alias DaProductApp.Settlements.AlipayPlus.Processor
  alias DaProductApp.Repo

  describe "AlipayPlus Settlement Integration" do
    test "end-to-end processing of AlipayPlus settlement file" do
      # Sample AlipayPlus CSV content
      filename = "settlement_1000012345_AED_202000000000000000_AGR5678_001.csv"
      csv_content = """
      settleDate,valueDate,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      20220421,20220421,CREDIT,AED,1960,AED,1960,
      clearingBatchId,clearingDate,totalCount,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      202000000000000000,20220419,6,CREDIT,AED,1960,AED,1960,
      """

      # Test CSV parsing
      assert {:ok, parsed_data} = CsvParser.parse_file(filename, csv_content)
      assert parsed_data.participant_id == "1000012345"
      assert parsed_data.settlement_currency == "AED"
      assert parsed_data.settlement_batch_id == "202000000000000000"

      # Test settlement processing
      assert {:ok, result} = Processor.process_settlement_file(filename, csv_content, %{})
      assert result.status == :success
      assert result.settlement_id == "SETT20220421-202000000000000000"

      # Verify settlement was created in database
      settlement = Repo.get_by(Settlement, settlement_id: result.settlement_id)
      assert settlement != nil
      assert settlement.participant_id == "1000012345"
      assert settlement.settlement_batch_id == "202000000000000000"
      assert settlement.participant_agreement_id == "AGR5678"
      assert settlement.fund_direction == "CREDIT"
      assert settlement.status == "settled"
      assert Decimal.equal?(settlement.amount, Decimal.new("19.60"))

      # Test that settlement is retrievable via context
      retrieved_settlement = Settlements.get_settlement_by_settlement_id(result.settlement_id)
      assert retrieved_settlement.id == settlement.id

      # Test listing settlements includes AlipayPlus settlement
      settlements = Settlements.list_settlements(%{})
      assert Enum.any?(settlements, &(&1.settlement_id == result.settlement_id))

      # Test AlipayPlus provider filtering
      alipayplus_settlements = Settlements.list_settlements_by_provider("ALIPAYPLUS")
      assert Enum.any?(alipayplus_settlements, &(&1.settlement_id == result.settlement_id))
    end

    test "processing multiple clearing cycles correctly" do
      filename = "settlement_1000012345_AED_202000000000000000_AGR5678_001.csv"
      csv_content = """
      settleDate,valueDate,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      20220421,20220421,CREDIT,AED,3920,AED,3920,
      clearingBatchId,clearingDate,totalCount,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      202000000000000001,20220419,3,CREDIT,AED,1960,AED,1960,
      202000000000000002,20220419,3,CREDIT,AED,1960,AED,1960,
      """

      assert {:ok, result} = Processor.process_settlement_file(filename, csv_content, %{})
      
      settlement = Repo.get_by(Settlement, settlement_id: result.settlement_id)
      assert Decimal.equal?(settlement.amount, Decimal.new("39.20"))
      assert settlement.total_transaction_count == 6

      # Verify details structure
      assert settlement.details["clearing_cycles"]
      assert length(settlement.details["clearing_cycles"]) == 2
    end

    test "handles currency conversion correctly for different currencies" do
      # Test with JPY (no decimal places)
      filename = "settlement_1000012345_JPY_202000000000000000_AGR5678_001.csv"
      csv_content = """
      settleDate,valueDate,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      20220421,20220421,CREDIT,JPY,1960,JPY,1960,
      clearingBatchId,clearingDate,totalCount,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      202000000000000000,20220419,6,CREDIT,JPY,1960,JPY,1960,
      """

      assert {:ok, result} = Processor.process_settlement_file(filename, csv_content, %{})
      
      settlement = Repo.get_by(Settlement, settlement_id: result.settlement_id)
      # JPY doesn't use decimal places, so 1960 smallest units = 1960 JPY
      assert Decimal.equal?(settlement.amount, Decimal.new("1960"))
    end

    test "sets correct status based on fund direction" do
      # Test DEBIT fund direction
      filename = "settlement_1000012345_AED_202000000000000000_AGR5678_001.csv"
      csv_content = """
      settleDate,valueDate,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      20220421,20220421,DEBIT,AED,1960,AED,1960,
      clearingBatchId,clearingDate,totalCount,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      202000000000000000,20220419,6,DEBIT,AED,1960,AED,1960,
      """

      assert {:ok, result} = Processor.process_settlement_file(filename, csv_content, %{})
      
      settlement = Repo.get_by(Settlement, settlement_id: result.settlement_id)
      assert settlement.status == "pending"
      assert settlement.fund_direction == "DEBIT"
    end

    test "handles optional fields correctly" do
      filename = "settlement_1000012345_AED_202000000000000000_AGR5678_001.csv"
      csv_content = """
      settleDate,valueDate,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      20220421,20220421,CREDIT,AED,1960,,,"Extended info"
      clearingBatchId,clearingDate,totalCount,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      202000000000000000,20220419,6,CREDIT,AED,1960,,,
      """

      assert {:ok, result} = Processor.process_settlement_file(filename, csv_content, %{})
      
      settlement = Repo.get_by(Settlement, settlement_id: result.settlement_id)
      assert settlement.transaction_currency == nil
      assert settlement.net_transaction_amount_value == nil
      assert settlement.extend_info == "Extended info"
    end

    test "reprocessing existing settlement updates data" do
      # First processing
      filename = "settlement_1000012345_AED_202000000000000000_AGR5678_001.csv"
      csv_content_v1 = """
      settleDate,valueDate,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      20220421,20220421,CREDIT,AED,1960,AED,1960,
      clearingBatchId,clearingDate,totalCount,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      202000000000000000,20220419,6,CREDIT,AED,1960,AED,1960,
      """

      assert {:ok, result1} = Processor.process_settlement_file(filename, csv_content_v1, %{})
      settlement1 = Repo.get_by(Settlement, settlement_id: result1.settlement_id)
      
      # Second processing with updated data
      csv_content_v2 = """
      settleDate,valueDate,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      20220421,20220421,CREDIT,AED,2940,AED,2940,
      clearingBatchId,clearingDate,totalCount,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      202000000000000000,20220419,9,CREDIT,AED,2940,AED,2940,
      """

      assert {:ok, result2} = Processor.process_settlement_file(filename, csv_content_v2, %{})
      
      # Should be same settlement record, but updated
      assert result1.settlement_id == result2.settlement_id
      
      settlement2 = Repo.get_by(Settlement, settlement_id: result2.settlement_id)
      assert settlement1.id == settlement2.id
      assert Decimal.equal?(settlement2.amount, Decimal.new("29.40"))
      assert settlement2.total_transaction_count == 9
    end

    test "using context functions for AlipayPlus operations" do
      filename = "settlement_1000012345_AED_202000000000000000_AGR5678_001.csv"
      csv_content = """
      settleDate,valueDate,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      20220421,20220421,CREDIT,AED,1960,AED,1960,
      clearingBatchId,clearingDate,totalCount,fundDirection,settlementCurrency,netSettlementAmountValue,transactionCurrency,netTransactionAmountValue,extendInfo
      202000000000000000,20220419,6,CREDIT,AED,1960,AED,1960,
      """

      # Test processing via context
      assert {:ok, result} = Settlements.process_alipayplus_file(filename, csv_content, %{})
      assert result.status == :success

      # Test scheduling retrieval via context
      assert {:ok, _job} = Settlements.schedule_alipayplus_retrieval("1000012345", "20220421", %{})
    end
  end
end