defmodule DaProductApp.SettlementsTest do
  use DaProductApp.DataCase

  alias DaProductApp.Settlements

  describe "settlements" do
    alias DaProductApp.Settlements.Settlement
    alias DaProductApp.Settlements.MerchantBatchNumber

    import DaProductApp.SettlementsFixtures

    @invalid_attrs %{settlement_id: nil, date: nil, amount: nil, status: nil}

    test "list_settlements/0 returns all settlements" do
      settlement = settlement_fixture()
      assert Settlements.list_settlements() == [settlement]
    end

    test "get_settlement_by_settlement_id/1 returns the settlement with given settlement_id" do
      settlement = settlement_fixture()
      assert Settlements.get_settlement_by_settlement_id(settlement.settlement_id) == settlement
    end

    test "create_settlement/1 with valid data creates a settlement" do
      valid_attrs = %{
        settlement_id: "SETT20250629",
        date: ~D[2025-06-29],
        amount: Decimal.new("1000.00"),
        status: "pending",
        total_transaction_count: 5,
        provider_id: 1,
        merchant_id: "MRC123"
      }

      assert {:ok, %Settlement{} = settlement} = Settlements.create_settlement(valid_attrs)
      assert settlement.settlement_id == "SETT20250629"
      assert settlement.date == ~D[2025-06-29]
      assert settlement.amount == Decimal.new("1000.00")
      assert settlement.status == "pending"
      assert settlement.total_transaction_count == 5
      assert settlement.provider_id == 1
      assert settlement.merchant_id == "MRC123"
    end

    test "create_settlement/1 with invalid data returns error changeset" do
      assert {:error, %Ecto.Changeset{}} = Settlements.create_settlement(@invalid_attrs)
    end

    test "update_settlement/2 with valid data updates the settlement" do
      settlement = settlement_fixture()

      update_attrs = %{
        status: "settled",
        amount: Decimal.new("1500.00"),
        total_transaction_count: 8
      }

      assert {:ok, %Settlement{} = settlement} =
               Settlements.update_settlement(settlement, update_attrs)

      assert settlement.status == "settled"
      assert settlement.amount == Decimal.new("1500.00")
      assert settlement.total_transaction_count == 8
    end

    test "get_settlement_summary/1 returns correct summary data" do
      _settlement1 = settlement_fixture(%{status: "settled", amount: Decimal.new("1000.00")})
      _settlement2 = settlement_fixture(%{status: "pending", amount: Decimal.new("500.00")})
      _settlement3 = settlement_fixture(%{status: "exception", amount: Decimal.new("200.00")})

      summary = Settlements.get_settlement_summary()

      assert summary.total_settled == Decimal.new("1000.00")
      assert summary.pending == Decimal.new("500.00")
      assert summary.exception_count == 1
    end

    test "generate_settlement_id/0 generates a valid settlement ID" do
      settlement_id = Settlements.generate_settlement_id()
      assert String.starts_with?(settlement_id, "SETT")
      assert String.length(settlement_id) == 12
    end
  end

  describe "merchant_batch_numbers" do
    test "get_and_increment_batch_number/2 creates new record when none exists" do
      merchant_id = "TEST_MERCHANT_001"

      assert {:ok, batch_number} = Settlements.get_and_increment_batch_number(merchant_id)
      assert batch_number == 1

      # Verify record was created
      batch_record = Settlements.get_merchant_batch_number(merchant_id)
      assert batch_record != nil
      assert batch_record.merchant_id == merchant_id
      assert batch_record.batch_number == 1
    end

    test "get_and_increment_batch_number/2 increments existing batch number" do
      merchant_id = "TEST_MERCHANT_002"

      # Create initial record
      {:ok, _} =
        Settlements.create_merchant_batch_number(%{
          merchant_id: merchant_id,
          batch_number: 5
        })

      # Increment batch number
      assert {:ok, batch_number} = Settlements.get_and_increment_batch_number(merchant_id)
      assert batch_number == 6

      # Verify record was updated
      batch_record = Settlements.get_merchant_batch_number(merchant_id)
      assert batch_record.batch_number == 6
    end

    test "get_current_batch_number/1 returns 1 when no record exists" do
      assert Settlements.get_current_batch_number("NON_EXISTENT_MERCHANT") == 1
    end

    test "get_current_batch_number/1 returns current batch number when record exists" do
      merchant_id = "TEST_MERCHANT_003"

      {:ok, _} =
        Settlements.create_merchant_batch_number(%{
          merchant_id: merchant_id,
          batch_number: 10
        })

      assert Settlements.get_current_batch_number(merchant_id) == 10
    end

    test "provider_wise_batch_number_enabled?/0 returns configuration value" do
      # Test that it returns the configured value
      result = Settlements.provider_wise_batch_number_enabled?()
      assert is_boolean(result)
    end
  end
end
